package org.lsst.ccs.config.remote;

import org.lsst.ccs.config.*;

import java.rmi.RemoteException;
import java.util.List;
import java.util.Properties;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 * A raw implementation of Configuration Service that delegates to a <TT>ConfigurationFacade</TT>
 * object.
 * @author bamade
 */
// Date: 06/06/12

public class ConfigurationServiceImpl
        implements ConfigurationService {
    ConfigurationFacade facade ;
    public static ConfigurationServiceImpl ON_DATABASE ;
    static {
        String clazzName = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.configDAO", "org.lsst.ccs.config.dao.hibernate.HibernateDAO");
        DBInterface dao = null;
        try {
            dao = (DBInterface) Class.forName(clazzName).newInstance();
            ON_DATABASE = new ConfigurationServiceImpl(new ConfigurationFacade(dao));
        } catch (Exception e) {
            throw new Error(e);
        }
    }
    public static ConfigurationServiceImpl ON_LOCAL_FILES =
            new ConfigurationServiceImpl(new ConfigurationFacade(new FIleBasedDAO()));


    public ConfigurationServiceImpl(ConfigurationFacade facade) {
        this.facade = facade ;
    }

    public ConfigurationFacade getFacade() {
        return facade ;
    }

    @Override
    public synchronized DescriptionResult registerSubsystemDescription(SubsystemDescription newDescription) throws RemoteException {
        final DescriptionResult res = new DescriptionResult();
        facade.setDeprecationListener(new DeprecationListener() {
            @Override
            public void subsystemDeprecating(SubsystemDescription description) {
                res.setDescriptionDeprecationEvent(new DescriptionDeprecationEvent(description, ""));
            }

            @Override
            public void configProfileDeprecating(ConfigProfile profile) {
                res.addProfileDeprecationEvent(new ProfileDeprecationEvent(profile, ""));
            }
        });
        try {
            SubsystemDescription description = facade.registerSubsystemDescription(newDescription);
            res.setResult(description);
            return res ;
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server registration fail", e);
        }
    }

    @Override
    public synchronized DescriptionResult deprecateSubsystemDescription(String subsystemName, String tag) throws RemoteException {
        final DescriptionResult res = new DescriptionResult();
        facade.setDeprecationListener(new DeprecationListener() {
            @Override
            public void subsystemDeprecating(SubsystemDescription description) {
                res.setDescriptionDeprecationEvent(new DescriptionDeprecationEvent(description, ""));
            }

            @Override
            public void configProfileDeprecating(ConfigProfile profile) {
                res.addProfileDeprecationEvent(new ProfileDeprecationEvent(profile, ""));
            }
        });
        try {
            SubsystemDescription deprecated = facade.deprecateSubsystemDescription(subsystemName, tag);
            res.setResult(deprecated);
            return res;
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server deprecation fail", e);
        } finally {
            facade.setDeprecationListener(null);
        }
    }

    @Override
    public SubsystemDescription getActiveSubsystemDescription(String name, String tag) throws RemoteException {

        try {
            return facade.getActiveSubsystemDescription(name, tag) ;
        } catch (PersistenceLayerException e) {
           throw new RemoteException("description query" , e) ;
        }
    }

    @Override
    public synchronized ProfileResult registerConfigProfile(ConfigProfile newProfile) throws RemoteException {
        final ProfileResult res = new ProfileResult();
        facade.setDeprecationListener(new DeprecationListener() {
            @Override
            public void subsystemDeprecating(SubsystemDescription description) {
            }

            @Override
            public void configProfileDeprecating(ConfigProfile profile) {
                res.setProfileDeprecationEvent(new ProfileDeprecationEvent(profile, ""));
            }
        });
        try {
            ConfigProfile profile = facade.registerConfigProfile(newProfile);
            res.setResult(profile) ;
            return res ;
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server registration fail", e);
        }
    }

    @Override
    public ProfileResult registerConfiguration(String subsystemName, String configurationName, String tag,
                                               String user, int level , Properties props) throws RemoteException{
        SubsystemDescription description = getActiveSubsystemDescription(subsystemName,"") ;
        // creates a raw configProfile
        ConfigProfile profile = Factories.createRawConfigProfile(description, configurationName, tag, user, level) ;
        // modifies the configProfile
        profile.mergeProperties(props);

        return registerConfigProfile(profile) ;
    }

    @Override
    public synchronized ProfileResult deprecateConfigProfile(String name, String tag) throws RemoteException {
        final ProfileResult res = new ProfileResult();
        facade.setDeprecationListener(new DeprecationListener() {
            @Override
            public void subsystemDeprecating(SubsystemDescription description) {
            }

            @Override
            public void configProfileDeprecating(ConfigProfile profile) {
                res.setProfileDeprecationEvent(new ProfileDeprecationEvent(profile, ""));
            }
        });
        try {
            ConfigProfile deprecated = facade.deprecateConfigProfile(name, tag);
            res.setResult(deprecated);
            return res;
        } catch (PersistenceLayerException exc) {
            throw new RemoteException("server deprecation fail", exc);
        } finally {
            facade.setDeprecationListener(null);
        }
    }

    @Override
    public ConfigProfile getActiveConfigProfile(String name, String tag) throws RemoteException {
        try {
            return facade.getActiveConfigProfile(name, tag) ;
        } catch (PersistenceLayerException e) {
           throw new RemoteException("configuration query", e) ;
        }
    }

    @Override
    public ParameterConfiguration modifyParmConf(ParameterConfiguration engineeringModeParm) throws RemoteException {
        try {
            ParameterConfiguration res = facade.engineerParmConfig(engineeringModeParm) ;
            return res ;
        } catch (PersistenceLayerException exc) {
            throw new RemoteException("parameter configuration fail :", exc);
        }
    }


    @Override
    public ConfigProfile getConfigRunningAt(String subsystemName, long date) throws RemoteException {
        try {
            return facade.getConfigRunningAt(subsystemName, date);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public String getActiveValueAt(String subsystemName, String parameterPath, long date) throws RemoteException {
        try {
            return facade.getActiveValueAt(subsystemName,parameterPath, date);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public ConfigProfile getConfigValidAt(String name, String tag, long date) throws RemoteException {
        try {
            return facade.getConfigValidAt(name, tag, date);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public String getValueValidAt(String profileName, String profileTag, String parameterPath, long date) throws RemoteException {
        try {
            return facade.getValueValidAt(profileName, profileTag, parameterPath, date);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }


    @Override
    public ConfigProfile getPrevious(ConfigProfile current) throws RemoteException {
        try {
            return facade.getPrevious(current);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public ConfigProfile getNext(ConfigProfile current) throws RemoteException {
        try {
            return facade.getNext(current);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public SubsystemDescription getPrevious(SubsystemDescription current) throws RemoteException {
        try {
            return facade.getPrevious(current);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public void createPreparedConfiguration(String subsystemName, String configName, String tag, String user) throws RemoteException {
        try {
            facade.registerPreparedConfiguration(subsystemName,configName,tag,user) ;
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public void registerMachineConfiguration(MachineConfiguration machineConfiguration) throws RemoteException {
        try {
            facade.registerMachineConfiguration(machineConfiguration);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public MachineConfiguration getMachineConfiguration(String macAddress) throws RemoteException {
        try {
            MachineConfiguration res = facade.getMachineConfiguration(macAddress) ;
            return res ;
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public List<?> simpleHQLRequest(String hqlString) throws RemoteException {
        try {
            return facade.simpleHQLRequest(hqlString);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server request fail", e);
        }
    }
}
