package org.lsst.ccs.config;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Writer;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.rest.file.server.client.RestFileSystemOptions;

/**
 * A wrapper class to grant access to a remote FileSystem.
 * There can be multiple instances of this class. For the same cache name
 * they will share the same instance of the FileSystem.
 * 
 * @author LSST CCS Team
 */
public class RemoteFileServer {

    private static final java.util.logging.Logger LOG = Logger.getLogger(RemoteFileServer.class.getName());
    
    private final static Map<String,FileSystem> fileSystems = new ConcurrentHashMap<>();

    private final String cacheName;
    private final String mountPoint;
    
    /**
     * Create an instance of a remote file server with a given cache name.The cache name will be used as the mount point for a local cache.
     * 
     * @param descName   The description name for the agent using this class
     * @param mountPoint The mount point
     */
    public RemoteFileServer(String descName, String mountPoint) {
        this(descName, mountPoint, new Properties());
    }
    public RemoteFileServer(String descName, String mountPoint, Properties p) {
        this.mountPoint = mountPoint.replace("/","");
        this.cacheName = descName+"/"+mountPoint+"/";
        getFileSystem(p);
    }


    /**
     * Get an InputStream for reading data for a given Path and OpenOptions.
     * 
     * @param path    The Path of the file to open.
     * @param options The OpenOption array to open the file with.
     * @return        The corresponding InputStream
     * @throws IOException 
     */
    public InputStream getInputStream(Path path, OpenOption... options) throws IOException {
        //The following line needs to be encosed in a try-catch block
        //to protect the local cache functionality in offline mode.
        //See https://jira.slac.stanford.edu/browse/LSSTCCS-2731
        try {
            Files.createDirectories(path.getParent());       
        } catch (Exception e) {}
        try ( InputStream in = Files.newInputStream(path, options)) {
            LOG.log(Level.FINE, "Loading input stream {0} with options {1}", new Object[]{path,Arrays.asList(options)});
            return in;
        } catch (IOException x) {                
            LOG.log(Level.WARNING, "Did not find {0} with options {1}", new Object[]{path,Arrays.asList(options)});                
            throw x;
        }
    }

    /**
     * Check if a file exists in the remote file server
     * @param path The path to check.
     * @return true/false if the path exists or not.
     */
    public boolean exists(Path path) {
        return Files.exists(path);       
    }

    
    /**
     * Get an BufferedWriter for writing data for a given Path and OpenOptions.
     * 
     * @param path    The Path of the file to open.
     * @param options The OpenOption array to open the file with.
     * @return        The corresponding BufferedWriter
     * @throws IOException 
     */
    public BufferedWriter getBufferedWriter(Path path, OpenOption... options) throws IOException {
        Files.createDirectories(path.getParent());
        LOG.log(Level.FINE, "Getting buffered writer {0} with options {1}", new Object[]{path,Arrays.asList(options)});        
        return Files.newBufferedWriter(path, options);
    }
        
    /**
     * Get an OutputStream for writing data for a given Path and OpenOptions.
     * 
     * @param path    The Path of the file to open.
     * @param options The OpenOption array to open the file with.
     * @return        The corresponding OutputStream
     * @throws IOException 
     */
    public OutputStream getOutputStream(Path path, OpenOption... options) throws IOException {
        Files.createDirectories(path.getParent());
        LOG.log(Level.FINE, "Getting OutputStream {0} with options {1}", new Object[]{path,Arrays.asList(options)});        
        return Files.newOutputStream(path, options);
    }
        


    private static FileSystem getFileSystem(String cacheName,String mountPoint, Properties p) throws ConfigurationServiceException {
        synchronized(fileSystems) {
            return fileSystems.computeIfAbsent(cacheName, (k) -> {return createFileSystem(k,mountPoint, p);});
        }
    }

    private synchronized static FileSystem createFileSystem(String cacheName, String mountPoint, Properties p) throws ConfigurationServiceException {
        try {
            final Properties bootstrapSystemProperties = BootstrapResourceUtils.getBootstrapSystemProperties();
            //Provided from upstream.
            String uri = p.getProperty("org.lsst.ccs.remote.server.uri", "");
            if ( uri.isEmpty() ) {
                //If not provided from upstream, check the new property
                uri = bootstrapSystemProperties.getProperty("org.lsst.ccs.remote.server.uri", "");
                if ( uri.isEmpty() ) {
                    //If the new property is not provided, use the old one.
                    uri = bootstrapSystemProperties.getProperty("org.lsst.ccs.config.remote.uri", "ccs://lsst-camera-dev.slac.stanford.edu/RestFileServer/");
                }
            }
            
            boolean cacheOnly = "true".equalsIgnoreCase(bootstrapSystemProperties.getProperty("org.lsst.ccs.config.remote.cacheOnly", "false"));
            Path cacheDir = Paths.get(System.getProperty("user.home") + "/ccs/cache/" + cacheName);
            LOG.log(Level.INFO, "Connecting to remote file server: {0} with mount point \"{1}\" (local cache {2})", new Object[]{uri, mountPoint, cacheDir});
            RestFileSystemOptions.CacheFallback cacheFallback = RestFileSystemOptions.CacheFallback.valueOf(p.getProperty("cacheFallback", "offline").toUpperCase());
            if (cacheOnly) {
                cacheFallback = RestFileSystemOptions.CacheFallback.ALWAYS;
            }
            if ( !uri.endsWith("/") ) {
                uri += "/";
            }
            URI mountPointURI = URI.create(mountPoint+"/");
            URI restRootURI = new URI(uri);
            Files.createDirectories(cacheDir);
            Map<String, Object> env = RestFileSystemOptions.builder()
                    .mountPoint(mountPointURI)
                    .cacheLocation(cacheDir.toFile())
                    .set(RestFileSystemOptions.CacheOptions.MEMORY_AND_DISK)
                    .ignoreLockedCache(true)
                    .set(cacheFallback)
                    // Somewhat ugly workaround for LSSTCCS-2418
                    .set(uri.contains("//lsst-camera-dev.slac.stanford.edu") ? RestFileSystemOptions.SSLOptions.TRUE : RestFileSystemOptions.SSLOptions.AUTO)
                    .build();
            return FileSystems.newFileSystem(restRootURI, env);
        } catch (IOException | URISyntaxException x) {
            throw new ConfigurationServiceException("Unable to create RestFileServerRemoteDAO", x);
        }
    }

    /**
     * Get the underlying FileSystem.
     * 
     * @return The FileSystem.
     */    
    public FileSystem getFileSystem() {
        return getFileSystem(cacheName, mountPoint, null);
    }
    //This is used internally to create the file system.
    private FileSystem getFileSystem(Properties p) {
        return getFileSystem(cacheName, mountPoint, p);
    }
    
    /**
     * Close the connection to the underlying FileSystem.
     * This method can be called at any point by any of the clients using any
     * of the RemoteFileServers sharing the same FileSystem.
     * 
     * Not particularly a nice implementation.
     * A better implementation could be to turn this class into an AgentService
     * and manage the lifetime of the FileSystem in the service itself, rather
     * than delegating to external clients.
     * 
     */
    public void close() {
        synchronized(fileSystems) {
            try {
                //If the file system has already been closed by another
                //client, then it's null in the map; in which case we don't
                //need to close it.
                FileSystem fs = fileSystems.get(cacheName);
                if ( fs != null ) {
                    if (fs.isOpen()) {
                        fs.close();
                        if ( fileSystems.remove(cacheName) == null ) {
                            throw new IllegalArgumentException("Something went wrong when clearing the cache while closing file system with cache "+cacheName+". Available caches: "+fileSystems.keySet());
                        }
                    }
                }
            } catch (IOException x) {
                LOG.log(Level.WARNING, "Error while closing rest file system", x);
            }
        }
    }

        
}
