package org.lsst.ccs.config;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Writer;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.rest.file.server.client.RestFileSystemOptions;

/**
 * A wrapper class to grant access to a remote FileSystem.
 * There can be multiple instances of this class. For the same cache name
 * they will share the same instance of the FileSystem.
 * 
 * @author LSST CCS Team
 */
public class RemoteFileServer {

    private static final java.util.logging.Logger LOG = Logger.getLogger(RemoteFileServer.class.getName());
    
    private final static Map<String,FileSystem> fileSystems = new ConcurrentHashMap<>();

    private final String cacheName;
    private final String mountPoint;
    
    /**
     * Create an instance of a remote file server with a given cache name.
     * The cache name will be used as the mount point for a local cache.
     * 
     * @param cacheName 
     */
    public RemoteFileServer(String descName, String mountPoint) {
        this.mountPoint = mountPoint.replace("/","");
        this.cacheName = descName+"/"+mountPoint+"/";
        getFileSystem();
    }


    /**
     * Get an InputStream for reading data for a given Path and OpenOptions.
     * 
     * @param path    The Path of the file to open.
     * @param options The OpenOption array to open the file with.
     * @return        The corresponding InputStream
     * @throws IOException 
     */
    public InputStream getInputStream(Path path, OpenOption... options) throws IOException {
        Files.createDirectories(path.getParent());       
        try ( InputStream in = Files.newInputStream(path, options)) {
            LOG.log(Level.FINE, "Loading input stream {0} with options {1}", new Object[]{path,Arrays.asList(options)});
            return in;
        } catch (IOException x) {                
            LOG.log(Level.WARNING, "Did not find {0} with options {1}", new Object[]{path,Arrays.asList(options)});                
            throw x;
        }
    }

    /**
     * Check if a file exists in the remote file server
     * @param path The path to check.
     * @return true/false if the path exists or not.
     */
    public boolean exists(Path path) {
        return Files.exists(path);       
    }

    
    /**
     * Get an BufferedWriter for writing data for a given Path and OpenOptions.
     * 
     * @param path    The Path of the file to open.
     * @param options The OpenOption array to open the file with.
     * @return        The corresponding BufferedWriter
     * @throws IOException 
     */
    public BufferedWriter getBufferedWriter(Path path, OpenOption... options) throws IOException {
        Files.createDirectories(path.getParent());
        LOG.log(Level.FINE, "Getting buffered writer {0} with options {1}", new Object[]{path,Arrays.asList(options)});        
        return Files.newBufferedWriter(path, options);
    }
        
    /**
     * Get an OutputStream for writing data for a given Path and OpenOptions.
     * 
     * @param path    The Path of the file to open.
     * @param options The OpenOption array to open the file with.
     * @return        The corresponding OutputStream
     * @throws IOException 
     */
    public OutputStream getOutputStream(Path path, OpenOption... options) throws IOException {
        Files.createDirectories(path.getParent());
        LOG.log(Level.FINE, "Getting OutputStream {0} with options {1}", new Object[]{path,Arrays.asList(options)});        
        return Files.newOutputStream(path, options);
    }
        


    private static FileSystem getFileSystem(String cacheName,String mountPoint) throws ConfigurationServiceException {
        synchronized(fileSystems) {
            return fileSystems.computeIfAbsent(cacheName, (k) -> {return createFileSystem(k,mountPoint);});
        }
    }

    private synchronized static FileSystem createFileSystem(String cacheName, String mountPoint) throws ConfigurationServiceException {
        try {
            LOG.log(Level.INFO, "Creating RemoteFileServer with cacheName {0} for mount point {1}", new Object[]{cacheName,mountPoint});
            final Properties bootstrapSystemProperties = BootstrapResourceUtils.getBootstrapSystemProperties();
            String uri = bootstrapSystemProperties.getProperty("org.lsst.ccs.config.remote.uri", "ccs://lsst-camera-dev.slac.stanford.edu/RestFileServer/");
            boolean cacheOnly = "true".equalsIgnoreCase(bootstrapSystemProperties.getProperty("org.lsst.ccs.config.remote.cacheOnly", "false"));
            if ( !uri.endsWith("/") ) {
                uri += "/";
            }
            URI mountPointURI = URI.create(mountPoint+"/");
            URI restRootURI = new URI(uri);
            Path cacheDir = Paths.get(System.getProperty("user.home") + "/ccs/cache/" + cacheName);
            Files.createDirectories(cacheDir);
            Map<String, Object> env = RestFileSystemOptions.builder()
                    .mountPoint(mountPointURI)
                    .cacheLocation(cacheDir.toFile())
                    .set(RestFileSystemOptions.CacheOptions.MEMORY_AND_DISK)
                    .ignoreLockedCache(true)
                    .set(cacheOnly ? RestFileSystemOptions.CacheFallback.ALWAYS : RestFileSystemOptions.CacheFallback.OFFLINE)
                    // Somewhat ugly workaround for LSSTCCS-2418
                    .set(uri.contains("//lsst-camera-dev.slac.stanford.edu") ? RestFileSystemOptions.SSLOptions.TRUE : RestFileSystemOptions.SSLOptions.AUTO)
                    .build();
            return FileSystems.newFileSystem(restRootURI, env);
        } catch (IOException | URISyntaxException x) {
            throw new ConfigurationServiceException("Unable to create RestFileServerRemoteDAO", x);
        }
    }

    /**
     * Get the underlying FileSystem.
     * 
     * @return The FileSystem.
     */
    public FileSystem getFileSystem() {
        return getFileSystem(cacheName, mountPoint);
    }
    
    /**
     * Close the connection to the underlying FileSystem.
     * This method can be called at any point by any of the clients using any
     * of the RemoteFileServers sharing the same FileSystem.
     * 
     * Not particularly a nice implementation.
     * A better implementation could be to turn this class into an AgentService
     * and manage the lifetime of the FileSystem in the service itself, rather
     * than delegating to external clients.
     * 
     */
    public void close() {
        synchronized(fileSystems) {
            try {
                //If the file system has already been closed by another
                //client, then it's null in the map; in which case we don't
                //need to close it.
                FileSystem fs = fileSystems.get(cacheName);
                if ( fs != null ) {
                    if (fs.isOpen()) {
                        fs.close();
                        if ( fileSystems.remove(cacheName) == null ) {
                            throw new IllegalArgumentException("Something went wrong when clearing the cache while closing file system with cache "+cacheName+". Available caches: "+fileSystems.keySet());
                        }
                    }
                }
            } catch (IOException x) {
                LOG.log(Level.WARNING, "Error while closing rest file system", x);
            }
        }
    }

        
}
