package org.lsst.ccs.config;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentInfo;

/**
 * A Class to chain together SingleCategoryTagData entries for a given category.
 * 
 * This class is to be used to figure out which SingleCategoryTag a configuration
 * parameter value came from.
 * 
 * 
 * @author The LSST CCS Team
 */
public class CategoryDataChain {

    private static final Logger LOG = Logger.getLogger(CategoryDataChain.class.getName());
    
    private final List<SingleCategoryTag> singleCategoryTagsChain = new CopyOnWriteArrayList<>();
    private final Map<SingleCategoryTag,Map<String,ConfigurationParameterTaggedValue>> tagDataMap = new ConcurrentHashMap<>();
    private final Map<SingleCategoryTag,SingleCategoryTagData> singleCategoryTagDataMap = new ConcurrentHashMap<>();
    //The key is the path of the configuration parameter
    private final Map<String,ConfigurationParameterTaggedValue> taggedValues = new ConcurrentHashMap<>();
    
    private final String category;
    
    public CategoryDataChain(String category) {
        this.category = category;
    }

    /**
     * Add a SingleCategoryTagData to the chain of existing SingleCategoryTagData.
     * If the corresponding SingleCategoryTag already exists, then its value is
     * replaced with the input tagData at its original position, otherwise we add the new
     * data at the end of the chain.
     * 
     * @param tagData The SingleCategoryTagData to be added
     */
    public void addSingleCategoryTagData(SingleCategoryTagData tagData, AgentInfo ai) {
        if ( !tagData.getSingleCategoryTag().getCategory().equals(category) ) {
            throw new IllegalArgumentException("Incompatible category \""+tagData.getSingleCategoryTag().getCategory()+"\" cannot be added to category data chain for \""+category+"\"");
        }
        SingleCategoryTag singleTag = tagData.getSingleCategoryTag();
        Map<String, ConfigurationParameterTaggedValue> taggedData = new ConcurrentHashMap<>();
        Map<String, String> inputData = tagData.getConfigurationDataForAgent(ai);
        for (Entry<String, String> e : inputData.entrySet()) {
            String path = e.getKey();
            taggedData.put(path, new ConfigurationParameterTaggedValue(path, e.getValue(), singleTag));
        }
        singleCategoryTagDataMap.put(singleTag, tagData);
        addTaggedDataForSingleCategoryTag(singleTag, taggedData);                        
    }
    
    private void addTaggedDataForSingleCategoryTag(SingleCategoryTag singleTag, Map<String,ConfigurationParameterTaggedValue> taggedData) {
        synchronized(taggedValues) {
            int indexOfExistingData = singleCategoryTagsChain.indexOf(singleTag);
            if (indexOfExistingData >= 0) {
                singleCategoryTagsChain.remove(indexOfExistingData);
                singleCategoryTagsChain.add(indexOfExistingData, singleTag);
            } else {
                singleCategoryTagsChain.add(singleTag);
            }
            tagDataMap.put(singleTag, taggedData);
            taggedValues.clear();
        }        
    }
    
    /**
     * 
     * @param chain 
     */
//    public void addCategoryDataChain(CategoryDataChain chain) {
//        if ( !chain.getCategory().equals(getCategory()) ) {
//            throw new IllegalArgumentException("Incompatible category \""+chain.getCategory()+"\" cannot be added to category data chain for \""+category+"\"");
//        }
//        synchronized(taggedValues) {
//            for (SingleCategoryTag singleTag : chain.singleCategoryTagsChain) {
//                addTaggedDataForSingleCategoryTag(singleTag, chain.tagDataMap.get(singleTag));
//            }
//        }
//    }
    
    /**
     * Get the category for this CategoryDataChain.
     * 
     * @return The category for this CategoryDataChain.
     */
    public String getCategory() {
        return category;
    }
    
    /**
     * Get a copy of the ConfiguraiontParameterTaggedValue map.
     * This is built based on the chain of SingleCategoryTags.
     * This map contains the values as defined in their top-most tag.
     * 
     * @return A copy of the Map containing the ConfigurationParameterTaggedValues
     */
    public Map<String,ConfigurationParameterTaggedValue> getConfigurationParameterTaggedValues() {
        synchronized(taggedValues) {
            if ( taggedValues.isEmpty() ) {
                for ( SingleCategoryTag singleTag : singleCategoryTagsChain ) {
                    taggedValues.putAll(tagDataMap.get(singleTag));
                }
            }            
            return new HashMap<>(taggedValues);
        }
    }
    
    /**
     * For a given parameter path get the List of the ConfigurationParameterTaggedValues 
     * as defined in the data chain. The first element is the value as defined by the bottom
     * tag, while the last element is the one defined by the top-most tag.
     * @param path The configuration parameter path
     * @return The List of ConfigurationParameterTaggedValues
     */
    public List<String> getConfigurationParameterValueList(String path) {
        List<String> res = new ArrayList<>();
        for ( SingleCategoryTag singleTag : singleCategoryTagsChain ) {
            ConfigurationParameterTaggedValue cp = tagDataMap.get(singleTag).get(path);
            if ( cp != null ) {
                res.add(cp.getParameterValue());
            }
        }
        return res;
    }
    
    public List<SingleCategoryTag> getSingleCategoryTagList() {
        return new ArrayList(singleCategoryTagsChain);
    }
    
    public SingleCategoryTag getTopMostCategoryTag() {
        return singleCategoryTagsChain.get(singleCategoryTagsChain.size()-1);
    }
    
    public SingleCategoryTagData getSingleCategoryTagData(SingleCategoryTag tag) {
        return singleCategoryTagDataMap.get(tag);
    }
    
    public boolean needsSaving() {
        for ( SingleCategoryTag singleTag : singleCategoryTagsChain ) {
            if ( getSingleCategoryTagData(singleTag).needsSaving() ) {
                return true;
            }
        }        
        return false;
    }
}
