package org.lsst.ccs.config;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.net.URL;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 * Reads and Writes properties files from/to a FileSystem, using 
 *
 * @author LSST CCS Team
 */
class FileWriterProvider extends WriterProvider {

    private static final java.util.logging.Logger LOG = Logger.getLogger(FileWriterProvider.class.getName());
    
    @Override
    public PrintWriter getPrintWriter(String fileName) throws IOException {
        String tmpFileName = fileName;
        //Check if the fileName specifies a directory.
        //If it does and it exists, use it. Otherwise fetch it from the bootstrap path
        File parentFile = new File(fileName).getParentFile();
        if ( parentFile == null ) {
            //Check if the properties file already exists in the Bootstrap Environment
            String pathInBootstrap = BootstrapResourceUtils.getPathOfPropertiesFileInUserResourceDirectories(fileName);
            //If it does not exist...
            if (pathInBootstrap == null) {
                // then check if a there is a Bootstrap user defined directory in which to write it.                
                String topMostUserDirectory = BootstrapResourceUtils.getTopUserResourceDirectory();
                //If there is not Bootstrap user Directory defined.....
                if (topMostUserDirectory == null) {
                    throw new RuntimeException("There is no external resource directory in which to write the file. Please make sure to"
                            + " define one using environment variable CCS_RESOURCE_PATH.");
                } else {
                    fileName = topMostUserDirectory + fileName;
                }
            } else {
                fileName = pathInBootstrap;
            }
        } else {
            if ( !parentFile.exists() ) {
                parentFile.mkdirs();
            }
        }
        LOG.log(Level.INFO, "Saving {0} to {1}", new Object[]{tmpFileName, fileName});
 
        return new PrintWriter(fileName, "ISO-8859-1");
    }

    @Override
    public InputStream getInputStream(String fileName) throws IOException {
        return BootstrapResourceUtils.getBootstrapResource(fileName);
    }
    
    
    @Override
    Set<String> findMatchingPropertiesFiles(String pattern) {
        return BootstrapResourceUtils.findMatchingResources(pattern);
    }

    @Override
    String locateFile(String fileName) {
        try {
            URL fileUrl = BootstrapResourceUtils.getResourceURL(fileName);
            return fileUrl != null ? fileUrl.getPath() : null;
        } catch (Exception e) {
        }
        return null;
    }        
}
