package org.lsst.ccs.config;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.freehep.util.VersionComparator;
import org.lsst.ccs.bus.data.AgentInfo;

/**
 * A Class containing data for a single Configuration Category Tag.
 * 
 * This object contains information for a subset of the category tag's configuration
 * parameters, namely the value and the comments associated with the parameter.
 * 
 * This class is meant to be used when reading or writing configuration parameters.
 * 
 * This class is also used internally by the configuration service to chain 
 * tags together and keep track of the origin of a configuration parameter's value.
 * 
 * @author The LSST CCS Team
 */
public class SingleCategoryTagData {

    private static final Logger LOG = Logger.getLogger(SingleCategoryTagData.class.getName());
    
    private final SingleCategoryTag singleCategoryTag;
    private final Map<String,String> comments = new ConcurrentHashMap<>();
    private final Map<String,String> conditions = new ConcurrentHashMap<>();
    private final Map<String,List<String>> conditionsToPars = new ConcurrentHashMap<>();
    private final Map<String,String> data = Collections.synchronizedMap(new LinkedHashMap<>());
    private boolean needsSaving = false;
    
    public SingleCategoryTagData(SingleCategoryTag singleCategoryTag) {
        this.singleCategoryTag = singleCategoryTag;
    }

    //Copy Constructor
    public SingleCategoryTagData(SingleCategoryTagData singleCategoryTagData) {
        this.singleCategoryTag = new SingleCategoryTag(singleCategoryTagData.singleCategoryTag);
        this.comments.putAll(singleCategoryTagData.comments);
        this.conditions.putAll(singleCategoryTagData.conditions);
        this.data.putAll(singleCategoryTagData.data);
        this.conditionsToPars.putAll(singleCategoryTagData.conditionsToPars);
    }

    public SingleCategoryTag getSingleCategoryTag() {
        return singleCategoryTag;
    }
    
    public Map<String, String> getConfigurationData() {
        return data;
    }
    public Map<String, String> getConfigurationDataForAgent(AgentInfo agentInfo) {
        Map<String,String> dataForAgent = new LinkedHashMap<>();
        for ( String key : data.keySet() ) {
            String condition = getConditionParameter(key);
            if ( condition != null ) {
                if ( ! acceptConditionForAgent(condition,agentInfo) ) {
                    continue;
                }
            }
            dataForAgent.put(key, data.get(key));
        }
        return dataForAgent;
    }
    
    private boolean acceptConditionForAgent(String condition, AgentInfo ai) {
        String[] conditionComponents = condition.replace("[", "").replace("]", "").trim().split(" ");
        if ( conditionComponents.length != 3 ) {
            throw new RuntimeException("Invalid condition "+condition);
        }
        String versionType = conditionComponents[0].trim();
        String operand = conditionComponents[1].trim();
        String versionValue = conditionComponents[2].trim();
        
        String currentVersionValue = ai.getAgentProperty(versionType).replace("-SNAPSHOT", "");
        if ( currentVersionValue == null ) {
            throw new RuntimeException("Could not find current version for \""+versionType+"\" in "+ai.getAgentProperties());
        }
        
        int comparison = VersionComparator.compareVersion(currentVersionValue, versionValue);
        if (comparison == 0 && operand.contains("=") ) {
            return true;
        }
        
        if ( comparison > 0 && operand.contains(">") ) {
            return true;
        }
        
        if ( comparison < 0 && operand.contains("<") ) {
            return true;
        }
        return false;
        
    }
    
    public String getCommentParameter(String prop) {
        return comments.get(prop);
    }
    
    public String getConditionParameter(String prop) {
        return conditions.get(prop);
    }

    public void mergeSingleCategoryTagData(SingleCategoryTagData inputData) {
        if ( inputData == null ) {
            return;
        }
        if ( !inputData.singleCategoryTag.equals(singleCategoryTag) ) {
            throw new RuntimeException("It's not possible to merge the data from different category tags. Merging input: "+inputData.singleCategoryTag+" onto "+singleCategoryTag);
        }
        comments.putAll(inputData.comments);
        data.putAll(inputData.data);
    }
    
    public void load(InputStream is) throws IOException {
        if ( is == null ) {
            return;
        }
        try (BufferedReader reader = new BufferedReader( new InputStreamReader(is) ) ) {
            String comment = "";
            String condition = "";
            boolean conditionBlock = false;           
            int lineCount = 0;
            //Go over the input line by line and add them together as long as they
            //start with either "#" or "!" (ignoring spaces)
            //Once we find a line defining a property we store the previous comment 
            //(if it exists) in a Map with the property key
            while (reader.ready()) {
                lineCount++;
                String line = reader.readLine();
                String trimmedLine = line.trim();

                //This is a comment
                if (trimmedLine.startsWith("#") || trimmedLine.startsWith("!")) {
                    comment += line+"\n";
                }
                //This is a condition and possibly the start of a condition block
                else if ( trimmedLine.startsWith("[") ) {
                    if ( conditionBlock ) {
                        throw new RuntimeException("Conditions are not allowed inside a condition block "+trimmedLine+" (line "+lineCount+")");                                
                    }
                    if ( condition.isEmpty() ) {
                        if ( trimmedLine.endsWith("{") ) {
                            conditionBlock = true;
                            condition = trimmedLine.replace("{","").trim();
                        } else {
                            condition = line.trim();
                        }
                    } else {
                        throw new RuntimeException("We currently support only one condition for configuration parameter. Please remove either "+condition+" or "+line+" (line "+lineCount+")");
                    }                    
                } 
                //This is the start of a condition block
                else if ( trimmedLine.startsWith("{") ) {
                    if ( conditionBlock ) {
                        throw new RuntimeException("Nested condition blocks are not supported (line "+lineCount+")");                        
                    }
                    conditionBlock = true;
                }
                //This is the end of a condition block
                else if ( trimmedLine.startsWith("}") ) {
                    if ( ! conditionBlock ) {
                        throw new RuntimeException("Unmatched end of condition block \"}\" (line "+lineCount+")");
                    }
                    conditionBlock = false;
                    condition = "";
                }
                //This is a property line definition
                else {
                    int propertyEnd = trimmedLine.indexOf("=");
                    if (propertyEnd < 0) {
                        propertyEnd = trimmedLine.indexOf(":");
                    }
                    if (propertyEnd < 0) {
                        propertyEnd = trimmedLine.length() - 1;
                    }
                    if (propertyEnd < 0) {
                        if (!trimmedLine.isEmpty()) {
                            LOG.log(Level.WARNING, "Skipping line {0}", line);
                        }
                    } else {
                        String propertyKey = trimmedLine.substring(0, propertyEnd).trim();
                        if (!comment.isEmpty()) {
                            comments.put(propertyKey, comment);
                            comment = "";
                        }
                        if ( !condition.isEmpty() ) {
                            conditions.put(propertyKey, condition);
                            conditionsToPars.computeIfAbsent(condition, (c) -> new ArrayList<>()).add(propertyKey);
                            if ( ! conditionBlock ) {
                                condition = "";
                            }
                        }
                        String propertyValue = trimmedLine.substring(propertyEnd+1).trim();
                        data.put(propertyKey, propertyValue);
                    }
                }
            }
            if ( conditionBlock ) {
                throw new RuntimeException("A condition block is still open. Needs } to close it (line "+lineCount+")");
                        
            }

        }
        
        //Sanity check: all comments must have a corresponding property key
        comments.keySet().stream().filter(prop -> ( !data.containsKey(prop) )).forEachOrdered(prop -> {
            throw new RuntimeException("Property name "+prop+" does not exist for comment "+comments.get(prop));
        });
        
    }

    /**
     * Dump the content of this object in a String.
     * The configuration parameters will be sorted alphabetically and the
     * comments will be displayed above a property if available.
     * @return The full data of this object in a String.
     */
    public String fullDataString() {
        StringBuilder result = new StringBuilder();
        Set<String> doneConditions = new HashSet<>();
        data.keySet().forEach(prop -> {
            String condition = getConditionParameter(prop);
            if ( condition != null ) {
                if (doneConditions.add(condition)) {
                    boolean singleCondition = conditionsToPars.get(condition).size() == 1;
                    result.append(condition);
                    if (!singleCondition) {
                        result.append(" {");
                    }
                    result.append("\n");
                    for (String par : conditionsToPars.get(condition)) {
                        writeParameter(par, result, !singleCondition);
                    }
                    if (!singleCondition) {
                        result.append("}\n");
                    }
                }
            } else {
                writeParameter(prop,result, false);
            }
        });
        return result.toString();
    }
    
    private void writeParameter(String par, StringBuilder sb, boolean indent) {
        String comment = getCommentParameter(par);        
        if (comment != null) {
            if ( indent ) {
                sb.append("   ");
            }
            sb.append(comment);
        }
            if ( indent ) {
                sb.append("   ");
            }
        sb.append(par).append(" = ").append(data.get(par)).append("\n");        
    }
    
    public boolean needsSaving() {
        return needsSaving;
    }
    
    public void removeConfigurationParameter(String parameterPath) {
        data.remove(parameterPath);
        needsSaving = true;
    }
    
}
