package org.lsst.ccs.config;

import java.util.Map;

/**
 * An interface that gives a component more control on how to validate / set a
 * bulk change of configuration parameters. The two configuration steps covered
 * by this interface are: - parameter bulk validation - parameter bulk setting
 *
 * <p>
 * Bulk changes of configuration parameters happen anytime there is a configuration change :
 * <ul>
 * <li>loading configurations</li>
 * <li>dropping changes</li>
 * <li>changing one parameter</li>
 * <li>committing a set of submitted changes of configuration parameters.</li>
 * </ul>
 * A bulk change consists in traveling the tree of components twice : a first
 * time to validate the submitted changes and a second time to set the submitted
 * changes. Components may be skipped if there are no changes for it.
 * This interface offers two methods for implementing those two steps
 * for a given component.
 *
 * <p>
 * The bulk validation step is intended to validate that all (or a subset) of
 * the provided parameters full view is in a consistent state. If problems are
 * detected an exception is meant to be thrown, in which case the bulk change
 * process is interrupted and the configuration state is left unchanged.
 * Please note that the provided Map will contain all parameters with their values,
 * whether the parameter has been changed or not. If not changed the parameter's value
 * is the current value.
 *
 * <p>
 * The bulk parameter setting method is invoked after a successful validation
 * step. Exceptions should normally not be thrown at this step. If it happens,
 * the configuration state may have been modified.
 * The provided Map will only contain the parameters that have changed and that
 * need to be set.
 * 
 * @author The LSST CCS Team
 */
public interface ConfigurationBulkChangeHandler {
    
    /**
     * Validate a set of proposed changes of configuration parameters of this
     * component.
     * 
     * This method is passed a map of all configuration parameter names that belong to this component, mapped to their value,
     * where the value is either the new proposed value if it has been changed, or
     * the current value of the parameter if it has not been changed.
     * 
     * This method is meant to throw an exception when one or more
     * parameters have a proposed value that is decided as not valid by the
     * user. This method is called by the framework only when necessary.
     *
     * @param parametersView the whole set of parameters for this component with
     * their value to be validated. it maps the parameter names to their value.
     */
    public default void validateBulkChange(Map<String, Object> parametersView) {
        
    }
    
    /**
     * Set one or more configuration parameters to a new value. Configuration
     * parameters that have been set by this method must have been set to the
     * value defined in the passed map, otherwise the framework will detect it
     * and throw a {@code BulkSettingException}. All parameters present in the
     * passed map do not have to be handled by this method : the parameters that
     * have not been set at this step will be set by the framework using an
     * available setter method, or directly on the corresponding fields using
     * introspection. This method is called by the framework only when
     * necessary.
     *
     * Final configuration parameters will be involved in this step only during
     * the initial safe or build step if they have been changed.
     * 
     * @param parametersView a map of the parameters that have changed and
     * need to be set. The map contains the name of the parameters that are to be
     * set and their new value.
     * 
     */
    public default void setParameterBulk(Map<String, Object> parametersView) {
        
    }
    
    
}
