package org.lsst.ccs.config;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.rest.file.server.client.RestFileSystemOptions;

/**
 * A wrapper class to grant access to a remote FileSystem.
 * There can be multiple instances of this class. For the same cache name
 * they will share the same instance of the FileSystem.
 * 
 * @author LSST CCS Team
 */
public class RemoteFileServer {

    private static final java.util.logging.Logger LOG = Logger.getLogger(RemoteFileServer.class.getName());
    
    private final static Map<String,FileSystem> fileSystems = new ConcurrentHashMap<>();

    private final String cacheName;
    
    /**
     * Create an instance of a remote file server with a given cache name.
     * The cache name will be used as the mount point for a local cache.
     * 
     * @param cacheName 
     */
    public RemoteFileServer(String cacheName) {
        this.cacheName = cacheName;
        getFileSystem();
    }


    /**
     * Get an InputStream for reading data for a given Path and OpenOptions.
     * 
     * @param path    The Path of the file to open.
     * @param options The OpenOption array to open the file with.
     * @return        The corresponding InputStream
     * @throws IOException 
     */
    public InputStream getInputStream(Path path, OpenOption... options) throws IOException {
        
        try ( InputStream in = Files.newInputStream(path, options)) {
            LOG.log(Level.FINE, "Loading input stream {0} with options {1}", new Object[]{path,Arrays.asList(options)});
            return in;
        } catch (IOException x) {                
            LOG.log(Level.WARNING, "Did not find {0} with options {1}", new Object[]{path,Arrays.asList(options)});                
            throw new IOException(x);
        }
    }
    
    /**
     * Get an BufferedWriter for writing data for a given Path and OpenOptions.
     * 
     * @param path    The Path of the file to open.
     * @param options The OpenOption array to open the file with.
     * @return        The corresponding BufferedWriter
     * @throws IOException 
     */
    public BufferedWriter getBufferedWriter(Path path, OpenOption... options) throws IOException {
        Files.createDirectories(path.getParent());
        LOG.log(Level.FINE, "Getting buffered writer {0} with options {1}", new Object[]{path,Arrays.asList(options)});        
        return Files.newBufferedWriter(path, options);
    }
        


    private static FileSystem getFileSystem(String cacheName) throws ConfigurationServiceException {
        synchronized(fileSystems) {
            return fileSystems.computeIfAbsent(cacheName, (k) -> {return createFileSystem(k);});
        }
    }

    private synchronized static FileSystem createFileSystem(String cacheName) throws ConfigurationServiceException {
        try {
            LOG.log(Level.INFO, "Creating RemoteFileServer with cacheName {0}", cacheName);
            final Properties bootstrapSystemProperties = BootstrapResourceUtils.getBootstrapSystemProperties();
            String uri = bootstrapSystemProperties.getProperty("org.lsst.ccs.config.remote.uri", "ccs://lsst-camera-dev.slac.stanford.edu/RestFileServer/");
            boolean cacheOnly = "true".equalsIgnoreCase(bootstrapSystemProperties.getProperty("org.lsst.ccs.config.remote.cacheOnly", "false"));
            URI restRootURI = new URI(uri);
            Path cacheDir = Paths.get(System.getProperty("user.home") + "/ccs/cache/" + cacheName);
            Files.createDirectories(cacheDir);
            Map<String, Object> env = RestFileSystemOptions.builder()
                    .cacheLocation(cacheDir.toFile())
                    .set(RestFileSystemOptions.CacheOptions.MEMORY_AND_DISK)
                    .ignoreLockedCache(true)
                    .set(cacheOnly ? RestFileSystemOptions.CacheFallback.ALWAYS : RestFileSystemOptions.CacheFallback.OFFLINE)
                    .build();
            return FileSystems.newFileSystem(restRootURI, env);
        } catch (IOException | URISyntaxException x) {
            throw new ConfigurationServiceException("Unable to create RestFileServerRemoteDAO", x);
        }
    }

    /**
     * Get the underlying FileSystem.
     * 
     * @return The FileSystem.
     */
    public FileSystem getFileSystem() {
        return getFileSystem(cacheName);
    }
    
    /**
     * Close the connection to the underlying FileSystem.
     * This method can be called at any point by any of the clients using any
     * of the RemoteFileServers sharing the same FileSystem.
     * 
     * Not particularly a nice implementation.
     * A better implementation could be to turn this class into an AgentService
     * and manage the lifetime of the FileSystem in the service itself, rather
     * than delegating to external clients.
     * 
     */
    public void close() {
        synchronized(fileSystems) {
            try {
                //If the file system has already been closed by another
                //client, then it's null in the map; in which case we don't
                //need to close it.
                FileSystem fs = fileSystems.get(cacheName);
                if ( fs != null ) {
                    if (fs.isOpen()) {
                        fs.close();
                        fileSystems.remove(cacheName);
                    }
                }
            } catch (IOException x) {
                LOG.log(Level.WARNING, "Error while closing rest file system", x);
            }
        }
    }

        
}
