package org.lsst.ccs.config;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.ConfigurationParameterType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.utilities.constraints.Constraints;
import org.lsst.ccs.utilities.conv.InputConversionEngine;
import org.lsst.ccs.utilities.conv.TypeConversionException;
import org.lsst.ccs.utilities.conv.TypeUtils;
import org.lsst.ccs.utilities.structs.ParameterPath;
import org.lsst.ccs.utilities.structs.ViewValue;

/**
 *
 * @author LSST CCS Team.
 */
public class ConfigurationParameterHandler implements Comparable<ConfigurationParameterHandler>{
     
    private final ParameterPath parameterPath;
    
    /** The annotation associated to this parameter. Cannot be null */
    private final ConfigurationParameter annotation;
    
    /** The @ConfigurationParameterChanger annotated methods. Can be null */
    private final Method parameterChangerMethod;
    
    /** The parameter itself. Cannot be null */
    private final Field parameterField;
    
    private final Object target;
    
    private final ConfigurationParameterType configType;
    
    private final boolean hasLength;
    
    private String externalCategory = null;
    
    private ConfigurationParameterDescription configurationParameterDescription;
    
    private static final Logger LOG = Logger.getLogger(ConfigurationParameterHandler.class.getName());    
    
    ConfigurationParameterHandler(String componentName, Method m, Field f, Object target, ConfigurationParameterDescription parDescription) {
        this.parameterChangerMethod = m;
        // Note use setAccessible() only once. Otherwise even just looking at the
        // parameter value will be a mutating operation!
        if (m != null) {m.setAccessible(true);}
        this.parameterField = f;
        this.target = target;
        parameterField.setAccessible(true);
        annotation = f.getAnnotation(ConfigurationParameter.class);
        this.configurationParameterDescription = parDescription;

        String tmpParName = annotation.name().isEmpty() ? f.getName() : annotation.name();
        if ( configurationParameterDescription != null && configurationParameterDescription.getName() != null ) {
            tmpParName = configurationParameterDescription.getName();
        }
        
        this.parameterPath = new ParameterPath(componentName, tmpParName);
        if ( annotation.isFinal() ) {
            configType = ConfigurationParameterType.FINAL;
        } else if ( annotation.isBuild() || (annotation.category() != null && annotation.category().equals("build")) ) {
            configType = ConfigurationParameterType.BUILD;
        } else if ( annotation.isReadOnly() ) {
            configType = ConfigurationParameterType.READ_ONLY;
        } else {
            configType = ConfigurationParameterType.RUNTIME;        
        }

        Class type = parameterField.getType();
        hasLength = type.isArray() || Collection.class.isAssignableFrom(type) || Map.class.isAssignableFrom(type);
    }

    /**
     * Single setting of a parameter. It first attempts to find the
     * corresponding @ParameterChanger annotated method. If not found, it 
     * attempts to set the corresponding @Parameter annotated field.
     *
     * @param parameterName the parameter
     * @param arg the new value for the parameter
     */
    void invokeSetParameter(Object arg) {
            Method m = parameterChangerMethod;
            try {
                if (m != null) {
                    m.invoke(target, arg);
                } else {                    
                    Field f = parameterField;
                    f.set(target, arg);
                }
            } catch (IllegalAccessException | InvocationTargetException ex ) {
                Throwable cause = ex.getCause();
                String msg = cause == null ? ex.toString() : ex.getCause().getMessage();
                throw new RuntimeException("at parameter " + parameterPath + " : " + msg, ex);
            }
    }
    
    final void acceptValue(Object obj) {        
        Class type = parameterField.getType();        
        //For the future: if we want to make sure no Configuration Parameters are initialized to null.
//        if ( !type.isPrimitive() && obj == null ) {
//            throw new IllegalArgumentException("Configuration parameter cannot be initialized to null");
//        }
        if ( hasLength && getMaxLength() > 0 ) {
            if ( obj == null ) {
                throw new IllegalArgumentException("Null value for field "+parameterField.getName()+". Make sure to initialize the field.");
            }
            int objectSize = -1;
            if ( type.isArray() ) {
                objectSize = Array.getLength(obj);
            } else if (Collection.class.isAssignableFrom(type)) {
                objectSize = ((Collection)obj).size();
            } else if (Map.class.isAssignableFrom(type)) {
                objectSize = ((Map)obj).size();
            }
            if ( objectSize > getMaxLength() ) {
                throw new RuntimeException("Size of committed value ("+objectSize+") exceeds the maximum length:"+getMaxLength());
            }
        }
    }
    
    public ParameterPath getParameterPath() {
        return parameterPath;
    }
    
    public String getCategory() {        
        if (externalCategory != null) {
            return externalCategory;
        }
        if ( configurationParameterDescription != null && configurationParameterDescription.getCategory() != null ) {
            return configurationParameterDescription.getCategory();
        }
        return annotation.category();
    }
    
    void setCategory(String category) {
        this.externalCategory = category;
    }
    
    public boolean hasConfigChangerMethod() {
        return parameterChangerMethod != null;
    }
    public Method getConfigChangerMethod() {
        return parameterChangerMethod;
    }
    
    public boolean isFinal() {
        return configType == ConfigurationParameterType.FINAL;
    }

    public boolean isReadOnly() {
        return configType == ConfigurationParameterType.READ_ONLY;
    }

    public boolean isBuild() {
        return configType == ConfigurationParameterType.BUILD;
    }
    
    public ConfigurationParameterType getParameterType() {
        return configType;
    }
    
    public String getUnits() {
        return annotation.units();
    }
    
    String getDescription() {
        if ( configurationParameterDescription != null && configurationParameterDescription.getDescription() != null ) {
            return configurationParameterDescription.getDescription();
        }
        return annotation.description();
    }
    
    String getRange() {
        return annotation.range();
    }
    
    public final void checkAgainstConstraints(Object val) {
        checkAgainstConstraints(val, true);
    }
    private void checkAgainstConstraints(Object val,boolean throwException) {
        String range = annotation.range();
        if (range != null) {
            try {
                Constraints.check(val, range);
            } catch (Exception e) {
                if ( throwException ) {
                    throw new RuntimeException("Range check failed for parameter "+getParameterPath(),e);
                } else {
                    LOG.log(Level.WARNING, "Range check validation failed for parameter {0}. Value {1} is outside the defined range {2}.", new Object[] {getParameterPath(), getValue(), range});
                }
            }
        }
    }
        
    String getParameterName() {
        return parameterPath.getParameterName();
    }
    
    String getComponentName() {
        return parameterPath.getComponentName();
    }
    
    Type getType() { 
        return parameterField.getGenericType();
    }
    
    final Field getField() {
        return parameterField;
    }
    
    final int getMaxLength() {
        return annotation.maxLength();
    }
    
    final boolean hasLength() {
        return hasLength;
    }
    
    final String getValue() {
        return TypeUtils.stringify(getObjectValue());
    }
    
    final Object getObjectValue() {
        Object res;
        try {
            res = parameterField.get(target);
        } catch (IllegalAccessException ex) {
            throw new RuntimeException("could not access " + parameterField.getName(), ex);
        }
        return res;
    }
    
    public ViewValue convert(String strValue) {
        try {
            Object val = InputConversionEngine.convertArgToType(strValue,
                    parameterField.getGenericType());
            return new ViewValue(strValue, val);
        } catch (TypeConversionException ex) {
            throw new IllegalArgumentException(
                    "failure converting : "+ parameterPath.toString()
                            + " with value "+ strValue
                            + " to type "+ parameterField.getGenericType().getTypeName(),
                    ex);
        }
    }

    @Override
    public int compareTo(ConfigurationParameterHandler o) {
        return this.getParameterPath().compareTo(o.getParameterPath());
    }
    


}
