package org.lsst.ccs.config;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Logger;

/**
 * A Class to chain together SingleCategoryTagData entries for a given category.
 * 
 * This class is to be used to figure out which SingleCategoryTag a configuration
 * parameter value came from.
 * 
 * 
 * @author The LSST CCS Team
 */
public class CategoryDataChain {

    private static final Logger LOG = Logger.getLogger(CategoryDataChain.class.getName());
    
    private final List<SingleCategoryTag> singleCategoryTagsChain = new CopyOnWriteArrayList<>();
    private final Map<SingleCategoryTag,Map<String,ConfigurationParameterTaggedValue>> tagDataMap = new ConcurrentHashMap<>();
    private final Map<String,ConfigurationParameterTaggedValue> taggedValues = new ConcurrentHashMap<>();
    
    private final String category;
    
    public CategoryDataChain(String category) {
        this.category = category;
    }

    /**
     * Add a SingleCategoryTagData to the chain of existing SingleCategoryTagData.
     * If the corresponding SingleCategoryTag already exists, then its value is
     * replaced with the input tagData at its original position, otherwise we add the new
     * data at the end of the chain.
     * 
     * @param tagData The SingleCategoryTagData to be added
     */
    public void addSingleCategoryTagData(SingleCategoryTagData tagData) {
        if ( !tagData.getSingleCategoryTag().getCategory().equals(category) ) {
            throw new IllegalArgumentException("Incompatible category \""+tagData.getSingleCategoryTag().getCategory()+"\" cannot be added to category data chain for \""+category+"\"");
        }
        SingleCategoryTag singleTag = tagData.getSingleCategoryTag();
        Map<String, ConfigurationParameterTaggedValue> taggedData = new ConcurrentHashMap<>();
        Map<String, String> inputData = tagData.getConfigurationData();
        for (Entry<String, String> e : inputData.entrySet()) {
            String path = e.getKey();
            taggedData.put(path, new ConfigurationParameterTaggedValue(path, e.getValue(), singleTag));
        }
        addTaggedDataForSingleCategoryTag(singleTag, taggedData);                        
    }
    
    private void addTaggedDataForSingleCategoryTag(SingleCategoryTag singleTag, Map<String,ConfigurationParameterTaggedValue> taggedData) {
        synchronized(taggedValues) {
            int indexOfExistingData = singleCategoryTagsChain.indexOf(singleTag);
            if (indexOfExistingData >= 0) {
                singleCategoryTagsChain.remove(indexOfExistingData);
                singleCategoryTagsChain.add(indexOfExistingData, singleTag);
            } else {
                singleCategoryTagsChain.add(singleTag);
            }
            tagDataMap.put(singleTag, taggedData);
            taggedValues.clear();
        }        
    }
    
    /**
     * 
     * @param chain 
     */
//    public void addCategoryDataChain(CategoryDataChain chain) {
//        if ( !chain.getCategory().equals(getCategory()) ) {
//            throw new IllegalArgumentException("Incompatible category \""+chain.getCategory()+"\" cannot be added to category data chain for \""+category+"\"");
//        }
//        synchronized(taggedValues) {
//            for (SingleCategoryTag singleTag : chain.singleCategoryTagsChain) {
//                addTaggedDataForSingleCategoryTag(singleTag, chain.tagDataMap.get(singleTag));
//            }
//        }
//    }
    
    /**
     * Get the category for this CategoryDataChain.
     * 
     * @return The category for this CategoryDataChain.
     */
    public String getCategory() {
        return category;
    }
    
    /**
     * Get a copy of the ConfiguraiontParameterTaggedValue map.
     * This is built based on the chain of SingleCategoryTags.
     * 
     * @return A copy of the Map containing the ConfigurationParameterTaggedValues
     */
    public Map<String,ConfigurationParameterTaggedValue> getConfigurationParameterTaggedValues() {
        synchronized(taggedValues) {
            if ( taggedValues.isEmpty() ) {
                for ( SingleCategoryTag singleTag : singleCategoryTagsChain ) {
                    taggedValues.putAll(tagDataMap.get(singleTag));
                }
            }            
            return new HashMap<>(taggedValues);
        }
    }
    
    public List<SingleCategoryTag> getSingleCategoryTagList() {
        return new ArrayList(singleCategoryTagsChain);
    }
    
}
