package org.lsst.ccs.config;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;

/**
 * Builder for ComponentConfigurationHandler objects. It extracts Configuration and
 * ConfigurationParameter annotations from an object and builds a ParameterSetter out of it.
 * 
 * @author The LSST CCS Team
 */
public class ConfigurationHandlerBuilder {

    private ConfigurationHandlerBuilder() {
    }

    /**
     * Caches the dictionary of @ConfigurationParameterChanger by class
     */
    private static final Map<String, Map<String, Method>> CONFIG_CHANGER_MAP = new HashMap<>();

    /**
     * Caches the set of @ConfigurationParameter fields by class
     */
    private static final Map<String, Map<String, Field>> PARAMETER_MAP = new HashMap<>();
    
    /**
     * Caches the set of @ConfigurationParameter fields by class
     */
    private static final Map<Object, Map<String, ConfigurationParameterDescription>> PARAMETER_DESCRIPTOR_MAP = new HashMap<>();
    private static final Map<Object, Map<String, String>> PARAMETER_NAME_SWITCH_MAP = new HashMap<>();

    /**
     * Builds a ComponentConfigurationHandler object by introspecting the specified
     * object.
     * @param obj
     * @return the ConfigurationHandler for the specified object or null if this
     * object does not contain any configuration parameters.
     */
    static ConfigurationHandler buildParameterSetterFromObject(String objName, Object obj) {

        Class cls = obj.getClass();

        Map<String, Field> fieldsMap = buildParameterFieldsMap(cls, obj);
        
        if (fieldsMap.isEmpty()) return null;
        
        Map<String, Method> configChangerMap = buildConfigChangerMap(cls, obj);
        
        ConfigurationHandler res = new ConfigurationHandler(obj, objName);

        for (Map.Entry<String, Field> entry : fieldsMap.entrySet()) {
            ConfigurationParameterDescription cpd = PARAMETER_DESCRIPTOR_MAP.get(obj) != null ? PARAMETER_DESCRIPTOR_MAP.get(obj).get(entry.getKey()) : null;
            res.addParameter(entry.getKey(), new ConfigurationParameterHandler(objName, configChangerMap.get(entry.getKey()), entry.getValue(), obj, cpd));
        }
        return res;
    }
    
    /**
     * Introspects the class for @ConfigChanger methods.
     *
     * @param klass
     * @return
     */
    public static Map<String, Method> buildConfigChangerMap(Class klass, Object obj) {
        String klassName = klass.getName();

        // Introspects the class for @ConfigChanger annotated methods and 
        // stores them in METHOD_MAP
        Map<String, Method> configMethods = CONFIG_CHANGER_MAP.get(klassName);
        if (configMethods == null) {
            configMethods = new HashMap<>();
            CONFIG_CHANGER_MAP.put(klassName, configMethods);

            ///////// configMethod initializer!
            //This will introspect only public methods as the getMethods() method
            //will return any class or inherited methods that are public
            for (Method method : klass.getMethods()) {
                String methodName = method.getName();
                ConfigurationParameterChanger parameterChanger = method.getAnnotation(ConfigurationParameterChanger.class);
                if (parameterChanger != null) {
                    String propertyName = parameterChanger.propertyName();
                    if ("".equals(propertyName)) {
                        //if method is "setter" then gets name of property
                        if (methodName.startsWith("set")) {
                            char firstLetter = methodName.charAt(3);
                            propertyName = Character.toLowerCase(firstLetter) + methodName.substring(4);
                        }
                    }
                    configMethods.put(propertyName, method);
                }
            }
            
            //Check that there are no non-public methods that are annotated with
            //The ConfigurationParameterChanger annotation.
            Class clazz = klass;
            while (clazz != null) {
                for (Method method : clazz.getDeclaredMethods()) {
                    int modifiers = method.getModifiers();
                    ConfigurationParameterChanger parameterChanger = method.getAnnotation(ConfigurationParameterChanger.class);
                    if (parameterChanger != null && ! Modifier.isPublic(modifiers) ) {
                        throw new RuntimeException("Method "+clazz.getName()+"::"+method.getName()+" is annotated as a "
                                + "ConfigurationParameterChanger but it is not public. Please update your code: this method must be public.");           
                    }
                }
                clazz = clazz.getSuperclass();
            }
        }
        

        if ( PARAMETER_NAME_SWITCH_MAP.get(obj) != null ) {
            //Change the method's affiliation if there is a ConfigurationParameterDescription
            //for this object and this parameter name            
            Map<String, Method> objConfigMethods = new HashMap(configMethods);
            for ( Entry<String,String> e : PARAMETER_NAME_SWITCH_MAP.get(obj).entrySet() ) {
                objConfigMethods.put(e.getValue(), objConfigMethods.remove(e.getKey()));
            }
            return objConfigMethods;
        } else {
            return configMethods;
        }
    }
    
    /**
     * Introspect the class for @ConfigurationParameter fields.
     *
     * @param klass
     * @param obj The particular instance of the provided Class
     * @return a map of @ConfigurationParameter fields.
     * @throw RuntimeException if two parameters end up having the same name.
     */
    public static Map<String, Field> buildParameterFieldsMap(Class klass, Object obj) {
        String className = klass.getName();
        Map<String, Field> res = PARAMETER_MAP.get(className);
        if (res == null) {
            res = new HashMap<>();
            PARAMETER_MAP.put(className, res);
            // Looks for inherited fields as well
            Class toIntrospect = klass;
            do {
                Field[] fields = toIntrospect.getDeclaredFields();
                for (Field f : fields) {
                    ConfigurationParameter a = f.getAnnotation(ConfigurationParameter.class);
                    if (a != null) {
                        String parmName = (a.name().isEmpty()) ? f.getName() : a.name();
                        if(res.containsKey(parmName)) {
                            throw new RuntimeException("Class " + klass.getSimpleName() + " is erroneously annotated as it contains two parameters with same name : " + parmName);
                        }
                        res.put(parmName, f);
                    }
                }
                toIntrospect = toIntrospect.getSuperclass();
            } while (toIntrospect != null);
        }
        
        
        if ( obj instanceof HasConfigurationParameterDescription ) {
            HashMap new_res = new HashMap<>(res);
            HasConfigurationParameterDescription hasParDescriptor = (HasConfigurationParameterDescription)obj;
            Map<String, ConfigurationParameterDescription> parDescriptionMap = new HashMap<>();
            Map<String, String> parSwitchMap = new HashMap<>();
            for (String parName : res.keySet()) {
                ConfigurationParameterDescription parDescription = hasParDescriptor.getConfigurationParameterDescription(parName);
                if ( parDescription != null ) {
                    String tmpParName = parName;
                    if ( parDescription.getName() != null ) {
                        tmpParName = parDescription.getName();
                        new_res.put(tmpParName, new_res.remove(parName));                        
                    }
                    parDescriptionMap.put(tmpParName, parDescription);
                    if ( !tmpParName.equals(parName) ) {
                        parSwitchMap.put(parName, tmpParName);
                    }                                
                }
            }
            if (! parDescriptionMap.isEmpty()) {
                PARAMETER_DESCRIPTOR_MAP.put(obj, parDescriptionMap);
                if ( ! parSwitchMap.isEmpty() ) {
                    PARAMETER_NAME_SWITCH_MAP.put(obj, parSwitchMap);
                }
            }
            return new_res;
        } else {
            return res;
        }
    }

}
