package org.lsst.ccs.config.remote;

import java.util.Set;

import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.config.ConfigurationDAO;
import org.lsst.ccs.config.ConfigurationDescription;
import org.lsst.ccs.config.ConfigurationServiceException;
import org.lsst.ccs.config.ConfigurationView;
import org.lsst.ccs.config.LocalConfigurationDAO;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.utilities.logging.Logger;


public class DualConfigurationDAO implements ConfigurationDAO {

    Logger log = Logger.getLogger("org.lsst.ccs.config");
    
    public DualConfigurationDAO(AgentMessagingLayer aml, String descriptionName) {
        String agentName = aml.getAgentInfo().getName();

        local = new LocalConfigurationDAO(agentName, descriptionName);
        remote = new CCSBusesConfigurationDAO(aml);
    }

    LocalConfigurationDAO local;
    CCSBusesConfigurationDAO remote;

    @Override
    public ConfigurationView loadConfiguration(String agentName, ConfigurationDescription configDescription)
            throws ConfigurationServiceException {

        ConfigurationView localView = local.loadConfiguration(agentName, configDescription);
        try {
            ConfigurationView remoteView = remote.loadConfiguration(agentName, configDescription);
            ConfigurationView diff1 = localView.diff(remoteView);
            ConfigurationView diff2 = remoteView.diff(localView);
            if (!diff1.isEmpty() || !diff2.isEmpty()) {
                log.error("loadConfiguration local/remote mismatch");
                log.error("local  " + localView.toString());
                log.error("remote " + remoteView.toString());
            }
        } catch (Exception e) {
            log.error("Exception during remote loadConfiguration", e);
        }
        return localView;
    }

    @Override
    public ConfigurationDescription registerConfiguration(String agentName, ConfigurationInfo configInfo)
            throws ConfigurationServiceException {
        ConfigurationDescription localCD = local.registerConfiguration(agentName, configInfo);
        try {
            ConfigurationDescription remoteCD = remote.registerConfiguration(agentName, configInfo);
            if (!localCD.equals(remoteCD)) {
                log.error("registerConfiguration local/remote mismatch");
                log.error("local  " + localCD.toString());
                log.error("remote " + remoteCD.toString());
            }
        } catch (Exception e) {
            log.error("Exception during remote registerConfiguration", e);
        }

        return localCD;
    }

    @Override
    public ConfigurationDescription saveChangesForCategoriesAs(String agentName, ConfigurationDescription configDesc,
            ConfigurationInfo configInfo) throws ConfigurationServiceException {
        ConfigurationDescription localCD = local.saveChangesForCategoriesAs(agentName, configDesc, configInfo);
        try {
            ConfigurationDescription remoteCD = remote.saveChangesForCategoriesAs(agentName, configDesc, configInfo);
            if (!localCD.equals(remoteCD)) {
                log.error("saveChangesForCategoriesAs local/remote mismatch");
                log.error("local  " + localCD.toString());
                log.error("remote " + remoteCD.toString());
            }
        } catch (Exception e) {
            log.error("Exception during remote saveChangesForCategoriesAs", e);
        }

        return localCD;
    }

    @Override
    public ConfigurationView loadGlobalConfiguration(String agentName, String name, int version) {
        ConfigurationView localView = local.loadGlobalConfiguration(agentName, name, version);
        try {
            ConfigurationView remoteView = remote.loadGlobalConfiguration(agentName, name, version);
            ConfigurationView diff1 = localView.diff(remoteView);
            ConfigurationView diff2 = remoteView.diff(localView);
            if (!diff1.isEmpty() || !diff2.isEmpty()) {
                log.error("loadGlobalConfiguration local/remote mismatch");
                log.error("local  " + localView.toString());
                log.error("remote " + remoteView.toString());
            }
        } catch (Exception e) {
            log.error("Exception during remote loadGlobalConfiguration", e);
        }
        return localView;
    }

    @Override
    public Set<String> findAvailableConfigurationsForCategory(String agentName, String category) {
        Set<String> localConfs = local.findAvailableConfigurationsForCategory(agentName, category);
        try {
            Set<String> remoteConfs = remote.findAvailableConfigurationsForCategory(agentName, category);
            if (!localConfs.equals(remoteConfs)) {
                log.error("findAvailableConfigurationsForCategory local/remote mismatch");
                log.error("local  " + localConfs.toString());
                log.error("remote " + remoteConfs.toString());
            }
        } catch (Exception e) {
            log.error("Exception during remote findAvailableConfigurationsForCategory", e);
        }
        return localConfs;
    }

    @Override
    public boolean isAvailable() {
        return true;
    }

}
