package org.lsst.ccs.config;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.lsst.ccs.utilities.conv.InputConversionEngine;

import org.lsst.ccs.utilities.conv.TypeUtils;
import org.lsst.ccs.utilities.structs.ParameterPath;
import org.lsst.ccs.utilities.structs.ViewValue;

/**
 * Encapsulates the logic of setting configurable parameters for a given
 * component. It provides the validation step and the setting step, as well as
 * intermediate steps
 *
 * @author The LSST CCS Team
 */
public class ConfigurationHandler {
    
    // The targeted object
    final Object targetComponent;
    
    private final String name;
    
    // The set of submitted changes for this component
    private final Map<String, ViewValue> submittedChanges = new HashMap<>();
    
    private final Map<String, ConfigurationParameterHandler> parameters = new HashMap<>();
    
    private boolean hasRuntimeParameters = false;
    
    ConfigurationHandler(Object target, String name) {
        this.targetComponent = target;
        this.name = name;
    }
    
    void addParameter(String name, ConfigurationParameterHandler handler) {
        //Check if this ConfigurationHandler has runtime parameters.
        //This flag is used to decide if a command set is to be added or not.
        if ( ! hasRuntimeParameters ) {
            if ( ! handler.isBuild() && ! handler.isFinal() && ! handler.isReadOnly() ) {
                hasRuntimeParameters = true;
            }
        }
        parameters.put(name, handler); 
    }
    
    Object submitChange(String parameterName, Object value) {
        Object obj = InputConversionEngine
                .convertArgToType(TypeUtils.stringify(value), getConfigurationParameterHandler(parameterName).getType());
        // Normalizing the String representation of the input value
        String strValue = TypeUtils.stringify(obj);
        
        if ( value instanceof String ) {
            if ( !strValue.equals(value) ) {
                value = strValue;
            }
        }
        
        getConfigurationParameterHandler(parameterName).acceptValue(obj);        
        submittedChanges.put(parameterName, new ViewValue(strValue, obj));
        return value;
    }
    
    void dropSubmittedChanges() {
        submittedChanges.clear();
    }
    
    /**
     * Validation step
     *
     * @param isSafe
     *            true if the final property of a parameter has to be checked,
     *            false otherwise.
     */
    void invokeValidateBulkChange(boolean isSafe) {
        
        if (submittedChanges.isEmpty()) return;
        
        checkAgainstConstraints(submittedChanges, isSafe);
        if (targetComponent instanceof ConfigurationBulkChangeHandler) {
            Map<String, ViewValue> currentView = convert(getCurrentValues(Collections.EMPTY_SET));

            currentView.putAll(submittedChanges);
            Map<String, Object> toValidate = new HashMap<>();
            for(Map.Entry<String, ViewValue> entry : currentView.entrySet()) {
                toValidate.put(entry.getKey(), entry.getValue().getValue());
            }
            ((ConfigurationBulkChangeHandler) targetComponent).validateBulkChange(toValidate);
        }
    }

    /**
     * Global parameter setting algorithm. It first invokes the bulk setting on
     * the component, then invokes single setting on the remaining components.
     *
     * @throws Exception which might be a user exception.
     */
    void invokeSetParameters(Map<String, String> before, boolean isBuild) {
        
        if (!isBuild && targetComponent instanceof ConfigurationBulkChangeHandler && !submittedChanges.isEmpty()) {
            ((ConfigurationBulkChangeHandler) targetComponent).setParameterBulk(submittedChanges
                    .entrySet().stream().collect(Collectors.toMap(
                            entry -> entry.getKey(),
                            entry -> entry.getValue().getValue())));
            // Retrieving which of the submitted changes have been processed by the bulk change method
            for (Map.Entry<String, ViewValue> entry : submittedChanges.entrySet()) {
                String parmName = entry.getKey();
                String submittedVal = entry.getValue().getView();
                String actualVal = getConfigurationParameterHandler(parmName).getValue();
                if (!submittedVal.equals(actualVal)) {
                    if (actualVal.equals(before.get(parmName))) {
                        getConfigurationParameterHandler(parmName)
                                .invokeSetParameter(entry.getValue().getValue());
                    } else {
                        throw new RuntimeException(name+"//"+parmName + " value : " + actualVal + ", expected : " + before.get(parmName));
                    }
                }
            }
        } else {
            for (String parmName : submittedChanges.keySet()) {
                getConfigurationParameterHandler(parmName)
                        .invokeSetParameter(submittedChanges.get(parmName).getValue());
            }
        }
    }
    
    public boolean isBulkChanger() {
        return targetComponent instanceof ConfigurationBulkChangeHandler;
    }
    
    Collection<ConfigurationParameterHandler> getConfigurationParameterHandlers() {
        return parameters.values();
    }
    
    /**
     * Check against potential constraints defined in the annotation.
     *
     * @param parms
     * @param isSafe
     */
    private void checkAgainstConstraints(Map<String, ViewValue> parms,
            boolean isSafe) {
        parms.entrySet()
                .stream()
                .forEach(
                        entry -> {
                            String parmName = entry.getKey();
                            if (!isSafe && getConfigurationParameterHandler(parmName).isFinal()) {
                                throw new IllegalStateException("Final parameter "
                                        + parmName
                                        + " not modifiable at runtime");
                            }
                            Object val = entry.getValue().getValue();
                            getConfigurationParameterHandler(parmName).checkAgainstConstraints(val);
                        });
    }
    
    /**
     * Conversion from String to object using the InputConversionEngine utility.
     *
     * @param map
     * @return
     */
    private Map<String, ViewValue> convert(Map<ParameterPath, String> map) {
        return map.entrySet().stream().collect(
                Collectors.toMap(
                        entry -> entry.getKey().getParameterName(),
                        entry -> {
                            return getConfigurationParameterHandler(entry.getKey().getParameterName()).convert(entry.getValue());
                        }));
    }
    
    private static <A, B, C> Function<A, C> compose(Function<A, B> f1,
            Function<B, C> f2) {
        return f1.andThen(f2);
    }
    
    boolean hasSubmittedChanges() {
        return !submittedChanges.isEmpty();
    }
    
    Map<String, String> getSubmittedChanges() {
        return Collections.unmodifiableMap(submittedChanges.entrySet()
                .stream().collect(Collectors.toMap(
                        entry -> entry.getKey(),
                        compose(Entry<String, ViewValue>::getValue,
                                        ViewValue::getView)
                        )
                ));
    }
    
    Map<ParameterPath, String> getCurrentValues(Set<String> categorySet) {
        Map<ParameterPath, String> res = new HashMap<>();
        for (ConfigurationParameterHandler cph : parameters.values()) {
            if (categorySet.isEmpty() || categorySet.contains(cph.getCategory())) {
                //This is where the value is read from the corresponding object.
                String strVal = cph.getValue();
                res.put(new ParameterPath(cph.getComponentName(), cph.getParameterName()), strVal);
            } 
        }
        return res;
    }
    
    boolean isParameterConfigurable(String parameterName) {
        return parameters.containsKey(parameterName);
    }
    
    boolean isParameterReadOnly(String parameterName) {
        ConfigurationParameterHandler cph = parameters.get(parameterName);
        return cph == null ? false : cph.isReadOnly();
    }

    boolean isBuildParameter(String parameterName) {
        ConfigurationParameterHandler cph = parameters.get(parameterName);
        return cph == null ? false : cph.isBuild();
    }

    boolean isFinalParameter(String parameterName) {
        ConfigurationParameterHandler cph = parameters.get(parameterName);
        return cph == null ? false : cph.isFinal();
    }

    boolean isOptionalParameter(String parameterName) {
        ConfigurationParameterHandler cph = parameters.get(parameterName);
        return cph == null ? false : cph.isOptional();
    }

    public ConfigurationParameterHandler getConfigurationParameterHandler(String parmName) {
        ConfigurationParameterHandler res = parameters.get(parmName);
        if (res == null) {
            throw new IllegalArgumentException("no such config property : " + parmName);
        }
        return res;
    }

    public boolean hasRuntimeParameters() {
        return hasRuntimeParameters;
    }
    
    /**
     * Removes the submitted changes that are unnecessary.
     */
    void trimSubmittedChanges() {
        for (Iterator<Map.Entry<String, ViewValue>> it = submittedChanges.entrySet().iterator(); it.hasNext(); ) {
            Map.Entry<String, ViewValue> entry = it.next();
            if (parameters.get(entry.getKey()).getValue().equals(entry.getValue().getView())) {
                it.remove();
            }
        }
    }
    
}
