package org.lsst.ccs.config;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;

/**
 * Builder for ComponentConfigurationHandler objects. It extracts Configuration and
 * ConfigurationParameter annotations from an object and builds a ParameterSetter out of it.
 * 
 * @author The LSST CCS Team
 */
class ConfigurationHandlerBuilder {

    private ConfigurationHandlerBuilder() {
    }

    /**
     * Caches the dictionary of @ConfigurationParameterChanger by class
     */
    private static final Map<String, Map<String, Method>> CONFIG_CHANGER_MAP = new HashMap<>();

    /**
     * Caches the set of @ConfigurationParameter fields by class
     */
    private static final Map<String, Map<String, Field>> PARAMETER_MAP = new HashMap<>();
    
    /**
     * Builds a ComponentConfigurationHandler object by introspecting the specified
     * object.
     * @param obj
     * @return the ConfigurationHandler for the specified object or null if this
     * object does not contain any configuration parameters.
     */
    static ConfigurationHandler buildParameterSetterFromObject(String objName, Object obj) {

        Class cls = obj.getClass();

        Map<String, Field> fieldsMap = buildParameterFieldsMap(cls);
        
        if (fieldsMap.isEmpty()) return null;
        
        Map<String, Method> configChangerMap = buildConfigChangerMap(cls);
        
        ConfigurationHandler res = new ConfigurationHandler(obj, objName);

        for (Map.Entry<String, Field> entry : fieldsMap.entrySet()) {
            res.addParameter(entry.getKey(), new ConfigurationParameterHandler(objName, configChangerMap.get(entry.getKey()), entry.getValue(), obj));
        }
        return res;
    }
    
    /**
     * Introspects the class for @ConfigChanger methods.
     *
     * @param klass
     * @return
     */
    private static Map<String, Method> buildConfigChangerMap(Class klass) {
        String klassName = klass.getName();

        // Introspects the class for @ConfigChanger annotated methods and 
        // stores them in METHOD_MAP
        Map<String, Method> configMethods = CONFIG_CHANGER_MAP.get(klassName);
        if (configMethods == null) {
            configMethods = new HashMap<>();
            CONFIG_CHANGER_MAP.put(klassName, configMethods);

            ///////// configMethod initializer!
            //This will introspect only public methods as the getMethods() method
            //will return any class or inherited methods that are public
            for (Method method : klass.getMethods()) {
                String methodName = method.getName();
                ConfigurationParameterChanger parameterChanger = method.getAnnotation(ConfigurationParameterChanger.class);
                if (parameterChanger != null) {
                    String propertyName = parameterChanger.propertyName();
                    if (!"".equals(propertyName)) {
                        configMethods.put(propertyName, method);
                    } else {
                        //if method is "setter" then gets name of property
                        if (methodName.startsWith("set")) {
                            char firstLetter = methodName.charAt(3);
                            propertyName = Character.toLowerCase(firstLetter) + methodName.substring(4);
                            configMethods.put(propertyName, method);
                        }
                    }
                }
            }
            
            //Check that there are no non-public methods that are annotated with
            //The ConfigurationParameterChanger annotation.
            Class clazz = klass;
            while (clazz != null) {
                for (Method method : clazz.getDeclaredMethods()) {
                    int modifiers = method.getModifiers();
                    ConfigurationParameterChanger parameterChanger = method.getAnnotation(ConfigurationParameterChanger.class);
                    if (parameterChanger != null && ! Modifier.isPublic(modifiers) ) {
                        throw new RuntimeException("Method "+clazz.getName()+"::"+method.getName()+" is annotated as a "
                                + "ConfigurationParameterChanger but it is not public. Please update your code: this method must be public.");           
                    }
                }
                clazz = clazz.getSuperclass();
            }
        }
        
        return configMethods;
    }
    
    /**
     * Introspect the class for @ConfigurationParameter fields.
     *
     * @param klass
     * @return a map of @ConfigurationParameter fields.
     * @throw RuntimeException if two parameters end up having the same name.
     */
    private static Map<String, Field> buildParameterFieldsMap(Class klass) {
        String className = klass.getName();
        Map<String, Field> res = PARAMETER_MAP.get(className);
        if (res == null) {
            res = new HashMap<>();
            PARAMETER_MAP.put(className, res);
            // Looks for inherited fields as well
            Class toIntrospect = klass;
            do {
                Field[] fields = toIntrospect.getDeclaredFields();
                for (Field f : fields) {
                    ConfigurationParameter a = f.getAnnotation(ConfigurationParameter.class);
                    if (a != null) {
                        String parmName = (a.name().isEmpty()) ? f.getName() : a.name();
                        if(res.containsKey(parmName)) {
                            throw new RuntimeException("Class " + klass.getSimpleName() + " is erroneously annotated as it contains two parameters with same name : " + parmName);
                        }
                        res.put(parmName, f);
                    }
                }
                toIntrospect = toIntrospect.getSuperclass();
            } while (toIntrospect != null);
        }
        return res;
    }

}
