package org.lsst.ccs.config;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.stream.Collectors;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.utilities.conv.TypeUtils;
import org.lsst.ccs.utilities.logging.Logger; 
import org.lsst.ccs.utilities.structs.ParameterPath;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A set of several ConfigurationHandlers.
 * @author LSST CCS Team
 */
public class ConfigurationHandlerSet {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.config");
    private final  List<DataProviderInfo> list = new ArrayList();
    
    
    // The following fields are created and populated when the Agent is still
    // single threaded, so they are thread safe as long as their content is thread safe.    
    /** 
     * Configurable parameters ordered by the component they belong to. 
     * A LinkedHashMap is used to preserve insertion order, which is the order with which the 
     * configuration handlers have to be traveled when applying configurations / processing bulk changes
     */
    private final Map<String, ConfigurationHandler> componentParameterHandlers = new LinkedHashMap<>();

    /** 
     * Configurable parameters ordered by the category they belong to. 
     */
    private final Map<String, CategoryHandler> parametersByCategories = new HashMap<>();
    
    private ConfigurationView initialView;
    
    private final Set<String> categories = new HashSet<>();
    private final Map<String,String> componentToPath = new HashMap<>();
    
    private final boolean useFullPaths;
    private final boolean checkConfigView;
    private final boolean failOnViewProblems;
    private boolean failForBuildParameters = true;
    
    public ConfigurationHandlerSet(boolean usefullPaths, boolean checkConfigView, boolean failOnViewProblems) {
        this.useFullPaths = usefullPaths;
        this.checkConfigView = checkConfigView;
        this.failOnViewProblems = failOnViewProblems;
    }
    
    public ConfigurationHandlerSet() {
        this(true, true, false);
    }
    
    // -- Construction phase protected methods 
    //This method is curretnly only invoked by tests.
    public ConfigurationHandler addConfigurationHandlerForObject(String componentName, String componentPath, Object component) { 
        return addConfigurationHandlerForObject(componentName, componentPath, component, false); 
    }

    //This method is only invoked by the ConfigurationService during the HasLifecycle steps
    //when the agent is still single threaded.
    public ConfigurationHandler addConfigurationHandlerForObject(String componentName, String componentPath, Object component, boolean isNewImplementation) { 
        ConfigurationHandler handler = ConfigurationHandlerBuilder.buildParameterSetterFromObject(componentName, component);
        componentToPath.put(componentName, componentPath);
        
        if (handler == null) return null;
        
        componentParameterHandlers.put(componentName, handler);
        for (ConfigurationParameterHandler cph : handler.getConfigurationParameterHandlers()) {
            //Assign "build" category to all build parameters
            if (isNewImplementation && cph.isBuild()) {
                if ( cph.getCategory() != null && ! cph.getCategory().isEmpty() ) {
                    log.log(Level.WARNING, "Changing assigned category {0} for isBuild parameter {1} to \"build\"", new Object[] {cph.getCategory(),cph.getParameterPath()});
                }
                cph.setCategory("build");
            }
            String category = cph.getCategory();
            CategoryHandler set = parametersByCategories.get(category);
            if (set == null) {
                categories.add(category);
                set = new CategoryHandler(category);
                parametersByCategories.put(category, set);
            }
            set.addParameterHandler(cph);
        }
        return handler;
    }
    
    public ConfigurationInfo initialize(String descriptionName) {
        return initialize(descriptionName, new ConfigurationView(ConfigurationDescription.safeConfiguration(categories)));
    }
    
    public ConfigurationInfo initialize(String descriptionName, ConfigurationView safeView) {
        return initialize(descriptionName, safeView, false, false);        
    }
    public ConfigurationInfo initialize(String descriptionName, ConfigurationView safeView, boolean isBuild, boolean isSafe) {
        failForBuildParameters = false;
        initialView = new ConfigurationView(ConfigurationDescription.safeConfiguration(categories));
        initialView.putAll(getLiveConfigurationView());        
        //If we are in the initialization step we need to check if the configuration parameter fields
        //are declared as volatile and if the collections have maxLength defined.
        if ( isSafe && ! isBuild ) {
            List<ParameterPath> missingMaxLength = new ArrayList<>();
            List<ParameterPath> missingVolatile = new ArrayList<>();
            for ( ConfigurationHandler handler : componentParameterHandlers.values() ) {
                for ( ConfigurationParameterHandler configHandler : handler.getConfigurationParameterHandlers() ) {
                    if ( !isVolatile(configHandler) ) {
                        missingVolatile.add(configHandler.getParameterPath());                                                    
                    }
                    if ( configHandler.hasLength() && configHandler.getMaxLength() == 0 ) {
                        missingMaxLength.add(configHandler.getParameterPath());                                                    
                    }
                }
            }
            if ( missingMaxLength.size() > 0 ) {
                log.log(Level.WARNING,"Collections or Arrays annotated as Configuration Parameters must provide the \"maxLength\" argument.\n"
                        + "Please update your code to specify the maximum allowed length for the following parameters: \n{0}",missingMaxLength);
            }
            if ( missingVolatile.size() > 0 ) {
                log.log(Level.WARNING,"Fields annotated as Configuration Parameters must be declared as \"volatile\".\n"
                        + "Please update your code for the following parameters:\n{0}",missingVolatile);
            }
        }
        
        
        //If it's not the build step, then we need to remove all the build
        //parameters from the initial view.
        if ( !isBuild ) {
            for (ParameterPath p : safeView.getAsParameterPathMap().keySet() ) {
                ConfigurationHandler ch = componentParameterHandlers.get(p.getComponentName());
                if ( ch.isBuildParameter(p.getParameterName())) {
                    safeView.removeValueForPath(p);
                }                
            }
        }
        
        try {
            safeView = loadCategories(safeView, isBuild, isSafe);                                
            //Updating the initialView with all the parameters set by the safeView.
            initialView.putAll(safeView);            
            // Checking initial value of configuration parameters during the init phase
            if (! isBuild) {
                for (Map.Entry<ParameterPath, String> pp : initialView.getAsParameterPathMap().entrySet()) {
                    if (pp.getValue().equals(TypeUtils.NULL_STR)) {
                        log.warn("parameter " + pp.getKey() + " has not been assigned a safe ");
                    }
                }
            }
        } catch (Exception ex) {
            throw new RuntimeException("could not load initial configuration.", ex);
        }
        
        
        // First configurationInfo creation
        ConfigurationInfo.Builder ciBuilder = new ConfigurationInfo.Builder()
                .setDescription(descriptionName).updateConfigurationDescription(safeView.getConfigurationDescription().getDescriptionName());
        for (CategoryHandler ch : parametersByCategories.values()) {
            String category = ch.getCategory();
            CategoryTag categoryTag = safeView.getConfigurationDescription().getCategoryTag(category);            
            String tag0 = categoryTag != null ? categoryTag.getTags().iterator().next() : null;
            Integer version0 = categoryTag != null ? categoryTag.getTagVersion(tag0) : null;
                    
            ciBuilder.updateCategoryInformation(category, tag0, version0, false);
//            ciBuilder.updateCategoryInformation(ch.getCategory(), 
//                    safeView.getConfigurationDescription().getCategoryTags().get(ch.getCategory()),
//                    safeView.getConfigurationDescription().getCategoryVersions().get(ch.getCategory()),
//                    false);
            for (ConfigurationParameterHandler cph : ch.getParameters()) {
                ParameterPath pp = new ParameterPath(cph.getComponentName(), cph.getParameterName());
                ciBuilder.addParameter(pp, cph.getType(), cph.getCategory(), cph.getDescription(), cph.isFinal(), cph.isReadOnly(), cph.isBuild());
                
                DataProviderInfo dataProviderInfo = new DataProviderInfo(componentToPath.get(pp.getComponentName()), DataProviderInfo.Type.CONFIGURATION, pp.getParameterName());
                if ( ! useFullPaths ) {
                    //Add this property only if not using full path. If we are using full path
                    //The published path is identical to the DataProviderInfo path
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.PUBLISHED_PATH, pp.getComponentName()+"/"+pp.getParameterName());
                }
                if ( !cph.getUnits().isEmpty() ) {
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.UNITS, cph.getUnits());
                }
                dataProviderInfo.addAttribute(DataProviderInfo.Attribute.TYPE, cph.getType().getTypeName());
                if ( !cph.getDescription().isEmpty() ) {
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, cph.getDescription());
                }
                dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_TYPE, cph.getParameterType().name());
                dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_CATEGORY, cph.getCategory());
                String range = cph.getRange();
                if ( range != null && !range.isEmpty() ) {
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_RANGE, range);
                }
                //Uncomment this code for https://jira.slac.stanford.edu/browse/LSSTCCS-2141
//                if ( cph.hasLength() && cph.getMaxLength() != 0 ) {
//                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_MAX_LENGTH, String.valueOf(cph.getMaxLength()));                    
//                }
                list.add(dataProviderInfo);
                
                ciBuilder.updateParameter(pp, initialView.getPathValue(pp), initialView.getPathValue(pp), false);
            }
        }
        Collections.sort(list,comparator);
        failForBuildParameters = true;
        
        return ciBuilder.setConfigurationState(ConfigurationState.UNCONFIGURED)
                .setCCSTimeStamp(CCSTimeStamp.currentTime()).build();
    }
    
    private static boolean isVolatile(ConfigurationParameterHandler parameterHandler) {
        return Modifier.isVolatile(parameterHandler.getField().getModifiers()) || Modifier.isFinal(parameterHandler.getField().getModifiers());        
    }

    private static Comparator<DataProviderInfo> comparator = new Comparator<DataProviderInfo>() {
        @Override
        public int compare(DataProviderInfo o1, DataProviderInfo o2) {
            return o1.getFullPath().compareTo(o2.getFullPath());
        }
        
    };

    public List<DataProviderInfo> getDataPoviderInfoList() {
        return list;
    }
    
    // -- End construction phase protected methods.
    
    public ConfigurationHandler getConfigurationHandlerForGroovy(String componentName) {
//        if (!useFullPaths) {
//            String[] split = componentName.split("/");
//            if (split.length > 1) {
//                componentName = split[split.length - 1];
//            }
//        }
        try {
            return getParameterSet(componentName);
        } catch (Exception e) {
//            e.printStackTrace();
            return null;
        }
    }
    
    public ConfigurationHandler getParameterSet(String componentName) {
        ConfigurationHandler res = componentParameterHandlers.get(componentName);
        if (res == null) {
            throw new IllegalArgumentException("no such component : " + componentName);
        }
        return res;
    }

    public CategoryHandler getCategoryHandler(String category) {
        CategoryHandler res = parametersByCategories.get(category);
        if (res == null) {
            log.info("no such category : " + category);
            throw new RuntimeException("Category "+category+" does not exist");
        }
        return res;
    }
    
    public boolean isBuildParameter(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isBuildParameter(parameterName);
    }

    public boolean isOptionalParameter(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isOptionalParameter(parameterName);
    }

    public boolean isFinalParameter(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isFinalParameter(parameterName);
    }

    public boolean isParameterConfigurable(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isParameterConfigurable(parameterName);
    }
    
    public boolean isParameterReadOnly(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isParameterReadOnly(parameterName);
    }

    public Set<String> getCategorySet() {
        return Collections.unmodifiableSet(parametersByCategories.keySet());
    }
    
    public ConfigurationView loadCategories(ConfigurationView cv) {
        return loadCategories(cv,false, false);
    }
    private ConfigurationView loadCategories(ConfigurationView cv, boolean isBuild, boolean isSafe) {                
        ConfigurationDescription configDesc = cv.getConfigurationDescription();
        
        StringBuilder unexpectedParms = new StringBuilder("");
        for (Map.Entry<ParameterPath, String> entry : cv.getAsParameterPathMap().entrySet()) {
            ParameterPath parm = entry.getKey();
            //At build time handle only build configuration parameters
            String componentName = parm.getComponentName();
            String parameterName = parm.getParameterName();
            if ( isBuild ) {
                if ( ! isBuildParameter(componentName, parameterName) /*&& ! isFinalParameter(componentName, parameterName)*/) {
                    continue;
                }
            } else {
                if (isBuildParameter(componentName, parameterName)) {
                    log.log(Level.WARNING,"Attempting to set build ConfigurationParameter {0} past the HasLifecycle::build phase.", new Object[]{parm});
                    continue;
                }
            }
            if ( isParameterReadOnly(componentName, parameterName) ) {
                continue;
            }            
            if (isParameterConfigurable(componentName, parameterName) && ! isParameterReadOnly(componentName, parameterName)) {
                Object newValue = submitChange(componentName, parameterName, entry.getValue());
                if ( newValue instanceof String && ! ((String)newValue).equals(entry.getValue())) {
                    cv.putParameterValue(componentName, parameterName, (String)newValue);
                }
            } else {
                if ( cv.containsPath(parm) ) {
                    unexpectedParms.append(parm).append(" ");
                }
            }
        }
        // Submitting the initial value for other components
        // REVIEW: should we do this ?
        if ( configDesc != null ) {
            Map<String, String> missingParmsDigest = new HashMap<>();
            for (String category : configDesc.getCategoriesSet()) {
                StringBuilder missingParms = new StringBuilder("");
                CategoryHandler ch = getCategoryHandler(category);
                for (ConfigurationParameterHandler cph : ch.getParameters()) {
                    String componentName = cph.getComponentName();
                    String parameterName = cph.getParameterName();
                    ParameterPath parm = new ParameterPath(componentName, parameterName);

                    if (isBuild) {
                        if (!isBuildParameter(componentName, parameterName) /*&& ! isFinalParameter(componentName, parameterName)*/) {
                            continue;
                        }
                    } else {
                        if (isBuildParameter(componentName, parameterName)) {
                            continue;
                        }
                    }
                    //If the parameter is not final or readonly, then it should be loaded via a configuration
                    if (!cph.isReadOnly() && !cph.isFinal()) {
                        //If the parameter is optional, then we don't care that it's missing
                        if (!cph.isOptional()) {
                            if (!cv.containsPath(parm)) {
                                missingParms.append(new ParameterPath(componentName, parameterName).toString()).append(" ");
                            }
                        }
                    }
                }
                if (!missingParms.toString().isEmpty()) {
                    missingParmsDigest.put("\"" + configDesc.getCategoryTag(category) + "\"", missingParms.toString());
                }
            }
            // Configuration content diagnostic
            if (checkConfigView) {
                if (!unexpectedParms.toString().isEmpty() || !missingParmsDigest.isEmpty()) {
                    StringBuilder diag = new StringBuilder("the following problems occured when reading the configuration : ").append(configDesc.toString()).append("\n");
                    if (!unexpectedParms.toString().isEmpty()) {
                        diag.append("\t unexpected parameters : ").append(unexpectedParms.toString()).append("\n");
                    }
                    if (!missingParmsDigest.isEmpty()) {
                        for (Map.Entry<String, String> e : missingParmsDigest.entrySet()) {
                            diag.append("\t missing values for configuration ").append(e.getKey()).append(" : ").append(e.getValue()).append("\n");
                        }
                    }
                    log.warn(diag.toString() + "Saving the problematic configurations once they are loaded will fix them.");
                    if (BootstrapResourceUtils.getBootstrapSystemProperties().getOrDefault(PackCst.REQUIRE_COMPLETE_CONFIGURATIONS, "false").equals("true") || failOnViewProblems) {
                        throw new RuntimeException(diag.append("Load aborted.").toString());
                    }
                }
            }
        }        

        return  commitBulkChange(configDesc,isSafe, isBuild);
    }
        
    public void dropAllSubmittedChanges() {
        componentParameterHandlers.values()
                .stream().forEach(ConfigurationHandler::dropSubmittedChanges);
    }
    
    public void dropSubmittedChangesForComponent(String name) {
        getParameterSet(name).dropSubmittedChanges();
    }
    
    public Map<String, String> getSubmittedChangesForComponent(String name) {
        return getParameterSet(name).getSubmittedChanges().entrySet()
                .stream().collect(Collectors.toMap(
                        entry -> entry.getKey(), 
                        entry -> entry.getValue()));
    }
    
    public Map<String, Map<String, String>> getAllSubmittedChanges() {
        Map<String, Map<String, String>> res = new TreeMap<>();
        for (Map.Entry<String, ConfigurationHandler> entry : componentParameterHandlers.entrySet()) {
            Map<String, String> submittedChanges = getSubmittedChangesForComponent(entry.getKey());
            if (!submittedChanges.isEmpty()) {
                res.put(entry.getKey(), new TreeMap<String, String>(submittedChanges));
            }
        }
        return res;
    }
    
    public Object submitChange(String componentName, String parameterName, Object value, boolean readOnlyOk) {
        if (!isParameterConfigurable(componentName, parameterName)) {
            throw new IllegalArgumentException(componentName+"/"+parameterName + " is not a valid configuration parameter.");
        } else if (isParameterReadOnly(componentName, parameterName) && !readOnlyOk) {
            throw new IllegalArgumentException(componentName+"/"+parameterName + " is a read-only configuration parameter.");
        } else if (isBuildParameter(componentName, parameterName) && failForBuildParameters) {
            throw new IllegalArgumentException(componentName+"/"+parameterName + " is a build configuration parameter.");
        } else {
            return getParameterSet(componentName).submitChange(parameterName, value);
        }
    }

    public Object submitChange(String componentName, String parameterName, Object value) {
        return submitChange(componentName, parameterName, value, false);
    }

    public boolean hasSubmittedChanges() {
        for (Map.Entry<String, ConfigurationHandler> e1 : componentParameterHandlers.entrySet()) {
            if (e1.getValue().hasSubmittedChanges()) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * This commit operation should operate under the control of a configuration lock.
     * @param categoriesToCommit
     * @return 
     */
    public ConfigurationView commitBulkChange(ConfigurationDescription categoriesToCommit)  {
        return commitBulkChange(categoriesToCommit, false, false);
    }
    private ConfigurationView commitBulkChange(ConfigurationDescription categoriesToCommit, boolean isSafe, boolean isBuild)  {

        // TO-DO : should before == currentConfigurationInfo be checked ?
        ConfigurationView before = getLiveConfigurationView();
        ConfigurationView submittedChanges = new ConfigurationView();
        
        for (Map.Entry<String, ConfigurationHandler> e1 : componentParameterHandlers.entrySet()) {
            e1.getValue().trimSubmittedChanges();
            for(Map.Entry<String, String> e : e1.getValue().getSubmittedChanges().entrySet()) {
                submittedChanges.putParameterValue(e1.getKey(), e.getKey(), e.getValue());
            }
        }
        
        log.fine("processing the following submitted changes :" + submittedChanges.getAsParameterPathMap());    
        
        // Test against the configuration validation methods
        BulkValidationException excThrown = null;
        if ( ! isBuild ) {
            try {
                for (ConfigurationHandler ch : componentParameterHandlers.values()) {
                    ch.invokeValidateBulkChange(isSafe);
                }
            } catch (Exception ex) {
                excThrown = new BulkValidationException(ex);
            } finally {
                ConfigurationView afterValidation = getLiveConfigurationView();
                ConfigurationView diff = before.diff(afterValidation);
                if (!diff.isEmpty()) {
                    dropAllSubmittedChanges();
                    Set<ParameterPath> suspect = diff.getAsParameterPathMap().keySet();
                    throw new BulkSettingException("some parameters have been modified during the validation step : " + suspect, excThrown);
                }
                if (excThrown != null) {
                    throw excThrown;
                }
            }
        }
        
        // processing the bulk change
        for (Map.Entry<String, ConfigurationHandler> entry : componentParameterHandlers.entrySet()) {
            ConfigurationHandler ch = entry.getValue();
            try {
                ch.invokeSetParameters(before.getValuesForComponent(entry.getKey()),isBuild);
            } catch (Exception ex) {
                dropAllSubmittedChanges();
                throw new BulkSettingException(ex.getMessage(), ex);
            }
        }
        
        ConfigurationView after = getLiveConfigurationView();
        ConfigurationView diff = before.diff(after);
        
        // Check for consistency with the original submitted changes
        for (Map.Entry<ParameterPath, String>  pathVal : diff.getAsParameterPathMap().entrySet()) {
            ParameterPath path = pathVal.getKey();
            if (isParameterReadOnly(path.getComponentName(), path.getParameterName())) {                
                continue;
            }
            if(!submittedChanges.containsPath(path)) {
                dropAllSubmittedChanges();
                throw new BulkSettingException("the parameter " + path 
                        + " was not supposed to change from " + before.getPathValue(path)
                        + " to " + pathVal.getValue());
            } else if (!submittedChanges.getPathValue(path).equals(pathVal.getValue())) {
                dropAllSubmittedChanges();
                throw new BulkSettingException("wrong value for parameter : " 
                        + path.toString()
                        + ", expected : "+submittedChanges.getPathValue(path)
                        + ", actual : " + pathVal.getValue());
            }
        }
        
        dropAllSubmittedChanges();
        after.setConfigurationDescription(categoriesToCommit);
        return after;
    }
        
    public ConfigurationView setSingleParameter(String componentName, String parameterName, Object value) {
        return setSingleParameter(componentName, parameterName, value, false);        
    }
    public ConfigurationView setSingleParameter(String componentName, String parameterName, Object value, boolean isReadOnlyOk) {
        if (!isParameterConfigurable(componentName, parameterName)) {
            log.info("no such parameter " + new ParameterPath(componentName, parameterName));
            return new ConfigurationView();
        }
        if ( hasSubmittedChanges() ) {
            throw new RuntimeException("Cannot change single parameter with outstanding submitted changes.");
        }        
        submitChange(componentName, parameterName, value, isReadOnlyOk);
        ConfigurationView view = commitBulkChange(null);        
        return view;
    }
    
    
    
    /**
     * Returns a live map of the values of the configurable parameters for the
     * given component.
     * @param componentName
     * @param categorySet
     * @return 
     */
    public Map<String, String> getCurrentValuesForComponent(String componentName, Set<String> categorySet) {
        return getParameterSet(componentName).getCurrentValues(categorySet)
                .entrySet().stream().collect(Collectors.toMap(entry -> entry.getKey().getParameterName(), 
                        entry -> entry.getValue()));
    }

    public ConfigurationView getLiveConfigurationView()  {
        ConfigurationView res = new ConfigurationView();
        for(String compName : componentParameterHandlers.keySet()) {
            res.putValuesForComponent(compName, getCurrentValuesForComponent(compName, categories));
        }
        return res;
        
    }
    
}
