package org.lsst.ccs.config;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.utilities.structs.ParameterPath;

/**
 * A view of the configuration parameters values, ordered by the component they 
 * belong to.
 * It is used as the result of a request for loading a configuration.
 * @author LSST CCS Team
 */
public class ConfigurationView implements Serializable {

    private static final long serialVersionUID = -1596767856817881046L;
    
    private ConfigurationDescription configDesc;
    
    private final Map<String, Map<String, String>> componentsMap = new ConcurrentSkipListMap<>();

    private static final Logger LOG = Logger.getLogger(ConfigurationView.class.getName());
    
    public ConfigurationView() {
        
    }

    public ConfigurationView(ConfigurationView view) {
        this(view.getConfigurationDescription());
        putAll(view);        
    }

    
    public ConfigurationView(ConfigurationDescription configDesc) {
        this.configDesc = configDesc;
    }
    
    /**
     * 
     * @param parameterFullName the full parameter name, such as compName/parmName
     * @param value a String representation of the parameter value.
     */
    public void putParameterValue(String parameterFullName, String value) {
        try {
            ParameterPath pp = ParameterPath.valueOf(parameterFullName);
            putParameterValue(pp.getComponentName(), pp.getParameterName(), value);
        } catch (Exception e) {
            LOG.log(Level.FINER, "Skipping property "+parameterFullName+" "+value);
            
        }
    }
    
    public void putParameterValue(String componentName, String parmName, String value) {
        Map<String, String> subMap = componentsMap.get(componentName);
        if (subMap == null) {
            subMap=new ConcurrentSkipListMap<>();
            componentsMap.put(componentName, subMap);
        }
        subMap.put(parmName, value);
    }
    
    public void putAll(ConfigurationView cv) {
        //TO-DO: Better implementation!
        configDesc = cv.configDesc;
        for (Map.Entry<String, Map<String, String>> entry : cv.componentsMap.entrySet()) {
            Map<String, String> subMap = componentsMap.get(entry.getKey());
            if (subMap == null) {
                subMap = new HashMap<>();
                componentsMap.put(entry.getKey(), subMap);
            }
            subMap.putAll(entry.getValue());
        }
    }
    
    public Map<ParameterPath, String> getAsParameterPathMap() {
        Map<ParameterPath, String> res = new TreeMap<>();
        for (Map.Entry<String, Map<String, String>> e1 : componentsMap.entrySet()) {
            for (Map.Entry<String, String> e2 : e1.getValue().entrySet()) {
                res.put(new ParameterPath(e1.getKey(), e2.getKey()), e2.getValue());
            }
        }
        return res;
    }
    
    /**
     * Performs a diff view between this view and the one passed as argument.
     * Parameters that are not present in the other view or that are assigned a
     * different value are added to the resulting map.
     *
     * @param other
     * @return
     */
    public ConfigurationView diff(ConfigurationView other) {
        ConfigurationView diff = new ConfigurationView();
        for(Map.Entry<String, Map<String, String>> e1 : componentsMap.entrySet()) {
            String comp = e1.getKey();
            for(Map.Entry<String, String> e2 : e1.getValue().entrySet()) {
                String parm = e2.getKey();
                String val = e2.getValue();
                Map<String, String> subMap = other.getValuesForComponent(comp);
                String otherVal = subMap == null ? null : subMap.get(parm);
                if (!val.equals(otherVal)) {
                    diff.putParameterValue(comp, parm, otherVal);
                }
            }
        }
        return diff;
    }
    
    public boolean containsPath(ParameterPath path) {
        return componentsMap.containsKey(path.getComponentName()) && componentsMap.get(path.getComponentName()).containsKey(path.getParameterName());
    }
    
    public String getPathValue(ParameterPath path) {
        if ( ! componentsMap.containsKey(path.getComponentName()) ) {
            throw new RuntimeException("Parameter "+path.getParameterName()+" does not exist for component "+path.getComponentName()+" in this view.");
        }
        return componentsMap.get(path.getComponentName()).get(path.getParameterName());
    }
    
    public ConfigurationDescription getConfigurationDescription() {
        return configDesc;
    }
    
    public void setConfigurationDescription(ConfigurationDescription configDesc) {
        this.configDesc = configDesc;
    }
    
    public boolean isEmpty() {
        return getAsParameterPathMap().isEmpty();
    }
    
    public Map<String, String> getValuesForComponent(String componentName) {
        return componentsMap.get(componentName);
    }
    
    public void putValuesForComponent(String componentName, Map<String, String> values ) {
        for (Map.Entry<String, String> e : values.entrySet()) {
            putParameterValue(componentName, e.getKey(), e.getValue());
        }
    }
    
    public void removeValueForPath(ParameterPath pp) {        
        componentsMap.get(pp.getComponentName()).remove(pp.getParameterName());
    }
    
}


