package org.lsst.ccs.config;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import static java.util.Objects.requireNonNull;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.ConfigurationParameterType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.utilities.constraints.Constraints;
import org.lsst.ccs.utilities.conv.InputConversionEngine;
import org.lsst.ccs.utilities.conv.TypeConversionException;
import org.lsst.ccs.utilities.conv.TypeUtils;
import org.lsst.ccs.utilities.structs.ParameterPath;
import org.lsst.ccs.utilities.structs.ViewValue;

/**
 * Represents one configuration parameter. Contains all the static information about the parameter that was
 * extracted by scanning the code. The actual value isn't stored here but in the subsystem component which
 * declares the parameter, the "owning" component.
 * <p>
 * This class is immutable.
 *
 * @author LSST CCS Team.
 */
public class ConfigurationParameterHandler implements Comparable<ConfigurationParameterHandler> {

    private static final Logger LOG = Logger.getLogger(ConfigurationParameterHandler.class.getName());

    /**
     * The path of the owning component in the subsystem component tree, with the name of the parameter itself
     * included.
     */
    private final ParameterPath parameterPath;

    /**
     * The annotation used to define the parameter. Cannot be null.
     */
    private final ConfigurationParameter annotation;

    /**
     * Refers to the method annotated with @ConfigurationParameterChanger for this parameter in the owning
     * component. Will be null if there is no such method.
     */
    private final Method parameterChangerMethod;

    /**
     * Refers to the parameter's field in the owning component. Cannot be null.
     */
    private final Field parameterField;

    /**
     * Refers to the owning component. Cannot be null.
     */
    private final Object target;

    /**
     * Indicates whether the parameter requires special handlie, e.g., is read-only. <em>Not</em> the data type
     * of the parameter. Cannot be null.
     */
    private final ConfigurationParameterType configType;

    /**
     * True if and only if the parameter is an array, a collection or a map.
     */
    private final boolean hasLength;

    /**
     * The name of the category to which the parameter belongs. Normally this will be the category defined in
     * the parameter's defining annotation. The exception is when the parameter is a build parameter, which
     * has isBuild=true, in which case the reserved category name "build" is used. Never null but will be the
     * empty string if this isn't a build parameter and no category was declared in the annotation.
     */
    private final String categoryName;

    /**
     * Sets the fields describing the parameter.
     *
     * @param componentName the name of the owning component. Must not be null. Must be unique in the subsystem
     * so it will be either a full path name or just the component's own name, depending on whether
     * the CCS "use full paths" option is in effect.
     * @param m refers to the method that the owning component annotated as the configuration changer,
     * or null if there was no such method.
     * @param f refers to the parameter value field in the owning component. Must not be null.
     * @param target refers to the owning component. Must not be null.
     * @param isNewImplementation tells whether the new, in 2020, implementation of calibration is in effect.
     * If true then parameters annotated with {@code isBuild=true} are forced to be in the reserved category "build".
     * @throws NullPointerException if any of {@code componentName}, {@code parameterField} or {@code target} are null,
     * or if no annotation for the parameter field exists.
     * @throws SecurityException if for some reason the {@code Method parameterChangerMethod} or the {@code Field parameterField} can't
     * be made accessible via reflection.
     */
    ConfigurationParameterHandler(String componentName, Method m, Field f, Object target, boolean isNewImplementation) {
        this.parameterChangerMethod = m;
        if (m != null) {m.setAccessible(true);}
        this.parameterField = requireNonNull(f, "Parameter field reference must not be null.");
        f.setAccessible(true);
        this.target = requireNonNull(target, "target must not be null.");
        this.annotation = requireNonNull(f.getAnnotation(ConfigurationParameter.class), "No annotation found for the parameter.");
        this.parameterPath = new ParameterPath(
                requireNonNull(componentName, "Component name must not be null."), annotation.name().isEmpty() ? f.getName() : annotation.name());
        this.configType = annotation.isFinal() ? ConfigurationParameterType.FINAL
                : annotation.isBuild() ? ConfigurationParameterType.BUILD
                : annotation.isReadOnly() ? ConfigurationParameterType.READ_ONLY
                : ConfigurationParameterType.RUNTIME;

        final Class<?> type = parameterField.getType();
        this.hasLength = type.isArray() || Collection.class.isAssignableFrom(type) || Map.class.isAssignableFrom(type);
        String catName = requireNonNull(annotation.category(), "Null category name");
        if (isNewImplementation && annotation.isBuild()) {
            if (!catName.isEmpty()) {
                LOG.log(Level.WARNING, "Changing assigned category {0} for isBuild parameter {1} to \"build\"", new Object[]{catName, parameterPath});
            }
            catName = "build";
        } else {
            catName = annotation.category();
        }
        this.categoryName = catName;
    }

    /**
     * Sets a parameter value when a bulk change is not being used. If a configuration parameter changing
     * method was defined then it's called to make the change. Otherwise the parameter field is set
     * directly via reflection.
     *
     * @param arg the new value for the parameter.
     */
    void invokeSetParameter(Object arg) {
        try {
            if (parameterChangerMethod != null) {
                parameterChangerMethod.invoke(target, arg);
            } else {
                parameterField.set(target, arg);
            }
        } catch (IllegalAccessException | InvocationTargetException ex) {
            final Throwable cause = ex.getCause();
            final String msg = cause == null ? ex.toString() : ex.getCause().getMessage();
            throw new RuntimeException("at parameter " + parameterPath + " : " + msg, ex);
        }
    }

    /**
     * Checks whether the proposed new value for the parameter exceeds the length limit, if any,
     * specified in the parameter's annotation.
     *
     * @param obj the proposed new value. Must not be null.
     * @throws IllegalArgumentException if the value's length, if any, exceeds the maximum allowed.
     * @throws NullPointerException if {@code obj} is null.
     */
    void acceptValue(Object obj) {
        requireNonNull(obj, "New parameter value is null.");
        if (hasLength && getMaxLength() > 0) {
            Class type = parameterField.getType();
            int objectSize = -1;
            if (type.isArray()) {
                objectSize = Array.getLength(obj);
            } else if (Collection.class.isAssignableFrom(type)) {
                objectSize = ((Collection) obj).size();
            } else if (Map.class.isAssignableFrom(type)) {
                objectSize = ((Map) obj).size();
            }
            if (objectSize > getMaxLength()) {
                throw new IllegalArgumentException("Size of committed value (" + objectSize + ") exceeds the maximum length:" + getMaxLength());
            }
        }
    }

    /**
     * The component that owns the configuration parameter.
     * @return The component. Never null.
     */
    public Object getTarget() {
        return target;
    }

    /**
     * The path of the owning component in the subsystem component tree, with the name of the parameter itself
     * included at the end.
     *
     * @return The path.
     */
    public ParameterPath getParameterPath() {
        return parameterPath;
    }

    /**
     * The name of the category to which the parameter belongs, if any.
     *
     * @return The category name or the empty string if the parameter was not assigned to a category.
     */
    public String getCategory() {
        return categoryName != null ? categoryName : annotation.category();
    }

    /**
     * Does the component declaring the parameter also declare a configuration change method?
     *
     * @return {@code true} if the method exists else false.
     */
    public boolean hasConfigChangerMethod() {
        return parameterChangerMethod != null;
    }

    /**
     * The configuration changer method for this parameter, if any, declared by the owning component.
     *
     * @return The {@code Method} object for the changer or null if none was declared.
     */
    public Method getConfigChangerMethod() {
        return parameterChangerMethod;
    }

    /**
     * Was the parameter declared final?
     *
     * @return {@code true} if it's final else false.
     */
    public boolean isFinal() {
        return configType == ConfigurationParameterType.FINAL;
    }

    /**
     * Was the parameter declared read-only, that is, settable only by the subsystem and not by external
     * command?
     *
     * @return {@code true} if it's read-only, else false.
     */
    public boolean isReadOnly() {
        return configType == ConfigurationParameterType.READ_ONLY;
    }

    /**
     * Is the parameter's value needed at subsystem build time?
     *
     * @return {@code true} if it's a build-time parameter else false.
     */
    public boolean isBuild() {
        return configType == ConfigurationParameterType.BUILD;
    }

    /**
     * The parameter type, another way of seeing whether the parameter is final, read-only, needed at build
     * time or none of these. Not to be confused with the type of the parameter's value as
     * returned by {@code getType()}.
     *
     * @return The parameter type enumerator, never null.
     */
    public ConfigurationParameterType getParameterType() {
        return configType;
    }

    /**
     * Is this an optional parameter, i.e., never needs to be loaded, so that its value is whatever
     * is set in the code.
     * @return {code true} if it's optional else false.
     */
    public boolean isOptional() {
        return annotation.isOptional();
    }

    /**
     * A string giving the units in which the parameter value is always expressed.
     * @return The units string, which will be the empty string if no units were assigned. Never null.
     */
    String getUnits() {
        return annotation.units();
    }

    /**
     * The description supplied, if any, in the parameter's annotation.
     * @return The description string which is empty if no description was supplied. Never null.
     */
    String getDescription() {
        return annotation.description();
    }

    /**
     * The range constraint string, if any, in the parameter's annotation.
     * @return the range constraint which is empty if no constraint was supplied. Never null.
     */
    String getRange() {
        return annotation.range();
    }

    /**
     * Checks whether a proposed new value for the parameter meets the range constraints.
     * @param val the new value.
     * @throws IllegalArgumentException if the constraint string is badly formed or
     * if the proposed value doesn't satisfy the constraints.
     */
    public final void checkAgainstConstraints(Object val) {
        String range = annotation.range();
        if (range != null) {
            // Despite what's said in the Javadoc for check(), the value doesn't
            // have to be in string form.
            Constraints.check(val, range);
        }
    }

    /**
     * The parameter name (the last component of the parameter path). This will be either the name specified
     * in the annotation or failing that the name of the parameter's field in the owning component.
     * @return The name string, never null.
     */
    String getParameterName() {
        return parameterPath.getParameterName();
    }

    /**
     * The name of the owning component, as given to the constructor.
     * @return The name string, never null.
     * @see #ConfigurationParameterHandler(java.lang.String, java.lang.reflect.Method, java.lang.reflect.Field, java.lang.Object, boolean) 
     */
    String getComponentName() {
        return parameterPath.getComponentName();
    }

    /**
     * The generic data type of the parameter's field in the owning component.
     * @return The {@code Type} object, never null.
     */
    Type getType() {
        return parameterField.getGenericType();
    }

    /**
     * The {@code Field} object representing the parameter's field in the owning component.
     * @return 
     */
    final Field getField() {
        return parameterField;
    }

    /**
     * The upper limit specified, if any, for the parameter's value.
     * @return The length limit, which is zero if no limit has been defined or if {@code hasLength()}
     * is false. Never negative.
     * @see #hasLength() 
     */
    final int getMaxLength() {
        return annotation.maxLength();
    }

    /**
     * Is the parameter an array, collection or map?
     * @return {@code true} if the parameter value has a "length" attribute, else false.
     */
    final boolean hasLength() {
        return hasLength;
    }

    /**
     * The current value of the parameter value, in CCS string form.
     * @return The value string, never null.
     */
    final String getValue() {
        return TypeUtils.stringify(getObjectValue());
    }

    /**
     * The current value of the parameter, without conversion to string.
     * @return the value object.
     * @throws RuntimeException if an {@code IllegalAccessException} was thrown when attempting
     * to read the value field in the owning component.
     */
    public Object getObjectValue() {
        Object res;
        try {
            res = parameterField.get(target);
        } catch (IllegalAccessException ex) {
            throw new RuntimeException("Could not access " + parameterField.getName(), ex);
        }
        return res;
    }

    /**
     * Expresses a potential new value for this parameter in a standard form of
     * a {@code ViewValue v}.  {@code v.getValue()} will return the value
     * converted, if needed, to the type of the parameter. {@code v.getView()}
     * will return the standard CCS string form of {@code v.getValue()}.
     * @param value the potential new parameter value.
     * @return The resulting {@code ViewValue}.
     * @throws IllegalArgumentException if the conversion fails.
     */
    public ViewValue standardize(Object value) {
        String strValue = null;
        try {
            strValue = TypeUtils.stringify(value);
            Object convertedVal =
                    InputConversionEngine
                            .convertArgToType(
                                    strValue,
                                    getType()
                            );
            return new ViewValue(TypeUtils.stringify(convertedVal), convertedVal);
        } catch (TypeConversionException ex) {
            throw new IllegalArgumentException(
                    "Failure converting : " + parameterPath.toString()
                    + " with value " + strValue
                    + " to type " + getType().getTypeName(),
                    ex);
        }
    }

    /**
     * {@inheritDoc}
     * <p>The ordering defined for instances of this class is that of their 
     * {@code ParameterPath} fields.
     * @param o the other instance to be compared with this one.
     */
    @Override
    public int compareTo(ConfigurationParameterHandler o) {
        return this.getParameterPath().compareTo(o.getParameterPath());
    }

    /**
     * {@inheritDoc}
     * Implements a hash code which depends solely on the parameter path.
     * @return the has code.
     */
    @Override
    public int hashCode() {
        int hash = 7;
        hash = 17 * hash + Objects.hashCode(this.parameterPath);
        return hash;
    }

    /**
     * {@inheritDoc}
     * Implements an equality test based on parameter path.
     * @param obj the other object to compare with this one.
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ConfigurationParameterHandler other = (ConfigurationParameterHandler) obj;
        return 0 == this.compareTo(other);
    }

}
