package org.lsst.ccs.config;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.stream.Collectors;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.utilities.conv.TypeUtils;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.structs.ParameterPath;
import org.lsst.ccs.utilities.structs.ViewValue;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Manages the static information about the configuration parameters for the entire subsystem. By "static" we
 * mean information such as parameter name, type, flags, category, etc., that's extracted by scanning the class
 * definitions of the subsystem's components and its Groovy file. It does not include the actual values of the
 * parameters except for the initial set found in the Groovy file. Nor does it contain the set of submitted
 * changes.
 * <p>
 * The instance of this class must be created and the parameter information added when the subsystem is still
 * single-threaded, during the execution of life cycle methods {@code preBuild(), postBuild(), preInit(),
 * postInit()} and {@code preStart()}. If that restriction is obeyed then all the field values will
 automatically be made visible to any thread that is started afterward, since by the Java memory model any
 actions that happen-before Thread.start() also happen-before any actions in the new thread. If no changes
 are made after {@code preStart()} then the instance will be effectively thread-safe.
 *
 * @author LSST CCS Team
 */
public class ConfigurationHandlerSet {

    private static final Logger log = Logger.getLogger("org.lsst.ccs.config");
    private final List<DataProviderInfo> list = new ArrayList<>();

    /**
     * Configuration handlers for each component indexed by the component's full path name. A LinkedHashMap is
     * used to preserve insertion order, which is the order with which the configuration handlers have to be
     * traveled when applying configurations / processing bulk changes.
     */
    private final Map<String, ConfigurationHandler> componentParameterHandlers = new LinkedHashMap<>();

    // Category membership. Each category name is associated with a set of parameter handlers.
    private final Map<String, CategoryHandler> parametersByCategories = new HashMap<>();

    // The names of all categories used in the subsystem.
    private final Set<String> categories = new HashSet<>();

    // Translates component names to full path names. Redundant when the CCS "use full paths" build option
    // is used since then component names ARE full path names. Otherwise, we need translate simple
    // names to full paths ourselves. Using this map allows us to avoid constantly having to
    // check the "use full paths" setting when want the full path name.
    private final Map<String, String> componentToPath = new HashMap<>();

    // See constructor.
    private final boolean useFullPaths;

    // See constructor.
    private final boolean checkConfigView;

    // See constructor.
    private final boolean failOnViewProblems;

    /**
     * Sets option flags and defines empty set of static configuration info.
     *
     * @param usefullPaths Is the "use full paths" build option in effect? If it is then component names will
     * be full path names. Otherwise they'll be simple names. In either case each component name will be
     * unique within the subsystem.
     * @param checkConfigView Should we check for missing or unexpected parameters afterChange loading a
 configuration?
     * @param failOnViewProblems Should we throw an exception if we check for and find unexpected or missing
     * parameters?
     */
    public ConfigurationHandlerSet(boolean usefullPaths, boolean checkConfigView, boolean failOnViewProblems) {
        this.useFullPaths = usefullPaths;
        this.checkConfigView = checkConfigView;
        this.failOnViewProblems = failOnViewProblems;
    }

    /**
     * Calls the primary constructor with {@code useFullPaths=true, checkConfigView=true} and
     * {@code failOnViewProblems=false}.
     *
     * @see #ConfigurationHandlerSet(boolean, boolean, boolean)
     */
    public ConfigurationHandlerSet() {
        this(true, true, false);
    }

    /**
     * Calls the full form of the method with {@code isNewImplementation=false}. Used mostly by tests but also
     * by {@code SafeConfigurationPropertiesFileWriter}, the configuration file writer application.
     *
     * @see #addConfigurationHandlerForObject(java.lang.String, java.lang.String, java.lang.Object, boolean)
     * @see org.lsst.ccs.config.SafeConfigurationPropertiesFileWriter
     */
    public ConfigurationHandler addConfigurationHandlerForObject(String componentName, String componentPath, Object component) {
        return addConfigurationHandlerForObject(componentName, componentPath, component, false);
    }

    /**
     * Adds a new configuration handler for the calibration parameters belonging to a particular component.
     * Should only be called while the subsystem is still running only one thread. The configuration handler
     * is built be scanning the component's class definition via reflection. The name and full path name of
     * the component will be the same if "use full paths" is in effect, see
     * {@linkplain #ConfigurationHandlerSet(boolean, boolean, boolean)}.
     *
     * @param componentName the unique name of the component. Must not be null.
     * @param componentPath the full path name of the component. Must not be null.
     * @param component the component.
     * @param isNewImplementation are the new (as of 2020) rules in effect. For example, assigning all build
     * parameters to the reserved "build" category.
     * @return the new configuration handler, or null if the component has no configuration parameters.
     */
    public ConfigurationHandler addConfigurationHandlerForObject(String componentName, String componentPath, Object component, boolean isNewImplementation) {
        ConfigurationHandler handler = ConfigurationHandlerBuilder.buildParameterSetterFromObject(componentName, component, isNewImplementation);
        componentToPath.put(componentName, componentPath);

        if (handler == null) {
            return null;
        }

        componentParameterHandlers.put(componentName, handler);
        for (ConfigurationParameterHandler cph : handler.getConfigurationParameterHandlers()) {
            String category = cph.getCategory();
            CategoryHandler set = parametersByCategories.get(category);
            if (set == null) {
                categories.add(category);
                set = new CategoryHandler(category);
                parametersByCategories.put(category, set);
            }
            set.addParameterHandler(cph);
        }
        return handler;
    }

    /**
     * Calls the main {@code initializeConfiguration()} method with {@code isBuild} and {@code isSafe} both
     * set to false and with {@code safeConfig} set to a dummy view that has only category information.
     * Intended mainly for tests.
     *
     * @param descriptionName see the main {@code initializeConfiguration()} method.
     * @return See the main {@code initializeConfiguration()} method.
     * @see #initializeConfiguration(java.lang.String, org.lsst.ccs.config.ConfigurationView, boolean, boolean)
     */
    public ConfigurationInfo initializeConfiguration(String descriptionName) {
        return ConfigurationHandlerSet.this.initializeConfiguration(descriptionName, new ConfigurationView(ConfigurationDescription.safeConfiguration(categories)));
    }

    /**
     * Calls the main {@code initializeConfiguration()} method with {@code isBuild} and {@code isSafe} both
     * set to false. Intended mainly for testing.
     *
     * @param descriptionName see the main {@code initializeConfiguration()} method.
     * @param safeView see the main {@code initializeConfiguration()} method.
     * @return See the main {@code initializeConfiguration()} method.
     * @see #initializeConfiguration(java.lang.String, org.lsst.ccs.config.ConfigurationView, boolean, boolean)
     */
    public ConfigurationInfo initializeConfiguration(String descriptionName, ConfigurationView safeView) {
        return initializeConfiguration(descriptionName, safeView, false, false);
    }

    /**
     * Generates a {@code ConfigurationInfo} report containing a description of the configuration parameters,
     * categories and their values. During a normal subsystem
     * bootstrap this method will be called twice by the configuration service; once from {@code predBuild()}
     * and once again from {@code preInit()}.
     *
     * @param descriptionName basically the subsystem name, i.e., the agent name or the name of the top node
     * in the component tree.
     * @param safeView a recently loaded initial view of the configuration parameters, i.e., their
     * descriptions, values and categories. It's expected that the categories will all have the "safe" tag.
     * @param isBuild {@code true} if and only if we're at the build phase of subsystem bootstrapping.
     * @param isSafe {@code true} if and only if we're building an initial "safe" configuration at either
     * the build or init phases.
     * @return The configuration report.
     */
    public ConfigurationInfo initializeConfiguration(String descriptionName, ConfigurationView safeView, boolean isBuild, boolean isSafe) {
        // Create an initial view whose values are the current ones. This is to used to supply values
        // for parameters omitted from the loaded view (if permitted).
        final ConfigurationView initialView;
        initialView = new ConfigurationView(ConfigurationDescription.safeConfiguration(categories));
        initialView.putAll(getLiveConfigurationView());

        try {
            // Perform checks and apply changes.
            final ChangeList allChanges = loadCategories(initialView, safeView, isBuild);
            final ChangeList changes = allChanges.filteredBy(ChangeList.notRedundant());
            initialView.putAll(safeView); //Update the initialView with all the parameters in the safeView.
            commitBulkChange(changes, initialView.getConfigurationDescription(), isSafe, isBuild);
            // Checking initial value of configuration parameters
            for (Map.Entry<ParameterPath, String> pp : initialView.getAsParameterPathMap().entrySet()) {
                if (pp.getValue().equals(TypeUtils.NULL_STR)) {
                    log.warn("parameter " + pp.getKey() + " has not been assigned a safe non null value");
                }
            }
        } catch (Exception ex) {
            throw new RuntimeException("could not load safe configuration.", ex);
        }

        // First configurationInfo creation
        ConfigurationInfo.Builder ciBuilder = new ConfigurationInfo.Builder()
                .setDescription(descriptionName).updateConfigurationDescription(safeView.getConfigurationDescription().getDescriptionName());
        for (CategoryHandler ch : parametersByCategories.values()) {
            String category = ch.getCategory();
            CategoryTag categoryTag = safeView.getConfigurationDescription().getCategoryTag(category);
            String tag0 = categoryTag != null ? categoryTag.getTags().iterator().next() : null;
            Integer version0 = categoryTag != null ? categoryTag.getTagVersion(tag0) : null;

            ciBuilder.updateCategoryInformation(category, tag0, version0, false);
//            ciBuilder.updateCategoryInformation(ch.getCategory(), 
//                    safeView.getConfigurationDescription().getCategoryTags().get(ch.getCategory()),
//                    safeView.getConfigurationDescription().getCategoryVersions().get(ch.getCategory()),
//                    false);
            for (ConfigurationParameterHandler cph : ch.getParameters()) {
                ParameterPath pp = new ParameterPath(cph.getComponentName(), cph.getParameterName());
                ciBuilder.addParameter(pp, cph.getType(), cph.getCategory(), cph.getDescription(), cph.isFinal(), cph.isReadOnly());

                DataProviderInfo dataProviderInfo = new DataProviderInfo(componentToPath.get(pp.getComponentName()), DataProviderInfo.Type.CONFIGURATION, pp.getParameterName());
                if (!useFullPaths) {
                    //Add this property only if not using full path. If we are using full path
                    //The published path is identical to the DataProviderInfo path
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.PUBLISHED_PATH, pp.getComponentName() + "/" + pp.getParameterName());
                }
                if (!cph.getUnits().isEmpty()) {
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.UNITS, cph.getUnits());
                }
                dataProviderInfo.addAttribute(DataProviderInfo.Attribute.TYPE, cph.getType().getTypeName());
                if (!cph.getDescription().isEmpty()) {
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, cph.getDescription());
                }
                dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_TYPE, cph.getParameterType().name());
                dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_CATEGORY, cph.getCategory());
                String range = cph.getRange();
                if (range != null && !range.isEmpty()) {
                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_RANGE, range);
                }
                //Uncomment this code for https://jira.slac.stanford.edu/browse/LSSTCCS-2141
//                if ( cph.hasLength() && cph.getMaxLength() != 0 ) {
//                    dataProviderInfo.addAttribute(DataProviderInfo.Attribute.CONFIG_MAX_LENGTH, String.valueOf(cph.getMaxLength()));                    
//                }
                list.add(dataProviderInfo);

                ciBuilder.updateParameter(pp, initialView.getPathValue(pp), initialView.getPathValue(pp), false);
            }
        }
        Collections.sort(list, comparator);

        return ciBuilder.setConfigurationState(ConfigurationState.UNCONFIGURED)
                .setCCSTimeStamp(CCSTimeStamp.currentTime()).build();
    }

    /**
     * Log warnings for parameters that aren't volatile or that don't define maxLength when the values are
     * arrays, maps or collections.
     */
    public void checkParameterDeclarations() {
        List<ParameterPath> missingMaxLength = new ArrayList<>();
        List<ParameterPath> missingVolatile = new ArrayList<>();
        for (ConfigurationHandler handler : componentParameterHandlers.values()) {
            for (ConfigurationParameterHandler configHandler : handler.getConfigurationParameterHandlers()) {
                if (!isVolatile(configHandler)) {
                    missingVolatile.add(configHandler.getParameterPath());
                }
                if (configHandler.hasLength() && configHandler.getMaxLength() == 0) {
                    missingMaxLength.add(configHandler.getParameterPath());
                }
            }
        }
        if (missingMaxLength.size() > 0) {
            log.log(Level.WARNING, "Collections, Maps or Arrays annotated as Configuration Parameters must provide the \"maxLength\" argument.\n"
                    + "Please update your code to specify the maximum allowed length for the following parameters: \n{0}", missingMaxLength);
        }
        if (missingVolatile.size() > 0) {
            log.log(Level.WARNING, "Fields annotated as Configuration Parameters must be declared as \"volatile\".\n"
                    + "Please update your code for the following parameters:\n{0}", missingVolatile);
        }
    }

    private static boolean isVolatile(ConfigurationParameterHandler parameterHandler) {
        return Modifier.isVolatile(parameterHandler.getField().getModifiers()) || Modifier.isFinal(parameterHandler.getField().getModifiers());
    }

    private static Comparator<DataProviderInfo> comparator = new Comparator<DataProviderInfo>() {
        @Override
        public int compare(DataProviderInfo o1, DataProviderInfo o2) {
            return o1.getFullPath().compareTo(o2.getFullPath());
        }

    };

    public List<DataProviderInfo> getDataPoviderInfoList() {
        return list;
    }

    // -- End construction phase protected methods.
    public ConfigurationHandler getConfigurationHandlerForGroovy(String componentName) {
//        if (!useFullPaths) {
//            String[] split = ldCompName.split("/");
//            if (split.length > 1) {
//                ldCompName = split[split.length - 1];
//            }
//        }
        try {
            return getParameterSet(componentName);
        } catch (Exception e) {
//            e.printStackTrace();
            return null;
        }
    }

    public ConfigurationHandler getParameterSet(String componentName) {
        ConfigurationHandler res = componentParameterHandlers.get(componentName);
        if (res == null) {
            throw new IllegalArgumentException("no such component : " + componentName);
        }
        return res;
    }

    public CategoryHandler getCategoryHandler(String category) {
        CategoryHandler res = parametersByCategories.get(category);
        if (res == null) {
            log.info("no such category : " + category);
        }
        return res;
    }

    public boolean isBuildParameter(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isBuildParameter(parameterName);
    }

    public boolean isOptionalParameter(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isOptionalParameter(parameterName);
    }

    public boolean isFinalParameter(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isFinalParameter(parameterName);
    }

    public boolean isParameterConfigurable(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isParameterConfigurable(parameterName);
    }

    public boolean isParameterReadOnly(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isParameterReadOnly(parameterName);
    }

    public Set<String> getCategorySet() {
        return Collections.unmodifiableSet(parametersByCategories.keySet());
    }
    
    public ConfigurationParameterHandler getParameterHandler(String componentName, String parameterName) {
        final ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        if (isParameterConfigurable(componentName, parameterName)) {
            final ConfigurationParameterHandler cph = ch.getConfigurationParameterHandler(parameterName);
            return cph;
        }
        else {
            throw new IllegalArgumentException(
                        componentName + "/" + parameterName + " is not a configuration parameter");
        }
    }

    /**
     * Takes a configuration view produced by loading from an external source and compares it to the current
     * set of parameters, producing a set of changes to be applied to the current set of parameter values.
     * Checks for unexpected or missing parameters in the loaded view. Final parameters may not be changed.
     * No checking for missing or unexpected parameters is done for read-only parameters
     * since they are never changed as a result of a load. 
     * <p>
     * Despite the name, this method doesn't actually load any parameter values.
     *
     * @param loadedView the configuration view resulting from the load. Parameters whose values are put on
     * the submitted change list are changes to their canonical string forms.
     * @return The set of submitted changes generated.
     * @throws IllegalArgumentException if the size of any any non-scaler value exceeds the maximum allowed
     * for the parameter.
     */
    public ChangeList loadCategories(ConfigurationView loadedView) {
        final ConfigurationView initialView = getLiveConfigurationView();
        final boolean isBuild = false;
        return loadCategories(initialView, loadedView, isBuild);
    }

    /**
     * Takes a configuration view produced by loading from an external source and compares it to the current
     * set of parameters, producing a set of changes to be applied to the current set of parameter values.
     * Checks for unexpected or missing parameters in the loaded view. If called at build time, considers only
     * build-time parameters; at later times the build-time parameters are not considered. If called during
     * the construction of an initial, safe, configuration, allows final parameters to be set; at later times
     * this isn't allowed. No checking for missing or unexpected parameters is done for read-only parameters
     * since they are never changed as a result of a load. 
     * <p>
     * Despite the name, this method doesn't actually load parameter values.
     *
     * @param initialView a view used to supply values for parameters that weren't given any in the loaded
     * view.
     * @param loadedView the configuration view resulting from the load. Parameters whose values are put on
     * the submitted change list are changes to their canonical string forms.
     * @param isBuild {@code true} if and only if we're constructing a build-time configuration.
     * @return The set of submitted changes generated.
     * @throws IllegalArgumentException if the size of any any non-scaler value exceeds the maximum allowed
     * for the parameter.
     */
    private ChangeList loadCategories(ConfigurationView initialView, ConfigurationView loadedView, boolean isBuild) {
        ConfigurationDescription loadedDescription = loadedView.getConfigurationDescription();

        // Scan the loaded view and prepare a set submitted changes for eligible parameters while noting
        // unexpected parameters in the loaded view. Also replace the original form of each value in the
        // loaded configuration with its canonical form. We don't check for the setting of final parameters.
        // Read-only parameters are never eligible. Of the remaining parameters build-time parameters are
        // eligible only at build time and non-build-time (run-time) parameters are not eligible at build time.
        StringBuilder unexpectedParms = new StringBuilder("");
        final ChangeList changes = new ChangeList();
        for (Map.Entry<ParameterPath, String> ldViewPair : loadedView.getAsParameterPathMap().entrySet()) {
            final ParameterPath ldPath = ldViewPair.getKey();
            final String ldValue = ldViewPair.getValue();
            final String ldCompName = ldPath.getComponentName();
            final String ldParmName = ldPath.getParameterName();
            final boolean eligible = !(isParameterReadOnly(ldCompName, ldParmName)
                    || (isBuild && !isBuildParameter(ldCompName, ldParmName))
                    || (!isBuild && isBuildParameter(ldCompName, ldParmName)));
            if (eligible) {
                if (isParameterConfigurable(ldCompName, ldParmName)) {
                    final ConfigurationParameterHandler cph = componentParameterHandlers.get(ldCompName).getConfigurationParameterHandler(ldParmName);
                    changes.addChange(cph, ldValue);
                    // The string-value in the view value is in canonical form, which overrides the original form.
                    loadedView.putParameterValue(ldCompName, ldParmName, changes.getViewValue(cph).getView());
                } else {
                    unexpectedParms.append(ldPath).append(" ");
                }
            }
        }

        // Scan for missing parameters in the load view. For each category mentioned in the loaded view
        // see if each eligible parameter in the category is in the loaded view. If not then supply
        // the value in the initial view and, if the parameter is not optional make a note of it.
        // Read-only and final parameters are never eligible. Of the remaining parameters
        // build-time parameters are eligible only at build time and non-build-time (run-time) parameters
        // are not eligible at build time.
        // REVIEW: should we do this ?
        Map<String, String> missingByCategory = new HashMap<>();
        if (loadedDescription != null) {
            for (String category : loadedDescription.getCategoriesSet()) {
                final StringBuilder missingInCategory = new StringBuilder("");
                final CategoryHandler ch = getCategoryHandler(category);
                for (ConfigurationParameterHandler cph : ch.getParameters()) {
                    String componentName = cph.getComponentName();
                    String parameterName = cph.getParameterName();
                    ParameterPath parm = new ParameterPath(componentName, parameterName);
                    final boolean eligible = !(cph.isReadOnly()
                            || cph.isFinal()
                            || (isBuild && !cph.isBuild())
                            || (!isBuild && cph.isBuild()));
                    if (eligible) {
                        if (!loadedView.containsPath(parm)) {
                            //When using --initialConfiguration we should never get into this block.
                            changes.addChange(cph, initialView.getPathValue(parm));
                            if (!cph.isOptional()) {
                                missingInCategory.append(new ParameterPath(componentName, parameterName).toString()).append(" ");
                            }
                        }
                    }
                }
                if (!missingInCategory.toString().isEmpty()) {
                    missingByCategory.put("\"" + loadedDescription.getCategoryTag(category) + "\"", missingInCategory.toString());
                }
            }
        }

        // Now optionally complain about any unexpected or missing parameters that we noted earlier.
        if (checkConfigView) {
            if (!unexpectedParms.toString().isEmpty() || !missingByCategory.isEmpty()) {
                StringBuilder diag = new StringBuilder("the following problems occured when reading the configuration : ").append(loadedDescription.toString()).append("\n");
                if (!unexpectedParms.toString().isEmpty()) {
                    diag.append("\t unexpected parameters : ").append(unexpectedParms.toString()).append("\n");
                }
                if (!missingByCategory.isEmpty()) {
                    for (Map.Entry<String, String> e : missingByCategory.entrySet()) {
                        diag.append("\t missing values for configuration ").append(e.getKey()).append(" : ").append(e.getValue()).append("\n");
                    }
                }
                log.warn(diag.toString() + "Saving the problematic configurations once they are loaded will fix them.");
                if (BootstrapResourceUtils.getBootstrapSystemProperties().getOrDefault(PackCst.REQUIRE_COMPLETE_CONFIGURATIONS, "false").equals("true") || failOnViewProblems) {
                    throw new RuntimeException(diag.append("Load aborted.").toString());
                }
            }
        }

        return changes;
    }

    /**
     * Validates the requested changes then, if validation succeeds, carries out the changes.
     * Validation is performed by calling the bulk verification methods in the affected components.
     * The changes are made by first calling the bulk change methods in the affected components, then
     * if needed any individual configuration parameter change methods and as a last resort
     * changing the parameter values via reflection.
     * <p>
     * Build-time parameters are never changed, nor are final ones.
     * <p>
     * Prepares and returns a configuration view reflecting the new state of affairs.
     * @param submittedChanges describes the new values requested for a set of configuration parameters.
     * If only some categories are to be changed then this argument must contain changes only for
     * those categories.
     * @param configDesc a configuration description including the set of categories to be changed. Used to
     * make a new configuration view which is returned. Must cover the same set of categories as 
     * {@code submittedChanges}.
     * @return A configuration view in which the new values have been set.
     */
    public ConfigurationView commitBulkChange(ChangeList submittedChanges, ConfigurationDescription configDesc) {
        final boolean isSafe = false;
        final boolean isBuild = false;
        return commitBulkChange(submittedChanges, configDesc, isSafe, isBuild);
    }

    /**
     * If not in the subsystem build phase, validates the requested changes.
     * If validation succeeds or was skipped, carries out the changes.
     * Validation is performed by calling the bulk verification methods in the affected components.
     * The changes are made by first calling the bulk change methods in the affected components, then
     * if needed any individual configuration parameter change methods and as a last resort
     * changing the parameter values via reflection.
     * <p>
     * At build time only changes to build-time parameters are carried out. At other times
     * changes to build-time parameters are ignored.
     * <p>
     * When making an initial (a.k.a. safe) configuration, final parameters can be set. At
     * other times this isn't allowed.
     * <p>
     * Prepares and returns a configuration view reflecting the new state of affairs.
     * 
     * @param submittedChanges describes the new values requested for a set of configuration parameters.
     * If only some categories are to be changed then this argument must contain changes only for
     * those categories.
     * @param configDesc a configuration description including the set of categories to be changed. Used to
     * make a new configuration view which is returned. Must cover the same set of categories as 
     * {@code submittedChanges}.
     * @param isSafe is {code true} if and only if we're making an initial configuration.
     * @param isBuild is {@code true} if and only if we're in the subsystem build phase.
     * @return A configuration view in which the new values have been set.
     */
    private ConfigurationView commitBulkChange(ChangeList submittedChanges, ConfigurationDescription configDesc, boolean isSafe, boolean isBuild) {

        // TODO : should beforeValidation == currentConfigurationInfo be checked ?
        ConfigurationView beforeValidation = getLiveConfigurationView();

        log.fine("processing the following submitted changes :" + submittedChanges);

        // Test against the bulk validation methods.
        BulkValidationException excThrown = null;
        if (!isBuild) {
            try {
                for (ConfigurationHandler ch : componentParameterHandlers.values()) {
                    final ChangeList targetChanges =
                            submittedChanges.filteredBy(ChangeList.forComponent(ch.getTargetComponent()));
                    ch.invokeValidateBulkChange(targetChanges, isSafe);
                }
            } catch (Exception ex) {
                excThrown = new BulkValidationException(ex);
            } finally {
                ConfigurationView afterValidation = getLiveConfigurationView();
                ConfigurationView diff = beforeValidation.diff(afterValidation);
                if (!diff.isEmpty()) {
                    Set<ParameterPath> suspect = diff.getAsParameterPathMap().keySet();
                    throw new BulkSettingException("some parameters have been modified during the validation step : " + suspect, excThrown);
                }
                if (excThrown != null) {
                    throw excThrown;
                }
            }
        }

        // processing the bulk change
        for (Map.Entry<String, ConfigurationHandler> entry : componentParameterHandlers.entrySet()) {
            ConfigurationHandler ch = entry.getValue();
            try {
                final ChangeList targetChanges =
                    submittedChanges.filteredBy(ChangeList.forComponent(ch.getTargetComponent()));
                ch.invokeSetParameters(targetChanges, beforeValidation.getValuesForComponent(entry.getKey()), isBuild);
            } catch (Exception ex) {
                throw new BulkSettingException(ex.getMessage(), ex);
            }
        }

        ConfigurationView afterChange = getLiveConfigurationView();
        ConfigurationView diff = beforeValidation.diff(afterChange);

        // For the coming consistency check make a map of expected values keyed by ParameterPath.
        final Map<ParameterPath, String> changesByPath = new HashMap<>();
        for (final Map.Entry<ConfigurationParameterHandler, ViewValue> change: submittedChanges.getEntries()) {
            final ConfigurationParameterHandler cph = change.getKey();
            final ViewValue vwval = change.getValue();
            changesByPath.put(cph.getParameterPath(), vwval.getView());
        }

        // Check for consistency with the original submitted changes
        for (Map.Entry<ParameterPath, String> pathVal : diff.getAsParameterPathMap().entrySet()) {
            ParameterPath path = pathVal.getKey();
            if (!changesByPath.containsKey(path)) {
                throw new BulkSettingException("the parameter " + path
                        + " was not supposed to change from " + beforeValidation.getPathValue(path)
                        + " to " + pathVal.getValue());
            } else if (!changesByPath.get(path).equals(pathVal.getValue())) {
                throw new BulkSettingException("wrong value for parameter : "
                        + path.toString()
                        + ", expected : " + changesByPath.get(path)
                        + ", actual : " + pathVal.getValue());
            }
        }

        afterChange.setConfigurationDescription(configDesc);
        return afterChange;
    }

    /**
     * Returns a live map of the values of the configurable parameters for the given component.
     *
     * @param componentName
     * @param categorySet
     * @return
     */
    public Map<String, String> getCurrentValuesForComponent(String componentName, Set<String> categorySet) {
        return getParameterSet(componentName).getCurrentValues(categorySet)
                .entrySet().stream().collect(Collectors.toMap(entry -> entry.getKey().getParameterName(),
                        entry -> entry.getValue()));
    }

    /**
     *
     * @return
     */
    public ConfigurationView getLiveConfigurationView() {
        ConfigurationView res = new ConfigurationView();
        for (String compName : componentParameterHandlers.keySet()) {
            res.putValuesForComponent(compName, getCurrentValuesForComponent(compName, categories));
        }
        return res;

    }

}
