package org.lsst.ccs.config;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.Map;
import static java.util.Objects.requireNonNull;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;

/**
 * Contains a static utility method that scan a subsystem component's class for configuration-related
 * annotations and build a {@code ConfigurationHandler} from the extracted information. Caches
 * the results obtained for all scanned classes in order to avoid multiple scans of
 * superclasses common to several other classes.
 * <p>
 * This class is not thread-safe and requires that the client code does any required synchronization.
 * @author The LSST CCS Team
 * @see ConfigurationHandler
 */
public class ConfigurationHandlerBuilder {

    private ConfigurationHandlerBuilder() {
    }

    /**
     * The subsystem-wide set of @ConfigurationParameterChanger methods, indexed by full class name.
     */
    private static final Map<String, Map<String, Method>> CONFIG_CHANGER_MAP = new HashMap<>();

    /**
     * The subsystem-wide set of @ConfigurationParameter fields, indexed by full class name.
     */
    private static final Map<String, Map<String, Field>> PARAMETER_MAP = new HashMap<>();

    /**
     * Builds a {@code ConfigurationHandler} object by introspecting the specified subsystem component.
     *
     * @param componentName the name of the component to be scanned. Must not be null.
     * @param component the component to be scanned. Must not be null.
     * @param isNewImplementation {@code true} if and only if the rules introduced in 2020 are in effect.
     * @return The {@code ConfigurationHandler} for the specified component or null it does not contain any
     * configuration parameters.
     */
    static ConfigurationHandler buildParameterSetterFromObject(String componentName, Object component, boolean isNewImplementation) {
        requireNonNull(componentName, "Component name argument is null.");
        final Class<?> cls = requireNonNull(component.getClass(), "Component argument is null.");
        final Map<String, Field> fieldsMap = buildParameterFieldsMap(cls);

        if (fieldsMap.isEmpty()) {
            return null;
        }

        Map<String, Method> configChangerMap = buildConfigChangerMap(cls);
        ConfigurationHandler res = new ConfigurationHandler(component, componentName);
        for (Map.Entry<String, Field> entry : fieldsMap.entrySet()) {
            res.addParameter(new ConfigurationParameterHandler(
                    componentName,
                    configChangerMap.get(entry.getKey()),
                    entry.getValue(),
                    component,
                    isNewImplementation)
            );
        }
        return res;
    }

    /**
     * Introspects a class for @ConfigChanger methods.
     *
     * @param klass the class to scan for configuration changer methods.
     * @return the set of methods with the required annotation, indexed
     * by name. May be the empty set. Never null.
     */
    private static Map<String, Method> buildConfigChangerMap(Class<?> klass) {
        String klassName = klass.getName();

        // Introspects the class for @ConfigChanger annotated methods and 
        // stores them in METHOD_MAP
        Map<String, Method> configMethods = CONFIG_CHANGER_MAP.get(klassName);
        if (configMethods == null) {
            configMethods = new HashMap<>();
            CONFIG_CHANGER_MAP.put(klassName, configMethods);

            ///////// configMethod initializer!
            //This will introspect only public methods as the getMethods() method
            //will return any class or inherited methods that are public
            for (Method method : klass.getMethods()) {
                String methodName = method.getName();
                ConfigurationParameterChanger parameterChanger = method.getAnnotation(ConfigurationParameterChanger.class);
                if (parameterChanger != null) {
                    String propertyName = parameterChanger.propertyName();
                    if (!"".equals(propertyName)) {
                        configMethods.put(propertyName, method);
                    } else {
                        //if method is "setter" then gets name of property
                        if (methodName.startsWith("set")) {
                            char firstLetter = methodName.charAt(3);
                            propertyName = Character.toLowerCase(firstLetter) + methodName.substring(4);
                            configMethods.put(propertyName, method);
                        }
                    }
                }
            }

            //Check that there are no non-public methods that are annotated with
            //The ConfigurationParameterChanger annotation.
            Class<?> clazz = klass;
            while (clazz != null) {
                for (Method method : clazz.getDeclaredMethods()) {
                    int modifiers = method.getModifiers();
                    ConfigurationParameterChanger parameterChanger = method.getAnnotation(ConfigurationParameterChanger.class);
                    if (parameterChanger != null && !Modifier.isPublic(modifiers)) {
                        throw new RuntimeException("Method " + clazz.getName() + "::" + method.getName() + " is annotated as a "
                                + "ConfigurationParameterChanger but it is not public. Please update your code: this method must be public.");
                    }
                }
                clazz = clazz.getSuperclass();
            }
        }

        return configMethods;
    }

    /**
     * Introspect the class for @ConfigurationParameter fields.
     *
     * @param klass
     * @return a map of @ConfigurationParameter fields.
     * @throw RuntimeException if two parameters end up having the same name.
     */
    private static Map<String, Field> buildParameterFieldsMap(Class<?> klass) {
        String className = klass.getName();
        Map<String, Field> res = PARAMETER_MAP.get(className);
        if (res == null) {
            res = new HashMap<>();
            PARAMETER_MAP.put(className, res);
            // Looks for inherited fields as well
            Class<?> toIntrospect = klass;
            do {
                Field[] fields = toIntrospect.getDeclaredFields();
                for (Field f : fields) {
                    ConfigurationParameter a = f.getAnnotation(ConfigurationParameter.class);
                    if (a != null) {
                        String parmName = (a.name().isEmpty()) ? f.getName() : a.name();
                        if (res.containsKey(parmName)) {
                            throw new RuntimeException("Class " + klass.getSimpleName() + " is erroneously annotated as it contains two parameters with same name : " + parmName);
                        }
                        res.put(parmName, f);
                    }
                }
                toIntrospect = toIntrospect.getSuperclass();
            } while (toIntrospect != null);
        }
        return res;
    }

}
