package org.lsst.ccs.config;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import static java.util.Objects.requireNonNull;
import java.util.Set;
import org.lsst.ccs.utilities.structs.ViewValue;

/**
 * Maintains all the configuration parameters defined in a single subsystem component.
 * <p>
 * Not thread-safe. The clients must supply any need synchronization.
 *
 * @author The LSST CCS Team
 */
public class ConfigurationHandler {
    
    // The component that owns the parameters. Must not be null.
    private final Object targetComponent;
    
    // The targetName of the component. Might be a full path targetName or a simple targetName depending on
    // the setting of the CCS "use full paths" option. In either case the targetName is unique
    // within the subsystem. Must not be null.
    private final String targetName;
    
    // The set of handlers for the parameters of the target component, indexed by name within the component.
    private final Map<String, ConfigurationParameterHandler> parameters;
    
    // Tells whether the target component has run-time configuration parameters. That is,
    // one that are not final, read-only or build-time.
    private boolean hasRuntimeParameters;
    
    /**
     * Sets up a description of a component with no configuration parameters (yet).
     * @param target the component whose configuration parameters will be managed by this instance.
     * Must not be null.
     * @param name the targetName of the target component. May be a full path targetName or
     * a simple targetName depending on the setting of the CCS "use full paths" option. In either case the
     * targetName is unique within the subsystem. Must not be null.
     * @throws NullPointerException if either {@code target} or {@code targetName} is null.
     */
    ConfigurationHandler(Object target, String name) {
        this.targetComponent = requireNonNull(target, "Target component argument is null.");
        this.targetName = requireNonNull(name, "Target component name argument is null.");
        this.parameters = new HashMap<>();
        this.hasRuntimeParameters = false;
    }
    
    /**
     * Adds a handler for a parameter to be managed by this instance. Any existing handler with the same name
     * will be replaced.
     * @param handler the parameter handler to be added. Must not be null.
     * @throws NullPointerException if {@code handler} is null.
     */
    void addParameter(ConfigurationParameterHandler handler) {
        requireNonNull(handler, "Parameter handler argument is null.");
        hasRuntimeParameters |= ! (handler.isBuild() || handler.isFinal() || handler.isReadOnly());
        parameters.put(handler.getParameterName(), handler); 
    }
    
    
    
    /**
     * Validates a bulk change affecting at least some of the configuration parameters for the target
     * component. Changes to final parameters are forbidden once we've loaded something other than
     * the initial configuration. Nothing is done if the target component doesn't implement
     * the interface {@code ConfigurationBulkChangeHandler} or if none of the submitted
     * changes affect parameters of the target component.
     * @param targetChanges the submitted changes for the target component.
     * @param isSafe {@code true} if we have not yet gone past loading the initial configuration.
     * @throws IllegalArgumentException if a constraint check fails.
     * @throws IllegalStateException if a change was submitted for a final parameter of the target module
     * where such a change is now forbidden.
     */
    void invokeValidateBulkChange(final ChangeList targetChanges, final boolean isSafe) {
        if (targetChanges.isEmpty()) return;
        
        // Check for constraint violations and other forbidden changes.
        targetChanges.checkValidity(isSafe);

        // Call the bulk change handler for the target component, if any. A view of the parameter
        // values will be supplied that has the proposed new values replacing the current ones.
        // Currently the bulk validation method in ConfigurationBulkChangehandler::validateBulkChange()
        // requires a map of changes keyed y parameter targetName rather than by ConfigurationParameterHandler.
        if (targetComponent instanceof ConfigurationBulkChangeHandler) {
            final ChangeList changedView =
                    valuesAsChanges(parameters.values())
                    .mergedWith(targetChanges);
            ((ConfigurationBulkChangeHandler) targetComponent).validateBulkChange(changedView.asBulkChanges());
        }
    }

    /**
     * Gives new values to configuration parameters in the target component. If the target component has
     * a bulk change method, by implementing the interface ConfigurationBulkChangeHandler, then that
     * method is called to do the work. If for some reason the bulk change method doesn't set all
     * the parameters that have submitted changes then we set the leftover ones individually by calling the
     * configuration parameter changer method, if any, or else via reflection.
     * The individual method is also used for all changes if the target component lacks a bulk change method.
     * <p>
     * EXCEPTION: Bulk change handlers are not called during the build phase of a subsystem. In that case
     * all changes will be made individually. This prevents bulk changers from being called twice during
     * subsystem booting, first when the initial configuration is loaded and again when the
     * run-time configuration is loaded.
     *
     * @param targetChanges the set of submitted changes for the the target component. May be empty
     * but must not be null.
     * @param before a table of the string-form values of the target component's parameters before any changes
     * were applied, indexed by parameter targetName. If a bulk changer method is called then this is used to
     * tell which parameters were set by the method and which weren't. Must not be null.
     * @param isBuild {@code true} if and only if we're in the build phase, when no bulk change methods
     * should be called and the changing of final configuration parameters is allowed.
     * @throws BulkChangeException if a bulk change method didn't leave a parameter alone or make
     * the expected change.
     * @throws IllegalArgumentException if a proposed new value fails a range check.
     * @throws IllegalStateException if an attempt is made to change a final parameter when not in the build
     * phase.
     * @throws RuntimeException if other errors occur or if a bulk change method throws.
     */
    void invokeSetParameters(final ChangeList targetChanges, Map<String, String> before, boolean isBuild) {
        requireNonNull(targetChanges, "Submitted changes argument is null.");
        requireNonNull(before, "Map of prior parameter values argument is null.");
        // Extract those changes pertaining to the target component.
       
        // Call the bulk change method, if any and if appropriate.
        if (!isBuild && targetComponent instanceof ConfigurationBulkChangeHandler && !targetChanges.isEmpty()) {
            ((ConfigurationBulkChangeHandler) targetComponent).setParameterBulk(targetChanges.asBulkChanges());
            // Check whether all the changes have been made. Each parameter in the change set
            // should either have its "before" value or the new submitted value, anything else
            // is an error. Any of the targeted parameters having their "before" values are
            // set individually. Value comparisons use the string forms of the values.
            for (final Entry<ConfigurationParameterHandler, ViewValue> change: targetChanges.getEntries()) {
                final String parmName = change.getKey().getParameterName();
                final String beforeValue = before.get(parmName);
                final String submittedValue = change.getValue().getView();
                final String actualValue = change.getKey().getValue();
                if (actualValue.equals(beforeValue)) {
                        getConfigurationParameterHandler(parmName)
                                .invokeSetParameter(change.getValue().getValue());
                }
                else if (!actualValue.equals(submittedValue)) {
                    final String msg = String.format("After calling the bulk change method in component %s, " +
                            "the configuration parameter %s had neither its old value of %s " +
                            "nor the submitted value of %s. Actual value was %s.",
                            targetName,
                            change.getKey().getParameterName(),
                            beforeValue,
                            submittedValue,
                            actualValue);
                    throw new BulkChangeException(msg);
                }
            }
        }
        else {
            // No bulk change method exists or it isn't to be called.
            for (final Entry<ConfigurationParameterHandler, ViewValue> entry: targetChanges.getEntries()) {
                entry.getKey().invokeSetParameter(entry.getValue().getValue());
            }
        }
    }
    
    /**
     * Indicates a problem discovered during a bulk change operation.
     */
    public static class BulkChangeException  extends RuntimeException {
        private static final long serialVersionUID = 1L;
        public BulkChangeException(final String msg) {super(msg);}
    }
    
    public boolean isBulkChanger() {
        return targetComponent instanceof ConfigurationBulkChangeHandler;
    }
    
    Collection<ConfigurationParameterHandler> getConfigurationParameterHandlers() {
        return Collections.unmodifiableCollection(parameters.values());
    }
    
    /**
     * Takes a collection of parameters and provides a view of their current values
     * expressed as a {@code ChangeList}.
     *
     * @param parmSet the collection of parameters whose values are required.
     * @return the change list of values.
     */
    private ChangeList valuesAsChanges(Collection<ConfigurationParameterHandler> parmSet) {
        final ChangeList result = new ChangeList();
        for (final ConfigurationParameterHandler cph: parmSet) {
            result.addChange(cph, cph.getObjectValue());
        }
        return result;
    }
    
    
    /**
     * Gets all the current parameter values in string form for the target component
     * for all categories or a given set of categories.
     * @param categorySet the set of categories for which parameter values are to be obtained. If it's
     * empty then values for all categories will be gotten. Must not be null.
     * @return A map from parameter handler to string-form value. Never null.
     */
    Map<ConfigurationParameterHandler, String> getCurrentValues(Set<String> categorySet) {
        requireNonNull(categorySet, "Category-set argument is null.");
        Map<ConfigurationParameterHandler, String> result = new HashMap<>();
        for (ConfigurationParameterHandler cph : parameters.values()) {
            if (categorySet.isEmpty() || categorySet.contains(cph.getCategory())) {
                result.put(cph, cph.getValue());
                String strVal = cph.getValue();
            } 
        }
        return result;
    }
    
    /**
     * Does the target component contain a configuration parameter with the given name?
     * @param parameterName the name of the parameter as declared in the component. Must not be null.
     * @return {@code true} if and only if the component contains a parameter with the given name.
     * @throws NullPointerException if the parameter name argument is null.
     */
    boolean isParameterConfigurable(String parameterName) {
        return parameters.containsKey(requireNonNull(parameterName, "Pararameter name argument is null."));
    }
    
    /**
     * Does the target component contain a configuration parameter, of any kind, with the given name?
     * @param parameterName the name of the parameter as declared in the component.
     * @return {@code true} if and only if the component contains a parameter with the given name.
     * @throws NullPointerException if the parameter name argument is null.
     */
    boolean isParameterReadOnly(String parameterName) {
        ConfigurationParameterHandler cph = parameters.get(requireNonNull(parameterName, "Pararameter name argument is null."));
        return cph == null ? false : cph.isReadOnly();
    }

    /**
     * Does the target component contain a build-time configuration parameter with the given name?
     * @param parameterName the name of the parameter as declared in the component. Must not be null.
     * @return {@code true} if and only if the component contains a parameter with the given name and
     * that parameter is build-time.
     * @throws NullPointerException if the parameter name argument is null.
     */
    boolean isBuildParameter(String parameterName) {
        ConfigurationParameterHandler cph = parameters.get(requireNonNull(parameterName, "Pararameter name argument is null."));
        return cph == null ? false : cph.isBuild();
    }

    /**
     * Does the target component contain a final configuration parameter with the given name?
     * @param parameterName the name of the parameter as declared in the component. Must not be null.
     * @return {@code true} if and only if the component contains a parameter with the given name
     * and that parameter is final.
     * @throws NullPointerException if the parameter name argument is null.
     */
    boolean isFinalParameter(String parameterName) {
        ConfigurationParameterHandler cph = parameters.get(requireNonNull(parameterName, "Pararameter name argument is null."));
        return cph == null ? false : cph.isFinal();
    }

    /**
     * Does the target component contain an optional configuration parameter with the given name?
     * @param parameterName the name of the parameter as declared in the component. Must not be null.
     * @return {@code true} if and only if the component contains a parameter with the given name
     * and that parameter is optional.
     * @throws NullPointerException if the parameter name argument is null.
     */
    boolean isOptionalParameter(String parameterName) {
        ConfigurationParameterHandler cph = parameters.get(requireNonNull(parameterName, "Pararameter name argument is null."));
        return cph == null ? false : cph.isOptional();
    }

    /**
     * Returns the configuration parameter handler for the parameter of the given name in the target
     * component.
     * @param parameterName the name of the parameter as declared in the target component. Must not be null.
     * @return the handler object.
     * @throws NullPointerException if the parameter name argument is null.
     * @throws IllegalArgumentException if no such parameter exists.
     */
    public ConfigurationParameterHandler getConfigurationParameterHandler(String parameterName) {
        ConfigurationParameterHandler res = parameters.get(requireNonNull(parameterName, "Pararameter name argument is null."));
        if (res == null) {
            throw new IllegalArgumentException("No such configuration parameter: " + parameterName);
        }
        return res;
    }

    /**
     * Does the target component have any configuration parameters which are not build-time, final or read-only?
     * @return {code true} if and only if such a parameter exists.
     */
    public boolean hasRuntimeParameters() {
        return hasRuntimeParameters;
    }

    /**
     * Gets the subsystem component whose parameters are managed by this instance.
     * @return The component.
     */
    public Object getTargetComponent() {
        return targetComponent;
    }

    /**
     * Gets the name of the subsystem component whose parameters are managed by this instance.
     * @return The component name.
     */
    public String getTargetName() {
        return targetName;
    }
}
