package org.lsst.ccs.config;

import java.util.Set;
import org.lsst.ccs.bus.data.ConfigurationInfo;

/**
 * DAO interface to configuration information.
 * @author LSST CCS Team
 */
public interface ConfigurationDAO {
    
    /**
     * For each pair 'category:configuration' in taggedCategories, the category is loaded
     * with its specified configuration, categories that are not mentioned are left untouched.
     * The configuration context remains active
     * @param configDescription the description of the configuration to load.
     * @return a ConfigurationView representing the new values to apply to the parameters.
     * @throws org.lsst.ccs.config.ConfigurationServiceException
     */
    public ConfigurationView loadConfiguration(ConfigurationDescription configDescription) throws ConfigurationServiceException ;
    
    /**
     * Request for a registration to the remote configuration server
     * @param configInfo a ConfigurationInfo object that describes the current subsystem's configuration state
     * @return for each category, the name and version of the resulted configuration
     */
    public ConfigurationDescription registerConfiguration(ConfigurationInfo configInfo) throws ConfigurationServiceException;
    
    /**
     * Changes made in the specified categories are saved under the newly specified
     * name for this category, changes on parameters that belong to other categories
     * are left unchanged.
     * @param configDesc the categories to save and their new tag
     * @param configInfo the current configuration info
     * @return for each category, the name and version of the resulted configuration
     * @throws ConfigurationServiceException if the configuration service is not available.
     */
    public ConfigurationDescription saveChangesForCategoriesAs(ConfigurationDescription configDesc, ConfigurationInfo configInfo) throws ConfigurationServiceException ;
    
    /**
     * Loads a pre-defined combination of category configurations.
     *
     * @param name
     * @param version the version of the global configuration tag. If negative,
     * returns the latest version
     * @return 
     */
    ConfigurationView loadGlobalConfiguration(String name, int version);
    
    Set<String> findAvailableConfigurationsForCategory(String category);
    
    /**
     * Checks whether the configuration service is unavailable or not.
     * @return true if the configuration service is available, false otherwise.
     */
    boolean isAvailable();
    
    public default String locateConfigurations(ConfigurationInfo configInfo) {
        return null;
    }

    
}
