package org.lsst.ccs.config;

import java.util.AbstractMap.SimpleEntry;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import static java.util.Objects.requireNonNull;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;
import org.lsst.ccs.utilities.structs.ViewValue;

/**
 * Holds a set of proposed new parameter values indexed by the handler for the parameter. The change itself
 * is a pair consisting of the internal form, an arbitrary Object, and the external value, a string.
 * The external form conforms to the CCS standard for the data type. Lists are of the form
 * [a, b, c, ...]. Maps are of the form [key1:value1, key2:value2, ...]. Java Instants and
 * Durations have their standard forms. So do Integer, Double, and of course String values.
 * Other types can be represented as well provided that their classes have constructors
 * that take a single String argument.
 * <p>
 * An empty instance of this class can be created from scratch using the constructor. Other new instances
 * may be created from an existing one using its {@code filteredBy()} or {@code filteredByNot()} methods.
 * Filtering requires a {@code BiPredicate} whose first argument is a parameter handler and whose second is a
 * {@code ViewValue}.There are static methods in this class which make bi-predicates for common
 * filtering criteria. It is also possible to produce a new instance which is a merging of two existing ones.
 * <p>
 * Not thread-safe.
 * @author tether
 */
public final class ChangeList {
    

    /**
     * Creates a predicate which filters out redundant changes, that is, changes that don't actually
     * change the parameter's current value, as determined by comparison of the CCS string forms
     * of the values.
     * @return The required predicate.
     */
    public static
    BiPredicate<ConfigurationParameterHandler, ViewValue>
    notRedundant()
    {
        return (cph, viewVal) -> {
            final String newStrValue = viewVal.getView();
            final String oldStrValue = cph.getValue();
            return !oldStrValue.equals(newStrValue);
        };
    }

    /**
     * Creates a predicate which retains changes for parameters that are members of any
     * of the given categories.
     * @param categories the collection of category names.
     * @return The required predicate.
     * @throws NullPointerException if the {@code categories} argument is null.
     */
    public static
    BiPredicate<ConfigurationParameterHandler, ViewValue>
    forCategories(final Collection<String> categories)
    {
        requireNonNull(categories, "The categories argument is null.");
        return (cph, viewVal) -> {
            return categories.contains(cph.getCategory());
        };
    }

    /**
     * Creates a predicate which selects changes in which the parameter is in the given component.
     * @param component the component.
     * @throws NullPointerException if the {@code component} argument is null.
     * @return The required predicate.
     */
    public static
    BiPredicate<ConfigurationParameterHandler, ViewValue>
    forComponent(final Object component) {
        return (cph, viewVal) -> {
            return cph.getTarget() == component;
        };
    }

    /**
     * Creates a predicate which selects changes for parameters belonging to the compoent
     * with the given name.
     * @param componentName the name of the component. May not be null.
     * @throws NullPointerException if the {@code component} argument is null.
     * @return The required predicate.
     */
    public static
    BiPredicate<ConfigurationParameterHandler, ViewValue>
    forComponentNamed(final String componentName)
    {
        return (cph, viewVal) -> {
            return cph.getComponentName().equals(componentName);
        };
    }

    /**
     * Creates a predicate which selects changes for build-time parameters.
     * @return The required predicate.
     */
    public static
    BiPredicate<ConfigurationParameterHandler, ViewValue>
    isBuild()
    {
        return (cph, viewVal) -> {
            return cph.isBuild();
        };
    }

    /**
     * Creates a predicate which selects changes for final parameters.
     * @return The required predicate.
     */
    public static
    BiPredicate<ConfigurationParameterHandler, ViewValue>
    isFinal()
    {
        return (cph, viewVal) -> {
            return cph.isFinal();
        };
    }

    /**
     * Creates a predicate which selects changes for read-only parameters.
     * @return The required predicate.
     */
    public static
    BiPredicate<ConfigurationParameterHandler, ViewValue>
    isReadOnly()
    {
        return (cph, viewVal) -> {
            return cph.isReadOnly();
        };
    }
    
    private final Map<ConfigurationParameterHandler, ViewValue> changes;
        

    /**
     * Creates an empty change list.
     */
    public ChangeList() {
        this.changes = new HashMap<>();
    }
    
    private ChangeList(final Map<ConfigurationParameterHandler, ViewValue> changes) {
        this.changes = changes;
    }

    /**
     * Adds a new change to the list. The proposed new value in a standardized form,
     * see {@link ConfigurationParameterHandler#standardize(java.lang.Object)}.
     * @param cph The handler for the parameter whose value is to be changed. Must not be null.
     * @param newValue The proposed new value.
     * @throws IllegalArgumentException if the value can't be converted to the parameter's
     * type.
     */
    public void addChange(final ConfigurationParameterHandler cph, final Object newValue) {
        requireNonNull(cph, "The cph argument is null.");
        changes.put(cph, cph.standardize(newValue));
    }
    
    /**
     * Returns the proposed new value for a given parameter.
     * @param cph the handler for the parameter whose new value is wanted. Must not be null.
     * @return The {@code ViewValue} v, or null if the parameter has no change in this list.
     * If v is not null then {@code v.getValue()} is the internal form of the value and v.getView()
     * is the CCS string form of the value.
     */
    public ViewValue getViewValue(final ConfigurationParameterHandler cph) {
        requireNonNull(cph, "The cph argument is null.");
        return changes.get(cph);
    }

    /**
     * Creates a new change list where changes in a second list are added to those in this list.
     * If this list and the second list have a change for the same parameter then the change
     * in the second list is used.
     * @param other the other change list.
     * @return A new change
     */
    public ChangeList mergedWith(final ChangeList other) {
        final Map<ConfigurationParameterHandler, ViewValue> result = new HashMap<>(changes);
        result.putAll(other.changes);
        return new ChangeList(result);
    }
    
    /**
     * Provides a read-only set view of the entries in this instance. Each entry is a pair of a 
     * parameter handler and it's view-value, which is itself a pair of the actual value object and the
     * value in CCS string form.
     * @return 
     */
    public Set<Entry<ConfigurationParameterHandler, ViewValue>> getEntries() {
        return Collections.unmodifiableSet(changes.entrySet());
    }
    
    /**
     * Is the list of changes empty?
     * @return {@code true} if and only if this instance contains no changes.
     */
    public boolean isEmpty() {return changes.isEmpty();}
    
    /**
     * Creates a new change list whose entries pass the given filter
     * predicate.
     * @param keep the bi-predicate used to select changes. Its first argument
     * is the handler for the parameter to be changed and its second is the view-value
     * representing the desired new value.
     * @return 
     */
    public ChangeList
    filteredBy(final BiPredicate<ConfigurationParameterHandler, ViewValue> keep) {
        return new ChangeList(
                changes.entrySet().stream()
                .filter(  change -> keep.test(change.getKey(), change.getValue())  )
                .collect(Collectors.toMap(Entry::getKey, Entry::getValue))
        );
    }
    
    /**
     * Creates a new change list whose entries fail the given filter
     * predicate.
     * @param keep the bi-predicate used to select changes. Its first argument
     * is the handler for the parameter to be changed and its second is the view-value
     * representing the desired new value.
     * @return 
     */
    public ChangeList
    filteredByNot(final BiPredicate<ConfigurationParameterHandler, ViewValue> keep) {
        return new ChangeList(
                changes.entrySet().stream()
                .filter(  change -> !keep.test(change.getKey(), change.getValue())  )
                .collect(Collectors.toMap(Entry::getKey, Entry::getValue))
        );
    }
    
    /**
     * Converts the set of changes, assumed to be all for a single component, to a form acceptable
     * to a bulk change validation method or a bulk change method of a component that implements
     * @return The changes as a map from unqualified parameter name to raw value.
     * @see ConfigurationBulkChangeHandler
     */
    public Map<String, Object> asBulkChanges() {
            // A bulk validation or bulk change method knows nothing of the classes ConfigurationParameter or
            // ViewValue so we need to present it with parameter names and raw Object values.
            return changes.entrySet().stream()
                    .map(entry -> new SimpleEntry<>(entry.getKey().getParameterName(), entry.getValue().getValue()))
                    .collect(Collectors.toMap(Entry::getKey, Entry::getValue));
    }

    
    /**
     * Checks the proposed new parameter value against any restrictions on value range, collection size,
     * etc. Also detects attempted changes of final parameters when a configuration other
     * than the initial one has been loaded.
     *
     * @param isSafe {@code true} if and only if the initial configuration is still loaded.
     * @throws NullPointerException if the new value is null.
     * @throws IllegalArgumentException if the new value is non-null but invalid.
     * @throws IllegalStateException if a change was submitted for a final parameter of the target component
     * where such a change is now forbidden.
     */
    public void checkValidity(boolean isSafe) {
        for (final Entry<ConfigurationParameterHandler, ViewValue> entry: changes.entrySet()) {
            final ConfigurationParameterHandler cph = entry.getKey();
            final ViewValue newValue = entry.getValue();
            if (!isSafe && cph.isFinal()) {
                throw new IllegalStateException("Final parameter "
                                        + cph.getParameterPath()
                                        + " is not modifiable at runtime.");
            }
            else {
                cph.checkAgainstConstraints(newValue.getValue());
                cph.acceptValue(newValue.getValue());
            }
        }
    }
    
    /**
     * Returns the size of the change list.
     * @return The number of changes in the list.
     */
    public int size() {return changes.size();}

    @Override
    public String toString() {
        return "ChangeList{" + "values=" + changes + '}';
    }
}
