package org.lsst.ccs.config;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Describes a loaded configuration (or one to be loaded). Includes the category name and an ordered
 * set of tag names together with a version number for each tag. The order of the tags is that in which they
 * were added and is also the order in which the tags are to be loaded.
 * 
 * Example syntax for a category tag string: 
 * <p>
 * The full syntax in EBNF is:
 * <ol>
 * <li>tagString = [catName] [":" catTags] ["*"]</li>
 * <li>catTags =  tagSpec {"|" tagSpec}</li>
 * <li>tagSpec = tagName ["(" tagVer  ")" ]</li>
 * <li>tagVer = number | "l" | "d" | "u"</li>
 * </ol>
 * where "catName" and "tagName" stand for alphanumeric strings and "number" stands for a decimal integer.
 * "l" means "latest version", "d" means "default version" and "u" means "undefined version". A final
 * asterisk denotes changes to a category that have not been saved, i.e., it's a "dirty" mark. The order
 * from left to right in which the tags are listed is the order in which they will be or have been loaded.
 * <p>A construct
 * enclosed in square brackets is optional. One inside curly braces is repeated zero or more times. The
 * pipe symbol in EBNF separates mutually exclusive choices. A symbol in quotes is to be taken literally
 * and not interpreted as EBNF.
 * <p>
 * Some examples of valid tags:
 * <ul>
 * <li>"foo" - category "foo" with the default version of the default tag "".</li>
 * <li>":" -  default category "" with the default version of the default tag "".</li>
 * <li>":tag1" -  default category "" with the default version of tag "tag1".</li>
 * <li>"c:ta(l)|tb(5)*" - category "c" was loaded from the latest version of tag "ta", then
 * from version 5 of tag "tb" with the load being dirty.</li>
 * </ul>
 * <p>
 * This class is not thread-safe.
 * @author The LSST CCS Team
 */
public class CategoryTag implements Serializable {
    
    // TODO: Add serialVersionUID.
    
    private final String categoryName;
    private final List<String> tags = new ArrayList<>();
    private final Map<String,Integer> tagVersions = new HashMap<>();
    volatile boolean hasChanges = false;
    
    /** The name of the default category. */
    public static final String DEFAULT_CONFIG_NAME = "";
    
    /** The tag of the default initial configuration. */
    public static final String DEFAULT_INITIAL_CONFIG_NAME = "defaultInitial";
    
    // TODO: Those last two definitions, being configuration names, don't really belong here and are
    // duplicated in class ConfigurationDescription.
    
    /** The name of the default category. */
    public static final String DEFAULT_CAT = "";
    
    /** The tag of the safe (initially loaded) configuration. */
    public static final String SAFE_CONFIG_NAME = "safe";
    
    /** Denotes the default version of a tag in tag strings. */
    private static final String DEFAULT_V_STRING = "d";
    
    /** Denotes the latest version of a tag in tag strings. */
    private static final String LATEST_V_STRING = "l";
    
    /** Denotes the undefined version of a tag in tag strings. */
    private static final String UNDEF_V_STRING = "u";
    
    /** The numerical value assigned to the "default" tag version. */
    public static final int DEFAULT_VERSION = -1;
    
    /** The numerical value assigned to the "latest" tag version. */
    public static final int LATEST_VERSION = -2;
    
    /** The numerical value assigned to the "undefined" tag version. */
    public static final int UNDEF_VERSION = -3;
    
    private static final Pattern CATEGORY_TAG_PATTERN = Pattern.compile("((?<category>.*?)[:])?(?<tags>.*?)(?<hasChanged>\\*?)");
    private static final Pattern TAG_PATTERN = Pattern.compile("(?<tag>.*?)([(]{1}?((?<version>[0-9]*|l|d|u))[)]{1})?");

    /**
     * Creates an instance that has only the category name.
     * @param categoryName the category name string.
     */
    public CategoryTag(String categoryName) {
        this.categoryName = categoryName;
    }

    /**
     * The name of the category.
     * @return The category name part of the tag.
     */
    public String getCategoryName() {
        return categoryName;
    }
    
    /**
     * Does this instance have the given tag?
     * @param tag the tag for which to test.
     * @return {@code true} if and only if the tag is present.
     */
    public boolean containsTag(String tag) {
        return tags.contains(tag);
    }
    
    /**
     * The version associated with the given tag. Note that this is always a number
     * @param tag the tag whose version is to be returned.
     * @return The tag version number. Special values for the special version indicators.
     * @see #DEFAULT_VERSION
     * @see #LATEST_VERSION
     * @see #UNDEF_VERSION
     */
    public Integer getTagVersion(String tag) {
        if ( ! containsTag(tag) ) {
            throw new IllegalArgumentException("Tag \""+tag+"\" does not exist for this CategoryTag");
        }
        return tagVersions.get(tag);
    }

    /**
     * {@inheritDoc}
     * @return A string in a form acceptable to {@linkplain #parseCategoryTagInput(java.lang.String) }.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getCategoryName()).append(":");
        int nTags = getNumberOfTags();
        for(String tag : getTags()) {
            nTags--;
            sb.append(tag).append("(").append(getTagVersionAsString(tag)).append(")");
            if (nTags > 0) {
                sb.append("|");
            }
        }
        if ( hasChanges ) {
            sb.append("*");
        }
        return sb.toString();
    }

    /**
     * Marks the load as "dirty" or not.
     * @param hasChanges {@code true} to mark as dirty, otherwise {@code false}.
     */
    public void setHasChanges(boolean hasChanges) {
        this.hasChanges = hasChanges;
    }
    
    private String getTagVersionAsString(String tag) {
        if ( ! containsTag(tag) ) {
            throw new IllegalArgumentException("Tag \""+tag+"\" does not exist for this CategoryTag");
        }
        Integer ver = tagVersions.get(tag);
        if (ver == null) {
            return UNDEF_V_STRING;
        }
        switch (ver) {
            case UNDEF_VERSION:
                return UNDEF_V_STRING;
            case LATEST_VERSION:
                return LATEST_V_STRING;
            case DEFAULT_VERSION:
                return DEFAULT_V_STRING;
            default:
                return String.valueOf(ver);
        }
    }
    
    /**
     * Gets the list of tags without version numbers.
     * @return The list of tag name strings.
     */
    public List<String> getTags() {
        return (tags);
    }
    
    /**
     * Ensures that this instance contains the given tag with the given version number.
     * @param tag the tag name string.
     * @param version the version number. May be one of the special numbers.
     * @see #DEFAULT_VERSION
     * @see #LATEST_VERSION
     * @see #UNDEF_VERSION
     */
    public void addOrUpdateTagVersion(String tag, Integer version) {
        if ( !tags.contains(tag) ) {
            tags.add(tag);
        }
        tagVersions.put(tag, version);
    }

    private void addTagVersion(String tag, Integer version) {
        if ( tags.contains(tag) ) {
            throw new RuntimeException("Tag: "+tag+" already exists for this CategoryTag");
        }
        addOrUpdateTagVersion(tag, version);
    }


    
    int getNumberOfTags() {
        return tags.size();
    }
    
    /**
     * Add the tag information from another instance, overriding the information for any tags present
     * in both instances.
     * @param categoryTag the other instance.
     * @return {@code this}
     */
    public CategoryTag merge(CategoryTag categoryTag) {
        for(String tag : categoryTag.getTags()) {
            addOrUpdateTagVersion(tag, categoryTag.getTagVersion(tag));
        }
        return this;
    }
    
    /**
     * Compiles a category tag string into an instance of {@code CategoryTag}.
     * @param categoryTagInput the string to compile.
     * @return The new instance of {@code CategoryTag}.
     * @throws IllegalArgumentException if the string can't be parsed.
    */
    public static final CategoryTag parseCategoryTagInput(String categoryTagInput) {
        Matcher m = CATEGORY_TAG_PATTERN.matcher(categoryTagInput);
        if (m.matches()) {
            String cat = m.group("category");
            if ( cat == null ) {
                cat = DEFAULT_CAT;
            }
            CategoryTag categoryTag = new CategoryTag(cat);
            String allTags = m.group("tags");
            String[] tags = allTags == null ? new String[]{DEFAULT_CONFIG_NAME} : allTags.split("\\|");
            
            boolean hasChanged = "*".equals(m.group("hasChanged"));
            categoryTag.setHasChanges(hasChanged);
                        
            for (String t : tags) {
                Matcher tagMatcher = TAG_PATTERN.matcher(t);
                if ( tagMatcher.matches() ) {

                    String tag = tagMatcher.group("tag");
                    if (tag == null) {
                        tag = DEFAULT_CONFIG_NAME;
                    }
                    String v = tagMatcher.group("version");
                    int ver;
                    if (v == null || v.isEmpty() || v.equals(DEFAULT_V_STRING)) {
                        ver = DEFAULT_VERSION;
                    } else if (v.equals(LATEST_V_STRING)) {
                        ver = LATEST_VERSION;
                    } else if (v.equals(UNDEF_V_STRING)) {
                        ver = UNDEF_VERSION;
                    } else {
                        ver = Integer.valueOf(v);
                    }
                    categoryTag.addTagVersion(tag, ver);
                } else {
                    throw new IllegalArgumentException("Could not match tag: "+t);
                }
            }
            return categoryTag;
        }
        throw new IllegalArgumentException("Could not parse a CategoryTag from input string: \""+categoryTagInput+"\"");
    }
    
    
}
