package org.lsst.ccs.config;

import java.io.IOException;
import java.io.PrintWriter;

import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import static org.lsst.ccs.config.ConfigurationDescription.DEFAULT_CONFIG_NAME;
import static org.lsst.ccs.config.ConfigurationDescription.SAFE_CONFIG_NAME;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.structs.ParameterPath;


/**
 * Configuration proxy that registers configurations locally as properties
 * files.
 * 
 * @author bamade
 */
public class LocalConfigurationDAO implements ConfigurationDAO {

    Logger log = Logger.getLogger("org.lsst.ccs.config");
    
    public static final Comparator<ConfigurationParameterInfo> configurationParameterComparator = new Comparator<ConfigurationParameterInfo>() {

        @Override
        public int compare(ConfigurationParameterInfo o1, ConfigurationParameterInfo o2) {
            String componentName = o1.getComponentName();
            String otherComponentName = o2.getComponentName();

            if (componentName.equals("main") && !otherComponentName.equals("main")) {
                return 1;
            }
            if (otherComponentName.equals("main") && !otherComponentName.equals("main/")) {
                return -1;
            }
            if (componentName.equals(otherComponentName)) {
                return o1.getParameterName().compareTo(o2.getParameterName());
            } else {
                return componentName.compareTo(otherComponentName);
            }
        }
    };

    /** Takes in consideration configuration files starting with the agent name. */
    private final ConfigurationWriterProvider agentWriterProvider;

    /**
     * Takes in consideration configuration files starting with the description
     * name.
     */
    private final ConfigurationWriterProvider descriptionWriterProvider;

    /**
     * Builds a configuration proxy for the subsystem described by
     * {@code subsystemDesc}.
     * 
     * @param agentName
     *            the agent name on the buses.
     * @param descriptionName
     *            the name of the description the agent was built with.
     */
    public LocalConfigurationDAO(String agentName, String descriptionName) {
        agentWriterProvider = new ConfigurationWriterProvider(agentName);
        descriptionWriterProvider = agentName != null && agentName.equals(descriptionName) ? null : new ConfigurationWriterProvider(descriptionName);
        log.debug("instanciated LocalConfigurationDAO");
    }

    /**
     * creates a String to be included in a .properties file
     * 
     * @param parmInfo
     * @return a String representation of the parameter description with the given
     *         value
     */
    public static String toPropertyString(ConfigurationParameterInfo parmInfo) {
        String currentValue = parmInfo.getCurrentValue();
        if (currentValue == null) {
            currentValue = "";
        }
        StringBuilder builder = new StringBuilder();
        String pathName = parmInfo.getPathName();
        String propName = new ParameterPath(parmInfo.getComponentName(), parmInfo.getParameterName()).toString();
        if (propName == null || "".equals(propName.trim())) {
            propName = pathName;
        } else {
            propName = propName.trim().replaceAll(" ", "\\\\ ");
        }
        builder.append(propName).append(" = ")
                .append(currentValue);

        return builder.toString();
    }

    private static String getTypeInfo(String typeName) {
        String res = typeName;
        for (String[] keyVal : TABLE) {
            if (keyVal[0].equals(typeName)) {
                return keyVal[1];
            }
        }
        return res;
    }

    private static final String[][] TABLE = { { "java.lang.Integer", "integer (example : 6)" },
            { "java.lang.String", "String" }, { "java.lang.Double", "double (example : 3.0)" },
            { "java.lang.Float", "float (example : 3.0)" }, { "java.lang.Boolean", "boolean (example : true)" },
            { "[I", "array of integers (example : [1,34,666] )" },
            { "[D", "array of doubles (example : [1.0,34.0,666.66] )" },
            { "[F", "array of floats (example : [1.0,34.0,666.66] )" },
            { "[Ljava.lang.String;", "array of Strings (example : ['hello', 'world'] )" } };

    @Override
    public ConfigurationView loadConfiguration(String agentName, ConfigurationDescription configDesc)
            throws ConfigurationServiceException {
        ConfigurationView res = new ConfigurationView(configDesc);
        log.debug("local loadconfig desc tags " + configDesc.getCategoryTags().entrySet());

        for (Map.Entry<String, String> entry : configDesc.getCategoryTags().entrySet()) {
            String cat = entry.getKey();
            String tag = entry.getValue();
            Properties configProperties = getConfigurationProperties(tag, cat);
            for (Map.Entry<Object, Object> prop : configProperties.entrySet()) {
                String name = (String) prop.getKey();
                String val = (String) prop.getValue();
                res.putParameterValue(name, val);
            }
        }
        return res;
    }

    /**
     * Get configuration properties from the file system
     * for a given tag and category.
     * Properties will be loaded for both the agentName and the description name 
     * (groovy file name). The resulting properties will be merged with the agent 
     * name one above the ones for the description name.
     * @param tag
     * @param category
     * @return
     * @throws ConfigurationServiceException 
     */
    public Properties getConfigurationProperties(String tag, String category) throws ConfigurationServiceException {
        try {
            Properties configProps = agentWriterProvider.getConfigurationProperties(tag, category);
            Properties descriptionConfigProps = descriptionWriterProvider != null ? descriptionWriterProvider.getConfigurationProperties(tag, category) : null;

            log.debug(configProps == null ? "configProps=null" : "configProps " + configProps.toString());
            log.debug(descriptionConfigProps == null ? "descriptionConfigProps=null"
                    : "descriptionConfigProps " + descriptionConfigProps.toString());
            log.debug(tag + " cf :" + DEFAULT_CONFIG_NAME + "/" + SAFE_CONFIG_NAME + ".");

            if (configProps == null && descriptionConfigProps == null) {
                if (!tag.equals(DEFAULT_CONFIG_NAME) && !tag.equals(SAFE_CONFIG_NAME)) {
                    throw new IllegalArgumentException("could not find configuration file " + tag + " for category "
                            + (category.isEmpty() ? "default" : category));
                }
            } else {
                Properties mergedProps = new Properties();
                if (descriptionConfigProps != null) {
                    mergedProps.putAll(descriptionConfigProps);
                }
                if (configProps != null) {
                    mergedProps.putAll(configProps);
                }

                return mergedProps;
            }
        } catch (IOException ex) {
            throw new ConfigurationServiceException(ex.getMessage(), ex);
        }
        return new Properties();
    }
    
    
    @Override
    public ConfigurationView loadGlobalConfiguration(String agentName, String globalName, int version) {
        try {
            String fullConfigName = agentWriterProvider.getNamedConfiguration(globalName);
            if (fullConfigName == null) {
                throw new IllegalArgumentException("no such global configuration name : " + globalName);
            }
            ConfigurationDescription configDesc = new ConfigurationDescription();
            configDesc.parseConfigurationString(fullConfigName.split(","));
            configDesc.setName(globalName, ConfigurationInfo.UNDEF_VERSION);
            return loadConfiguration(agentName, configDesc);
        } catch (IOException ex) {
            throw new ConfigurationServiceException(ex.getMessage(), ex);
        }
    }

    @Override
    public Set<String> findAvailableConfigurationsForCategory(String agentName, String category) {
        Set<String> merge = new HashSet<>();
        merge.addAll(agentWriterProvider.findMatchingConfigurations(category));
        if ( descriptionWriterProvider != null ) {
            merge.addAll(descriptionWriterProvider.findMatchingConfigurations(category));
        }
        return merge;
    }

    @Override
    public boolean isAvailable() {
        return true;
    }

    @Override
    public ConfigurationDescription registerConfiguration(String agentName, ConfigurationInfo configInfo)
            throws ConfigurationServiceException {
        ConfigurationDescription cd = new ConfigurationDescription(configInfo.getCategorySet());
        for (String cat : configInfo.getCategorySet()) {
            String configName = configInfo.getConfigNameForCategory(cat);
            cd.putTagForCategory(cat, configName, configInfo.getConfigVersion(configName));
        }
        return cd;
    }

    @Override
    public ConfigurationDescription saveChangesForCategoriesAs(String agentName, ConfigurationDescription configDesc,
            ConfigurationInfo configInfo) throws ConfigurationServiceException {
        try {
            if (configDesc.getName() != null) {
                agentWriterProvider.setNamedConfiguration(configDesc);
            }
            Map<String, List<ConfigurationParameterInfo>> valuesPerCategory = ConfigurationInfo
                    .getParameterInfoGroupByCategory(configInfo.getAllParameterInfo());
            for (Map.Entry<String, String> entry : configDesc.getCategoryTags().entrySet()) {
                PrintWriter writer;
                writer = agentWriterProvider.getConfigurationWriter(entry.getValue(), entry.getKey());
                Set<ConfigurationParameterInfo> modifiedParameters = new TreeSet<ConfigurationParameterInfo>(
                        configurationParameterComparator);

                for (ConfigurationParameterInfo cpi : valuesPerCategory.get(entry.getKey())) {
                    if (!cpi.isFinal() && ! cpi.isReadOnly()) {
                        modifiedParameters.add(cpi);
                    }
                }
                for (ConfigurationParameterInfo parameter : modifiedParameters) {
                    writer.println(toPropertyString(parameter));
                }
                writer.flush();
                writer.close();
            }
        } catch (IOException ex) {
            throw new ConfigurationServiceException("could not open file", ex);
        }
        return configDesc;
    }

    public String locateConfigurations(ConfigurationInfo configInfo) {
        StringBuilder sb = new StringBuilder("current existing configurations location :");
        Set<String> categories = new TreeSet<>(configInfo.getCategorySet());
        for (String cat : categories) {
            String cname = configInfo.getConfigNameForCategory(cat);
            String pathAgent = agentWriterProvider.locateConfiguration(cname, cat);
            if (pathAgent != null) {
                sb.append("\n").append("\"").append(cat).append("\":\"").append(cname).append("\" found in : ")
                        .append(pathAgent);
            }
            String pathDesc = descriptionWriterProvider != null ? descriptionWriterProvider.locateConfiguration(cname, cat) : null;
            if (pathDesc != null) {
                if (pathAgent != null) {
                    sb.append("\n\t overrides ").append(pathDesc);
                } else {
                    sb.append("\n").append("\"").append(cat).append("\":\"").append(cname).append("\" found in : ")
                            .append(pathDesc);
                }
            }
        }
        return sb.toString();
    }
}
