package org.lsst.ccs.config;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Handles writing and reading of configuration properties files. It builds the
 * configuration file name and then delegates to a {@code WriterProvider}
 *
 * @author LSST CCS Team
 */
class ConfigurationWriterProvider {

    
    private final WriterProvider wp;

    private static final String GLOBAL_PREFIX = "configGlobal_";

    private final String agentName;

    
    ConfigurationWriterProvider(String agentName) {
        wp = WriterProvider.getInstance();
        this.agentName = agentName;
    }

    public PrintWriter getConfigurationWriter(String tagName, String categoryName) throws IOException {
        String newName = new ConfigurationFileName(agentName, tagName, categoryName).getFullName();
        return wp.getPrintWriter(newName);
    }

    public Properties getConfigurationProperties(String tagName, String categoryName) throws IOException {
        String newName = new ConfigurationFileName(agentName, tagName, categoryName).getFullName();
        return wp.getProperties(newName);
    }

    public Set<String> findMatchingConfigurations(String category) {
        String pattern = agentName;
        if (category.isEmpty()) {
            pattern += "(_[^_]+?)??";
        } else {
            pattern += "_" + ".*?" + "_" + category;
        }
        pattern += "\\.properties";
        Set<String> fullNames = wp.findMatchingPropertiesFiles(pattern);

        return fullNames.stream().map(s -> {
            return new ConfigurationFileName(agentName, s).configName;
        }).collect(Collectors.toSet());
    }

    public String getNamedConfiguration(String name) throws IOException {
        Properties props = wp.getProperties(GLOBAL_PREFIX + agentName);
        if (props != null) {
            return props.getProperty(name);
        }
        return null;
    }

    public void setNamedConfiguration(ConfigurationDescription configDesc) throws IOException {
        Properties props = wp.getProperties(GLOBAL_PREFIX + agentName);
        if (props == null) {
            props = new Properties();
        }
        props.put(configDesc.getName(), configDesc.toString());
        PrintWriter pw = wp.getPrintWriter(GLOBAL_PREFIX + agentName);
        for (String prop : props.stringPropertyNames()) {
            pw.println(prop + "=" + (String) props.get(prop));
        }
        pw.flush();
        pw.close();
    }

    public String locateConfiguration(String configName, String categoryName) {
        String fileName = new ConfigurationFileName(agentName, configName, categoryName).getFullName();
        return wp.locateFile(fileName);
    }

    static class ConfigurationFileName {

        /** The description name. */
        private final String tagName;

        /** The configuration name. */
        private final String configName;

        /** the category name. */
        private final String catName;

        /** The full configuration file name. */
        private final String fullName;

        ConfigurationFileName(String tagName, String configName, String categoryName) {
            this.tagName = tagName;
            this.configName = configName;
            this.catName = categoryName;
            String baseName = tagName;
            if (configName.isEmpty()) {
                if (!catName.isEmpty())
                    baseName += "__" + catName;
            } else {
                baseName += "_" + configName;
                if (!catName.isEmpty())
                    baseName += "_" + catName;
            }
            this.fullName = baseName + ".properties";
        }

        ConfigurationFileName(String tagName, String fullName) {
            this.tagName = tagName;
            String trimmed = fullName.replace(".properties", "").replace(this.tagName, "");
            String[] split = trimmed.split("_");
            switch (split.length) {
            case 1:
                configName = "";
                catName = "";
                break;
            case 2:
                configName = split[1];
                catName = "";
                break;
            case 3:
                configName = split[1];
                catName = split[2];
                break;
            default:
                throw new IllegalArgumentException("file name is not a configuration file name : " + fullName);
            }
            this.fullName = fullName;
        }

        public String getFullName() {
            return fullName;
        }
    }

}
