package org.lsst.ccs.config;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import static org.lsst.ccs.bus.data.ConfigurationInfo.DEFAULT_VERSION;
import static org.lsst.ccs.bus.data.ConfigurationInfo.LATEST_VERSION;
import static org.lsst.ccs.bus.data.ConfigurationInfo.UNDEF_VERSION;

/**
 * A configuration summary.
 * @author emarin
 */
public class ConfigurationDescription implements Serializable {
    
    private static final long serialVersionUID = 2262886830605244842L;
    
    public static final String DEFAULT_CONFIG_NAME = "";
    public static final String DEFAULT_CAT = "";
    public static final String SAFE_CONFIG_NAME = "safe";
    private static final String DEFAULT_V_STRING = "d";
    private static final String LATEST_V_STRING = "l";
    private static final String UNDEF_V_STRING = "u";
    
    private static final Pattern configInputPattern = Pattern.compile("((.*?)[:])?(.*?)([(]{1}?(([0-9]*|l|d|u))[)]{1})?");
    
    /** A global name. */
    private String name = null;
    // The global version
    private int version = UNDEF_VERSION;
    
    private final Set<String> categories;
    
    private final Map<String, String> categoryTags = new HashMap<>();
    
    private final Map<String, Integer> categoryVersions = new HashMap<>();
    
    public ConfigurationDescription() {
        categories = null;
    }
    
    public ConfigurationDescription(Set<String> categories) {
        this.categories = new HashSet<>(categories);
    }
    
    public static ConfigurationDescription safeConfiguration(Set<String> categories) {
        ConfigurationDescription res = new ConfigurationDescription(categories);
        for (String cat : categories) {
            res.putTagForCategory(cat, SAFE_CONFIG_NAME, null);
        }
        return res;
    }
    
    public ConfigurationDescription parseConfigurationString(String ... taggedCategories) {
        if (taggedCategories == null || taggedCategories.length == 0) {
            return this;
        }
        for (String s : taggedCategories){
            Matcher m = configInputPattern.matcher(s);
            if (m.matches()) {
                String cat = m.group(2) == null ? DEFAULT_CAT : m.group(2);
                String tag = m.group(3) == null ? DEFAULT_CONFIG_NAME : m.group(3);
                String v = m.group(5);
                int ver;
                if (v == null || v.isEmpty() || v.equals(DEFAULT_V_STRING)) {
                    ver = DEFAULT_VERSION;
                } else if (v.equals(LATEST_V_STRING)) {
                    ver = LATEST_VERSION;
                } else if (v.equals(UNDEF_V_STRING)) {
                    ver = UNDEF_VERSION;
                } else {
                    ver = Integer.valueOf(v);
                }
                putTagForCategory(cat, tag, ver);
            } else {
                throw new IllegalArgumentException("wrong configuration string representation : " + s);
            }
        }
        return this;
    }
    
    public ConfigurationDescription withDefaults(boolean withDefaults) {
        if (withDefaults) {
            for(String cat : categories) {
                if(!categoryTags.containsKey(cat)) {
                    putTagForCategory(cat, DEFAULT_CONFIG_NAME, DEFAULT_VERSION);
                }
            }
        }
        return this;
    }
    
    public void putTagForCategory(String cat, String tag, Integer version) {
        if(categories != null && !categories.contains(cat)) {
                throw new IllegalArgumentException("unexisting category " + cat);
        }
        categoryTags.put(cat, tag);
        categoryVersions.put(cat, version);
    }
    
    public Map<String, String> getCategoryTags() {
        return Collections.unmodifiableMap(categoryTags);
    }
    
    public Map<String, Integer> getCategoryVersions() {
        return Collections.unmodifiableMap(categoryVersions);
    }
    
    public boolean isEmpty() {
        return categoryTags.isEmpty();
    }
    
    public boolean containsCategory(String category) {
        return categoryTags.containsKey(category);
    }
    
    public String getName() {
        return name;
    }
    
    public int getVersion() {
        return version;
    }

    public void setName(String name, int version) {
        this.name = name;
        this.version = version;
    }
    
    private String getVersionAsString(String cat) {
        Integer ver = categoryVersions.get(cat);
        if (ver == null) {
            return UNDEF_V_STRING;
        }
        switch (ver) {
            case UNDEF_VERSION:
                return UNDEF_V_STRING;
            case LATEST_VERSION:
                return LATEST_V_STRING;
            case DEFAULT_VERSION:
                return DEFAULT_V_STRING;
            default :
                return String.valueOf(ver);
        }
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 89 * hash + Objects.hashCode(this.categories);
        hash = 89 * hash + Objects.hashCode(this.categoryTags);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ConfigurationDescription other = (ConfigurationDescription) obj;
        return categoryTags.equals(other.categoryTags);
    }
    
    public String toString() {
        StringBuilder sb = new StringBuilder(0); 
        for (Map.Entry<String, String> e : categoryTags.entrySet()) {
            sb.append(e.getKey()).append(":").append(e.getValue())
                    .append("(").append(getVersionAsString(e.getKey())).append(")")
                    .append(",");
        }
        return sb.substring(0, sb.length()-1);
    }

}
