package org.lsst.ccs.config;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Properties;
import java.util.Set;

/**
 * Read properties from files, write them in memory.
 *
 * This class delegates the first call to getProperties to FileWriterProvider
 * (initialisation) and all other methods and later calls to getProperties to
 * InMemoryWriterProvider.
 *
 * Use for test purposes only
 *
 * @author The LSST CCS Teams
 */

class InMemoryFileWriterProvider extends WriterProvider {

    private final WriterProvider inMemoryWriterProvider = new InMemoryWriterProvider();
    private final WriterProvider fileWriterProvider = new FileWriterProvider();

    InMemoryFileWriterProvider() {

    }

    @Override
    public PrintWriter getPrintWriter(String fileName) throws IOException {
        return inMemoryWriterProvider.getPrintWriter(fileName);
    }

    @Override
    public Properties getProperties(String fileName) throws IOException {
        Properties props = inMemoryWriterProvider.getProperties(fileName);

        // props is null when the propertiesMap attribute of InMemoryWriterProvider has
        // not been instanciated. In that case, we load the properties from a file..
        if (props == null) {
            Properties fileProps = fileWriterProvider.getProperties(fileName);
            // handle case where the properties file is empty or non-existing
            if (fileProps == null) {
                return null;
            }
            // store the properties in memory
            PrintWriter printW = getPrintWriter(fileName);
            String comments = "Properties loaded from file " + fileName;
            fileProps.store(printW, comments);

            props = fileProps;
        }

        return props;
    }

    @Override
    public Set<String> findMatchingPropertiesFiles(String pattern) {
        return inMemoryWriterProvider.findMatchingPropertiesFiles(pattern);
    }

    @Override
    public String locateFile(String fileName) {
        return inMemoryWriterProvider.locateFile(fileName);
    }

}
