package org.lsst.ccs.config;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.utilities.constraints.Constraints;
import org.lsst.ccs.utilities.conv.InputConversionEngine;
import org.lsst.ccs.utilities.conv.TypeConversionException;
import org.lsst.ccs.utilities.conv.TypeUtils;
import org.lsst.ccs.utilities.structs.ParameterPath;
import org.lsst.ccs.utilities.structs.ViewValue;

/**
 *
 * @author LSST CCS Team.
 */
class ConfigurationParameterHandler {
     
    private final ParameterPath parameterPath;
    
    /** The annotation associated to this parameter. Cannot be null */
    private final ConfigurationParameter annotation;
    
    /** The @ConfigurationParameterChanger annotated methods. Can be null */
    private final Method parameterChangerMethod;
    
    /** The parameter itself. Cannot be null */
    private final Field parameterField;
    
    private final Object target;
    
    ConfigurationParameterHandler(String componentName, Method m, Field f, Object target) {
        this.parameterChangerMethod = m;
        this.parameterField = f;
        this.target = target;
        parameterField.setAccessible(true);
        annotation = f.getAnnotation(ConfigurationParameter.class);
        checkAgainstConstraints(getObjectValue());
        this.parameterPath = new ParameterPath(componentName, (annotation.name().isEmpty()) ? f.getName() : annotation.name());
    }

    /**
     * Single setting of a parameter. It first attempts to find the
     * corresponding @ParameterChanger annotated method. If not found, it 
     * attempts to set the corresponding @Parameter annotated field.
     *
     * @param parameterName the parameter
     * @param arg the new value for the parameter
     */
    void invokeSetParameter(Object arg) {
            Method m = parameterChangerMethod;
            try {
                if (m != null) {
                    m.setAccessible(true);
                    m.invoke(target, arg);
                } else {
                    Field f = parameterField;
                    f.setAccessible(true);
                    f.set(target, arg);
                }
            } catch (IllegalAccessException | InvocationTargetException ex ) {
                Throwable cause = ex.getCause();
                String msg = cause == null ? ex.toString() : ex.getCause().getMessage();
                throw new RuntimeException("at parameter " + parameterPath + " : " + msg, ex);
            }
    }
    
    String getCategory() {
        return annotation.category();
    }
    
    boolean isFinal() {
        return annotation.isFinal();
    }

    boolean isReadOnly() {
        return annotation.isReadOnly();
    }

    
    String getUnits() {
        return annotation.units();
    }
    
    String getDescription() {
        return annotation.description();
    }
    
    final void checkAgainstConstraints(Object val) {
        String range = annotation.range();
        if (range != null) {
            Constraints.check(val, range);
        }
    }
        
    String getParameterName() {
        return parameterPath.getParameterName();
    }
    
    String getComponentName() {
        return parameterPath.getComponentName();
    }
    
    Type getType() { 
        return parameterField.getGenericType();
    }
    
    final String getValue() {
        return TypeUtils.stringify(getObjectValue());
    }
    
    private Object getObjectValue() {
        Object res;
        try {
            parameterField.setAccessible(true);
            res = parameterField.get(target);
            parameterField.setAccessible(false);
        } catch (IllegalAccessException ex) {
            throw new RuntimeException("could not access " + parameterField.getName(), ex);
        }
        return res;
    }
    
    ViewValue convert(String strValue) {
        try {
            Object val = InputConversionEngine.convertArgToType(strValue,
                    parameterField.getGenericType());
            return new ViewValue(strValue, val);
        } catch (TypeConversionException ex) {
            throw new IllegalArgumentException(
                    "failure converting : "+ parameterPath.toString()
                            + " with value "+ strValue
                            + " to type "+ parameterField.getGenericType().getTypeName(),
                    ex);
        }
    }
   
}
