package org.lsst.ccs.config;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.utilities.structs.ParameterPath;

/**
 * A view of the configuration parameters values, ordered by the component they 
 * belong to.
 * It is used as the result of a request for loading a configuration.
 * @author LSST CCS Team
 */
public class ConfigurationView implements Serializable {

    private static final long serialVersionUID = -1596767856817881046L;
    
    private ConfigurationDescription configDesc;
    
    private final Map<String, Map<String, String>> componentsMap = new HashMap<>();
    
    public ConfigurationView() {
        
    }
    
    public ConfigurationView(ConfigurationDescription configDesc) {
        this.configDesc = configDesc;
    }
    
    /**
     * 
     * @param parameterFullName the full parameter name, such as compName/parmName
     * @param value a String representation of the parameter value.
     */
    public void putParameterValue(String parameterFullName, String value) {
        ParameterPath pp = ParameterPath.valueOf(parameterFullName);
        putParameterValue(pp.getComponentName(), pp.getParameterName(), value);
    }
    
    public void putParameterValue(String componentName, String parmName, String value) {
        Map<String, String> subMap = componentsMap.get(componentName);
        if (subMap == null) {
            subMap=new HashMap<>();
            componentsMap.put(componentName, subMap);
        }
        subMap.put(parmName, value);
    }
    
    public void putAll(ConfigurationView cv) {
        configDesc = cv.configDesc;
        for (Map.Entry<String, Map<String, String>> entry : cv.componentsMap.entrySet()) {
            Map<String, String> subMap = componentsMap.get(entry.getKey());
            if (subMap == null) {
                subMap = new HashMap<>();
                componentsMap.put(entry.getKey(), subMap);
            }
            subMap.putAll(entry.getValue());
        }
    }
    
    public Map<ParameterPath, String> getAsParameterPathMap() {
        Map<ParameterPath, String> res = new HashMap<>();
        for (Map.Entry<String, Map<String, String>> e1 : componentsMap.entrySet()) {
            for (Map.Entry<String, String> e2 : e1.getValue().entrySet()) {
                res.put(new ParameterPath(e1.getKey(), e2.getKey()), e2.getValue());
            }
        }
        return res;
    }
    
    /**
     * Performs a diff view between this view and the one passed as argument.
     * Parameters that are not present in the other view or that are assigned a
     * different value are added to the resulting map.
     *
     * @param other
     * @return
     */
    public ConfigurationView diff(ConfigurationView other) {
        ConfigurationView diff = new ConfigurationView();
        for(Map.Entry<String, Map<String, String>> e1 : componentsMap.entrySet()) {
            String comp = e1.getKey();
            for(Map.Entry<String, String> e2 : e1.getValue().entrySet()) {
                String parm = e2.getKey();
                String val = e2.getValue();
                Map<String, String> subMap = other.getValuesForComponent(comp);
                String otherVal = subMap == null ? null : subMap.get(parm);
                if (!val.equals(otherVal)) {
                    diff.putParameterValue(comp, parm, otherVal);
                }
            }
        }
        return diff;
    }
    
    public boolean containsPath(ParameterPath path) {
        return componentsMap.containsKey(path.getComponentName()) && componentsMap.get(path.getComponentName()).containsKey(path.getParameterName());
    }
    
    public String getPathValue(ParameterPath path) {
        return componentsMap.get(path.getComponentName()).get(path.getParameterName());
    }
    
    public ConfigurationDescription getConfigurationDescription() {
        return configDesc;
    }
    
    public void setConfigurationDescription(ConfigurationDescription configDesc) {
        this.configDesc = configDesc;
    }
    
    public boolean isEmpty() {
        return componentsMap.isEmpty();
    }
    
    public Map<String, String> getValuesForComponent(String componentName) {
        return componentsMap.get(componentName);
    }
    
    public void putValuesForComponent(String componentName, Map<String, String> values ) {
        for (Map.Entry<String, String> e : values.entrySet()) {
            putParameterValue(componentName, e.getKey(), e.getValue());
        }
    }
    
}


