package org.lsst.ccs.config;

import java.io.IOException;
import java.io.PrintWriter;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import static org.lsst.ccs.config.ConfigurationDescription.DEFAULT_CONFIG_NAME;
import static org.lsst.ccs.config.ConfigurationDescription.SAFE_CONFIG_NAME;
import org.lsst.ccs.utilities.structs.ParameterPath;

/**
 * Configuration proxy that registers configurations locally as properties files.
 * @author bamade
 */
public class LocalConfigurationDAO implements ConfigurationDAO {
    
    public static final Comparator<ConfigurationParameterInfo> configurationParameterComparator = new Comparator<ConfigurationParameterInfo>() {

        @Override
        public int compare(ConfigurationParameterInfo o1, ConfigurationParameterInfo o2) {
            String componentName = o1.getComponentName();
            String otherComponentName = o2.getComponentName();
            
            if(componentName.equals("main") && ! otherComponentName.equals("main")) {
                return 1 ;
            }
            if(otherComponentName.equals("main") && ! otherComponentName.equals("main/")) {
                return -1 ;
            }
            if (componentName.equals(otherComponentName)) {
                return o1.getParameterName().compareTo(o2.getParameterName());
            } else {
                return componentName.compareTo(otherComponentName);
            }
        }
    };
    
    private final ConfigurationWriterProvider writerProvider;
    
    /**
     * Builds a configuration proxy for the subsystem described by {@code subsystemDesc}.
     * @param agentName the agent name on the buses.
     */
    public LocalConfigurationDAO(String agentName) {
        writerProvider  = new ConfigurationWriterProvider(agentName);
    }
    
    /**
     * creates a String to be included in a .properties file
     * @param parmInfo
     * @param commentOutValue if true the description is commented out
     * @return a String representation of the parameter description with the 
     * given value
     */
    public static String toPropertyString(ConfigurationParameterInfo parmInfo, boolean commentOutValue) {
        String currentValue = parmInfo.getCurrentValue();
        if(currentValue == null) {currentValue ="" ;}
        StringBuilder builder = new StringBuilder() ;
        String pathName = parmInfo.getPathName();
        builder.append("#********  ").append(pathName) ;
        String description = parmInfo.getDescription();
        if(description != null) {
            String withPounds = description.replaceAll("\n","\n#") ;
            builder.append('\n').append("#** ").append(withPounds) ;
        }
        builder.append("\n#** type : " )
                    .append(getTypeInfo( parmInfo.getType())) ;
        if(parmInfo.getCategoryName()!= null && !"".equals(parmInfo.getCategoryName().trim())) {
            builder.append("\n#** category : ").append(parmInfo.getCategoryName());
        }
        String propName = new ParameterPath(parmInfo.getComponentName(), parmInfo.getParameterName()).toString();
        if( propName== null || "".equals(propName.trim())) {
            propName = pathName ;
        } else {
            propName = propName.trim().replaceAll(" ", "\\\\ ");
        }
        builder.append("\n#********\n").append(commentOutValue?"#":"").append(propName).append(" = ").append(currentValue).append('\n') ;

        return builder.toString() ;
    }
    
    private static String getTypeInfo(String typeName) {
        String res = typeName ;
        for(String[] keyVal : TABLE) {
            if(keyVal[0].equals(typeName)){
                return keyVal[1] ;
            }
        }
        return res ;
    }
     
     private static final String[][] TABLE = {
            {"java.lang.Integer", "integer (example : 6)"}      ,
            {"java.lang.String", "String"}      ,
            {"java.lang.Double", "double (example : 3.0)"}      ,
            {"java.lang.Float", "float (example : 3.0)"}      ,
            {"java.lang.Boolean", "boolean (example : true)"}      ,
            {"[I", "array of integers (example : [1,34,666] )"}      ,
            {"[D", "array of doubles (example : [1.0,34.0,666.66] )"}      ,
            {"[F", "array of floats (example : [1.0,34.0,666.66] )"}      ,
            {"[Ljava.lang.String;", "array of Strings (example : ['hello', 'world'] )"}
    } ;
 
    @Override
    public ConfigurationView loadConfiguration(String agentName, ConfigurationDescription configDesc) throws ConfigurationServiceException {
        ConfigurationView res = new ConfigurationView(configDesc);
        for(Map.Entry<String,String> entry : configDesc.getCategoryTags().entrySet()){
            String cat = entry.getKey();
            String tag = entry.getValue();
            try {
                Properties configProps = writerProvider.getConfigurationProperties(tag, cat);
                if (configProps == null) {
                    if (!tag.equals(DEFAULT_CONFIG_NAME) && !tag.equals(SAFE_CONFIG_NAME))
                        throw new IllegalArgumentException("could not find configuration file " + tag + " for category " + (cat.isEmpty() ? "default" : cat));
                } else {
                    for (Map.Entry<Object, Object> prop : configProps.entrySet()) {
                        String name = (String)prop.getKey();
                        String val = (String)prop.getValue();
                        res.putParameterValue(name, val);
                    }
                }
            } catch (IOException ex) {
                throw new ConfigurationServiceException(ex.getMessage(), ex);
            }
        }
        return res;    
    }
    
    @Override
    public ConfigurationView loadGlobalConfiguration(String agentName, String globalName, int version) {
        try {
            String fullConfigName = writerProvider.getNamedConfiguration(globalName);
            if (fullConfigName == null) {
                throw new IllegalArgumentException("no such global configuration name : " + globalName);
            }
            ConfigurationDescription configDesc = new ConfigurationDescription();
            configDesc.parseConfigurationString(fullConfigName.split(","));
            configDesc.setName(globalName, ConfigurationInfo.UNDEF_VERSION);
            return loadConfiguration(agentName, configDesc);
        } catch (IOException ex) {
            throw new ConfigurationServiceException(ex.getMessage(), ex);
        }
    }
    
    @Override
    public Set<String> findAvailableConfigurationsForCategory(String agentName, String category) {
        return writerProvider.findMatchingConfigurations(category);
    }

    @Override
    public boolean isAvailable() {
        return true;
    }

    @Override
    public ConfigurationDescription registerConfiguration(String agentName, ConfigurationInfo configInfo) throws ConfigurationServiceException {
        ConfigurationDescription cd = new ConfigurationDescription(configInfo.getCategorySet());
        for (String cat : configInfo.getCategorySet()) {
            String configName = configInfo.getConfigNameForCategory(cat);
            cd.putTagForCategory(cat, configName, configInfo.getConfigVersion(configName));
        }
        return cd;
    }

    @Override
    public ConfigurationDescription saveChangesForCategoriesAs(String agentName, ConfigurationDescription configDesc, ConfigurationInfo configInfo) throws ConfigurationServiceException {
         try {
            if (configDesc.getName() != null) {
                writerProvider.setNamedConfiguration(configDesc);
            }
            Map<String, List<ConfigurationParameterInfo>> valuesPerCategory = ConfigurationInfo.getParameterInfoGroupByCategory(configInfo.getAllParameterInfo());
            for (Map.Entry<String, String> entry : configDesc.getCategoryTags().entrySet()) {
                PrintWriter writer;
                writer = writerProvider.getConfigurationWriter(entry.getValue(), entry.getKey());
                Set<ConfigurationParameterInfo> modifiedParameters = new TreeSet<ConfigurationParameterInfo>(configurationParameterComparator);
                
                for (ConfigurationParameterInfo cpi : valuesPerCategory.get(entry.getKey())) {
                    if (!cpi.isFinal()) {
                        modifiedParameters.add(cpi);
                    }
                }            
                for (ConfigurationParameterInfo parameter : modifiedParameters){
                    writer.println(toPropertyString(parameter, false));
                }
                writer.flush();
                writer.close();
            }
        }  catch (IOException ex) {
            throw new ConfigurationServiceException("could not open file", ex );
        }
         return configDesc;
    }
}
