package org.lsst.ccs.config;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.utilities.conv.TypeUtils;
import org.lsst.ccs.utilities.logging.Logger; 
import org.lsst.ccs.utilities.structs.ParameterPath;

/**
 * A set of several ConfigurationHandlers.
 * @author LSST CCS Team
 */
public class ConfigurationHandlerSet {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.config");
    
    /** 
     * Configurable parameters ordered by the component they belong to. 
     * A LinkedHashMap is used to preserve insertion order, which is the order with which the 
     * configuration handlers have to be traveled when applying configurations / processing bulk changes
     */
    private final Map<String, ConfigurationHandler> componentParameterHandlers = new LinkedHashMap<>();

    /** Configurable parameters ordered by the category they belong to. */
    private final Map<String, CategoryHandler> parametersByCategories = new HashMap<>();
    
    private ConfigurationView initialView;
    
    private final Set<String> categories = new HashSet<>();
    
    public ConfigurationHandlerSet() {

    }
    
    // -- Construction phase protected methods 
    
    public boolean addConfigurationHandlerForObject(String componentName, Object component) { 
        ConfigurationHandler handler = ConfigurationHandlerBuilder.buildParameterSetterFromObject(componentName, component);
        
        if (handler == null) return false;
        
        componentParameterHandlers.put(componentName, handler);
        for (ConfigurationParameterHandler cph : handler.getConfigurationParameterHandlers()) {
            String category = cph.getCategory();
            CategoryHandler set = parametersByCategories.get(category);
            if (set == null) {
                categories.add(category);
                set = new CategoryHandler(category);
                parametersByCategories.put(category, set);
            }
            set.addParameterHandler(cph);
        }
        return true;
    }
    
    public ConfigurationInfo initialize(String descriptionName) {
        return initialize(descriptionName, new ConfigurationView(ConfigurationDescription.safeConfiguration(categories)));
    }
    
    public ConfigurationInfo initialize(String descriptionName, ConfigurationView safeView) {
        initialView = new ConfigurationView(ConfigurationDescription.safeConfiguration(categories));
        initialView.putAll(getLiveConfigurationView());        
        // Extracting the configuration parameters from the top node tags
        initialView.putAll(safeView);
        
        try {
            loadCategories(safeView);
            // Checking initial value of configuration parameters
            for (Map.Entry<ParameterPath, String> pp : initialView.getAsParameterPathMap().entrySet()) {
                if (pp.getValue().equals(TypeUtils.NULL_STR)) {
                    log.warn("parameter " + pp.getKey() + " has not been assigned a safe non null value");
                }
            }
        } catch (Exception ex) {
            throw new RuntimeException("could not load safe configuration.", ex);
        }
        
        // First configurationInfo creation
        ConfigurationInfo.Builder ciBuilder = new ConfigurationInfo.Builder()
                .setDescription(descriptionName);
        for (CategoryHandler ch : parametersByCategories.values()) {
            ciBuilder.updateCategoryInformation(ch.getCategory(), 
                    safeView.getConfigurationDescription().getCategoryTags().get(ch.getCategory()),
                    safeView.getConfigurationDescription().getCategoryVersions().get(ch.getCategory()),
                    false);
            for (ConfigurationParameterHandler cph : ch.getParameters()) {
                ParameterPath pp = new ParameterPath(cph.getComponentName(), cph.getParameterName());
                ciBuilder.addParameter(pp, cph.getType().getTypeName(), cph.getCategory(), cph.getDescription(), cph.isFinal());
                ciBuilder.updateParameter(pp, initialView.getPathValue(pp), initialView.getPathValue(pp), false);
            }
        }
        return ciBuilder.setConfigurationState(ConfigurationState.INITIAL_SAFE)
                .setTime(System.currentTimeMillis()).build();
    }
    
    // -- End construction phase protected methods.
    
    private ConfigurationHandler getParameterSet(String componentName) {
        ConfigurationHandler res = componentParameterHandlers.get(componentName);
        if (res == null) {
            throw new IllegalArgumentException("no such component : " + componentName);
        }
        return res;
    }
    
    private CategoryHandler getCategoryHandler(String category) {
        CategoryHandler res = parametersByCategories.get(category);
        if (res == null) {
            log.info("no such category : " + category);
        }
        return res;
    }
    
    public boolean isParameterConfigurable(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isParameterConfigurable(parameterName);
    }
    
    public Set<String> getCategorySet() {
        return Collections.unmodifiableSet(parametersByCategories.keySet());
    }
    
    public ConfigurationView loadCategories(ConfigurationView cv) {
        ConfigurationDescription configDesc = cv.getConfigurationDescription();
        dropAllSubmittedChanges();

        for (Map.Entry<ParameterPath, String> entry : cv.getAsParameterPathMap().entrySet()) {
            ParameterPath parm = entry.getKey();
            if (isParameterConfigurable(parm.getComponentName(), parm.getParameterName())) {
                submitChange(parm.getComponentName(), parm.getParameterName(), entry.getValue());
            } else {
                log.info(parm + " is not a configuration parameter : ignored");
            }
        }
        
        // Submitting the initial value for other components
        // REVIEW: should we do this ?

        for (String category:configDesc.getCategoryTags().keySet()) {
            boolean complete = true;
            CategoryHandler ch = getCategoryHandler(category);
            for (ConfigurationParameterHandler cph : ch.getParameters()) {
                String componentName = cph.getComponentName();
                String parameterName = cph.getParameterName();
                if(!cph.isFinal() && !cv.containsPath(new ParameterPath(componentName, parameterName))) {
                    log.finest("configuration \""+category+"\":\""+configDesc.getCategoryTags().get(category)+"\" does not specify a value for " + cph.getComponentName()+"/"+cph.getParameterName());
                    complete = false;
                    submitChange(componentName, parameterName, initialView.getPathValue(new ParameterPath(componentName, parameterName)));
                }
            }
            if (!complete && !ConfigurationDescription.SAFE_CONFIG_NAME.equals(configDesc.getCategoryTags().get(category))) {
                log.warn("configuration \""+category+"\":\""+configDesc.getCategoryTags().get(category)+"\" is incomplete. You need to save it once it is loaded.");
            }
        }
       return  commitBulkChange(configDesc);
    }
    
    public void dropAllSubmittedChanges() {
        componentParameterHandlers.values()
                .stream().forEach(ConfigurationHandler::dropSubmittedChanges);
    }
    
    public void dropSubmittedChangesForComponent(String name) {
        getParameterSet(name).dropSubmittedChanges();
    }
    
    public Map<String, String> getSubmittedChangesForComponent(String name) {
        return getParameterSet(name).getSubmittedChanges().entrySet()
                .stream().collect(Collectors.toMap(
                        entry -> entry.getKey(), 
                        entry -> entry.getValue()));
    }
    
    public Map<String, Map<String, String>> getAllSubmittedChanges() {
        Map<String, Map<String, String>> res = new TreeMap<>();
        for (Map.Entry<String, ConfigurationHandler> entry : componentParameterHandlers.entrySet()) {
            Map<String, String> submittedChanges = getSubmittedChangesForComponent(entry.getKey());
            if (!submittedChanges.isEmpty()) {
                res.put(entry.getKey(), new TreeMap<String, String>(submittedChanges));
            }
        }
        return res;
    }
    
    public void submitChange(String componentName, String parameterName, Object value) {
        if (!isParameterConfigurable(componentName, parameterName)) {
            log.info("no such parameter : " + componentName+"//"+parameterName + " : ignored");
        } else {
            getParameterSet(componentName).submitChange(parameterName, value);
        }
    }

    /**
     * This commit operation should operate under the control of a configuration lock.
     * @param categoriesToCommit
     * @return 
     */
    public ConfigurationView commitBulkChange(ConfigurationDescription categoriesToCommit)  {

        // TO-DO : should before == currentConfigurationInfo be checked ?
        ConfigurationView before = getLiveConfigurationView();
        
        ConfigurationView submittedChanges = new ConfigurationView();
        
        for (Map.Entry<String, ConfigurationHandler> e1 : componentParameterHandlers.entrySet()) {
            e1.getValue().trimSubmittedChanges();
            for(Map.Entry<String, String> e : e1.getValue().getSubmittedChanges().entrySet()) {
                submittedChanges.putParameterValue(e1.getKey(), e.getKey(), e.getValue());
            }
        }
        
        log.fine("processing the following submitted changes :" + submittedChanges);    
        
        // Test against the configuration validation methods
        BulkValidationException excThrown = null;
        try {
            for (ConfigurationHandler ch : componentParameterHandlers.values()) {
                ch.invokeValidateBulkChange(ConfigurationDescription.safeConfiguration(categories).equals(categoriesToCommit));
            }
        } catch (Exception ex) {
            excThrown = new BulkValidationException(ex);
        } finally {
            ConfigurationView afterValidation = getLiveConfigurationView();
            ConfigurationView diff = before.diff(afterValidation);
            if (!diff.isEmpty()) {
                dropAllSubmittedChanges();
                Set<ParameterPath> suspect = diff.getAsParameterPathMap().keySet();
                throw new BulkSettingException("some parameters have been modified during the validation step : " + suspect, excThrown);
            }
            if (excThrown != null) throw excThrown;
        }
        
        // processing the bulk change
        for (Map.Entry<String, ConfigurationHandler> entry : componentParameterHandlers.entrySet()) {
            ConfigurationHandler ch = entry.getValue();
            try {
                ch.invokeSetParameters(before.getValuesForComponent(entry.getKey()));
            } catch (Exception ex) {
                dropAllSubmittedChanges();
                throw new BulkSettingException(ex.getMessage(), ex);
            }
        }
        
        ConfigurationView after = getLiveConfigurationView();
        ConfigurationView diff = before.diff(after);
        
        // Check for consistency with the original submitted changes
        for (Map.Entry<ParameterPath, String>  pathVal : diff.getAsParameterPathMap().entrySet()) {
            ParameterPath path = pathVal.getKey();
            if(!submittedChanges.containsPath(path)) {
                dropAllSubmittedChanges();
                throw new BulkSettingException("the parameter " + path 
                        + " was not supposed to change from " + before.getPathValue(path)
                        + " to " + pathVal.getValue());
            } else if (!submittedChanges.getPathValue(path).equals(pathVal.getValue())) {
                dropAllSubmittedChanges();
                throw new BulkSettingException("wrong value for parameter : " 
                        + path.toString()
                        + ", expected : "+submittedChanges.getPathValue(path)
                        + ", actual : " + pathVal.getValue());
            }
        }
        
        dropAllSubmittedChanges();
        after.setConfigurationDescription(categoriesToCommit);
        return after;
    }
        
    public ConfigurationView setSingleParameter(String componentName, String parameterName, Object value) {
        if (!isParameterConfigurable(componentName, parameterName)) {
            log.info("no such parameter " + new ParameterPath(componentName, parameterName));
            return new ConfigurationView();
        }
        // Pushing the existing submitted changes and drop them
        Map<String, Map<String, String>> previousChanges = getAllSubmittedChanges();
        dropAllSubmittedChanges();
        
        submitChange(componentName, parameterName, value);
        ConfigurationView view = commitBulkChange(null);
        
        // popping the exisiting submitted changes
        dropAllSubmittedChanges();
        for (Map.Entry<String, Map<String, String>> entry : previousChanges.entrySet()) {
            String comp = entry.getKey();
            for(Map.Entry<String, String> e : entry.getValue().entrySet()) {
                String parmName = e.getKey();
                String parmVal = e.getValue();
                submitChange(comp, parmName, parmVal);
            }
        }
        return view;
    }
    
    
    
    /**
     * Returns a live map of the values of the configurable parameters for the
     * given component.
     * @param componentName
     * @param categorySet
     * @return 
     */
    public Map<String, String> getCurrentValuesForComponent(String componentName, Set<String> categorySet) {
        return getParameterSet(componentName).getCurrentValues(categorySet)
                .entrySet().stream().collect(Collectors.toMap(entry -> entry.getKey().getParameterName(), 
                        entry -> entry.getValue()));
    }

    public ConfigurationView getLiveConfigurationView()  {
        ConfigurationView res = new ConfigurationView();
        for(String compName : componentParameterHandlers.keySet()) {
            res.putValuesForComponent(compName, getCurrentValuesForComponent(compName, categories));
        }
        return res;
        
    }
    
}
