package org.lsst.ccs.config;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import static org.lsst.ccs.bus.states.ConfigurationState.CONFIGURED;
import static org.lsst.ccs.bus.states.ConfigurationState.DIRTY;
import static org.lsst.ccs.bus.states.ConfigurationState.INITIAL_SAFE;
import org.lsst.ccs.utilities.logging.Logger; 
import org.lsst.ccs.utilities.structs.ParameterPath;

/**
 * A set of several ConfigurationHandlers.
 * @author LSST CCS Team
 */
public class ConfigurationHandlerSet {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.config");
    
    private String descriptionName;
    
    /** 
     * Configurable parameters ordered by the component they belong to. 
     * A LinkedHashMap is used to preserve insertion order, which is the order with which the 
     * configuration handlers have to be traveled when applying configurations / processing bulk changes
     */
    private final Map<String, ConfigurationHandler> componentParameterHandlers = new LinkedHashMap<>();

    /** Configurable parameters ordered by the category they belong to. */
    private final Map<String, CategoryHandler> parametersByCategories = new HashMap<>();
    
    /** This field can be modified when switching from local to remote. */
    private ConfigurationProxy configurationProxy;
    
    /** true until initialization phase is done.*/
    boolean isSafe = true;
    
    /** The current configuration info state. */
    private ConfigurationInfo currentConfigInfo;
    
    ConfigurationHandlerSet() {

    }
    
    // -- Construction phase protected methods 
    
    void addConfigurationHandlerForObject(String componentName, Object component) { 
        ConfigurationHandler handler = ConfigurationHandlerBuilder.buildParameterSetterFromObject(componentName, component);
        
        if (handler == null) return;
        
        componentParameterHandlers.put(componentName, handler);
        for (ConfigurationParameterHandler cph : handler.getConfigurationParameterHandlers()) {
            String category = cph.getCategory();
            CategoryHandler set = parametersByCategories.get(category);
            if (set == null) {
                set = new CategoryHandler(category);
                parametersByCategories.put(category, set);
            }
            set.addParameterHandler(cph);
        }
    }
    
    void setRemoteConfigurationProxy() {
        // check for property
    }
    
    
    boolean isEmpty() {
        return componentParameterHandlers.isEmpty();
    }
        
    void initialize(String descriptionName, Map<Object, Object> safeConfigFromTopNode) {
        this.descriptionName = descriptionName;
        configurationProxy = new LocalConfigurationProxy(descriptionName);
        Map<String,String> safeConfig = new HashMap<>();
        for (String category : getCategorySet()) {
            safeConfig.put(category, ConfigUtils.SAFE_CONFIG_NAME);
        }

        
        Map<String, Properties> safeValues = new HashMap<>();
        // Extracting the configuration parameters from the top node tags
        for (Map.Entry<Object, Object> entry : safeConfigFromTopNode.entrySet()) {
            if (entry.getKey() instanceof String && entry.getValue() instanceof String) {
                try {
                    String value = (String) entry.getValue();
                    ParameterPath path = ParameterPath.valueOf((String) entry.getKey());
                    if (isParameterConfigurable(path.getComponentName(), path.getParameterName())) {
                        Properties props = safeValues.get(path.getComponentName());
                        if (props == null) {
                            props = new Properties();
                            safeValues.put(path.getComponentName(), props);
                        }
                        props.put(path.getParameterName(), value);
                    }
                } catch(IllegalArgumentException ex) {
                    // Ignore silently : the tag may not be a configuration parameter
                }
            }
        }
        
        // Safe configuration values extracted from file overwrite the ones extracted from the top node tags
        Map<String, Properties> safeValuesFromFile = configurationProxy.loadCategories(safeConfig);
        for (Map.Entry<String, Properties> entry : safeValuesFromFile.entrySet()) {
            Properties props = safeValues.get(entry.getKey());
            if (props == null) {
                props = new Properties();
                safeValues.put(entry.getKey(), props);
            }
            props.putAll(entry.getValue());
        }
        
        try {
            doLoadParameters(safeValues, safeConfig);
            for (ConfigurationHandler ch : componentParameterHandlers.values()) {
                for (ConfigurationParameterHandler cph : ch.getConfigurationParameterHandlers()) {
                    cph.setInitialValue();
                }
            }
            isSafe = false;
        } catch(Exception ex) {
            throw new RuntimeException("could not load safe configuration", ex);
        }
    }
    
    // -- End construction phase protected methods.
    
    ConfigurationProxy getConfigurationProxy() {
        return configurationProxy;
    }
    
    private ConfigurationHandler getParameterSet(String componentName) {
        ConfigurationHandler res = componentParameterHandlers.get(componentName);
        if (res == null) {
            throw new IllegalArgumentException("no such component : " + componentName);
        }
        return res;
    }
    
    private CategoryHandler getCategoryHandler(String category) {
        CategoryHandler res = parametersByCategories.get(category);
        if (res == null) {
            log.info("no such category : " + category);
        }
        return res;
    }
    
    /**
     * Gets the name of the components for which configuration parameters have
     * been found.
     * @return a set of component names.
     */
    public Set<String> getComponents() {
        return Collections.unmodifiableSet(componentParameterHandlers.keySet());
    }
    
    public boolean isParameterConfigurable(String componentName, String parameterName) {
        ConfigurationHandler ch = componentParameterHandlers.get(componentName);
        return ch == null ? false : ch.isParameterConfigurable(parameterName);
    }
    
    public Map<String,String> getTaggedCategoriesForCats(Set<String> categories) {
        Map<String,String> res = new HashMap<>();
        for (String category : categories) {
            res.put(category, parametersByCategories.get(category).getTag());
        }
        return res;
    }
    
    /**
     * Builds a ConfigurationInfo object reflecting the configuration state and
     * ready to be sent on the buses.
     * @param configState
     * @param recentChangesPaths
     * @return the {@code ConfigurationInfo} object
     */
    
    private ConfigurationInfo buildConfigurationInfo(ConfigurationInfo before){
        Map<String,String> tags = new HashMap<>();
        Map<String, Boolean> hasCatChanges = new HashMap<>();
        ArrayList<ConfigurationParameterInfo> parametersView = new ArrayList<>();
        List<String> recentChanges = new ArrayList<>();
        
        ConfigurationState realState = CONFIGURED;
        
        for (CategoryHandler ch : parametersByCategories.values()){
            String category = ch.getCategory();
            // set tag for category
            tags.put(category, ch.getTag());
            boolean dirty = ch.isDirty();
            if (dirty) {
                realState = DIRTY;
            }
            hasCatChanges.put(category, dirty);
            for (ConfigurationParameterInfo cpi : ch.getConfigurationParameterInfo()) {
                parametersView.add(cpi);
                if(before != null && !cpi.getCurrentValue().equals(before.getCurrentValueForParameter(cpi.getPathName()))) {
                    recentChanges.add(cpi.getPathName());
                }
            }
        }
        return new ConfigurationInfo(isSafe ? INITIAL_SAFE : realState, descriptionName, tags, hasCatChanges, parametersView, recentChanges);
    }
    
    public String getDescriptionName() {
        return descriptionName;
    }
    
    public Set<String> getCategorySet() {
        return Collections.unmodifiableSet(parametersByCategories.keySet());
    }
    
    // -- Configuration API
    
    public void saveChangesForCategories(Map<String, String> taggedCategories) throws ConfigurationServiceException {
        Map<String, String> toSave = new HashMap<>();
        List<ConfigurationParameterInfo> list = new ArrayList<>();
        for (Map.Entry<String, String> taggedCategory : taggedCategories.entrySet()) {
            String category = taggedCategory.getKey();
            String tag = taggedCategory.getValue();
            CategoryHandler ch = parametersByCategories.get(category);
            toSave.put(category, tag);
            list.addAll(ch.getConfigurationParameterInfo());
        }
        configurationProxy.saveChangesForCategoriesAs(toSave, list);
        for (Map.Entry<String, String> taggedCategory : toSave.entrySet()) {
            parametersByCategories.get(taggedCategory.getKey())
                    .setRunningConfiguration(taggedCategory.getValue());
        }
        currentConfigInfo = buildConfigurationInfo(null);
    }
    
    public void loadCategories(Map<String, String> taggedCategories) {
        Map<String, Properties> changes = configurationProxy.loadCategories(taggedCategories);
        doLoadParameters(changes, taggedCategories);
    }
    
    private void doLoadParameters(Map<String, Properties> changes, Map<String, String> taggedCategories) {
        dropAllSubmittedChanges();
        for (Map.Entry<String, Properties> changesForComponent : changes.entrySet()) {
            String comp = changesForComponent.getKey();
            ConfigurationHandler ch = componentParameterHandlers.get(comp);
            Properties props = changesForComponent.getValue();
            for (String parm : props.stringPropertyNames()) {
                if (isParameterConfigurable(comp, parm)) {
                    ch.submitChange(parm, props.getProperty(parm));
                } else {
                    log.info("ignoring : " + new ParameterPath(comp, parm));
                }
            }
        }
        
        // Submitting the initial value for other components
        // REVIEW: should we do this ?
        for (String category:taggedCategories.keySet()) {
            CategoryHandler ch = getCategoryHandler(category);
            for (ConfigurationParameterHandler cph : ch.getParameters()) {
                String componentName = cph.getComponentName();
                String parameterName = cph.getParameterName();
                if(!(changes.containsKey(componentName) && changes.get(componentName).containsKey(parameterName))) {
                    submitChange(componentName, parameterName, cph.getInitialValue());
                }
            }
        }
        commitBulkChange(taggedCategories);
    }
    
    public void dropChangesForCategories(Set<String> categories) {
        dropAllSubmittedChanges();
        for (String category : categories) {
            CategoryHandler ch = parametersByCategories.get(category);
            for (ConfigurationParameterHandler cph : ch.getParameters()) {
                if (cph.isDirty()) {
                    getParameterSet(cph.getComponentName()).submitChange(cph.getParameterName(), cph.getConfiguredValue());
                }
            }
        }
        
        commitBulkChange(getTaggedCategoriesForCats(categories));

    }
    
    public void dropAllSubmittedChanges() {
        componentParameterHandlers.values()
                .stream().forEach(ConfigurationHandler::dropSubmittedChanges);
    }
    
    public void dropSubmittedChangesForComponent(String name) {
        getParameterSet(name).dropSubmittedChanges();
    }
    
    public Map<String, String> getSubmittedChangesForComponent(String name) {
        return getParameterSet(name).getSubmittedChanges().entrySet()
                .stream().collect(Collectors.toMap(
                        entry -> entry.getKey().getParameterName(), 
                        entry -> entry.getValue()));
    }
    
    public Map<String, Map<String, String>> getAllSubmittedChanges() {
        Map<String, Map<String, String>> res = new TreeMap<>();
        for (Map.Entry<String, ConfigurationHandler> entry : componentParameterHandlers.entrySet()) {
            Map<String, String> submittedChanges = getSubmittedChangesForComponent(entry.getKey());
            if (!submittedChanges.isEmpty()) {
                res.put(entry.getKey(), new TreeMap<String, String>(submittedChanges));
            }
        }
        return res;
    }
    
    public void submitChange(String componentName, String parameterName, Object value) {
        if (!isParameterConfigurable(componentName, parameterName)) {
            log.info("no such parameter : " + componentName+"//"+parameterName + " : ignored");
        } else {
            getParameterSet(componentName).submitChange(parameterName, value);
        }
    }

    /**
     * This commit operation should operate under the control of a configuration lock.
     * @param categoriesToCommit
     */
    public void commitBulkChange(Map<String, String> categoriesToCommit)  {

        ConfigurationInfo before = buildConfigurationInfo(null);
        
        Map<ParameterPath, String> submittedChanges = new HashMap<>();
        
        for (ConfigurationHandler ch : componentParameterHandlers.values()) {
            ch.trimSubmittedChanges();
            submittedChanges.putAll(ch.getSubmittedChanges());
        }
        
        log.fine("processing the following submitted changes :" + submittedChanges);    
        
        // Test against the configuration validation methods
        BulkValidationException excThrown = null;
        try {
            for (ConfigurationHandler ch : componentParameterHandlers.values()) {
                ch.invokeValidateBulkChange(isSafe);
            }
        } catch (Exception ex) {
            excThrown = new BulkValidationException(ex);
        }finally {
            ConfigurationInfo afterValidation = buildConfigurationInfo(before);
            if (!afterValidation.getLatestChanges().isEmpty()) {
                dropAllSubmittedChanges();
                Set<String> suspect = afterValidation.getLatestChanges().stream().map(ConfigurationParameterInfo::getPathName).collect(Collectors.toSet());
                currentConfigInfo = afterValidation;
                throw new BulkSettingException("some parameters have been modified during the validation step : " + suspect, excThrown);
            }
            if (excThrown != null) throw excThrown;
        }
        
        // processing the bulk change
        for (Map.Entry<String, ConfigurationHandler> entry : componentParameterHandlers.entrySet()) {
            ConfigurationHandler ch = entry.getValue();
            try {
                ch.invokeSetParameters(before.getCurrentValuesFor(entry.getKey()));
            } catch (Exception ex) {
                ConfigurationInfo ci = buildConfigurationInfo(before);
                dropAllSubmittedChanges();
                if (!ci.getLatestChanges().isEmpty()) {
                    // The configurationInfo needs to be updated
                    currentConfigInfo = ci;
                }
                throw new BulkSettingException(ex.getMessage(), ex);
            }
        }
        
        ConfigurationInfo after = buildConfigurationInfo(before);
        for (ConfigurationParameterInfo cpi : after.getLatestChanges()) {
            ParameterPath path = ParameterPath.valueOf(cpi.getPathName());
            if(!submittedChanges.containsKey(path)) {
                dropAllSubmittedChanges();
                currentConfigInfo = after;
                throw new BulkSettingException("the parameter " + cpi.getPathName() 
                        + " was not supposed to change from " + before.getCurrentValueForParameter(cpi.getPathName()) 
                        + " to " + cpi.getCurrentValue());
            } else if (!submittedChanges.get(path).equals(cpi.getCurrentValue())) {
                dropAllSubmittedChanges();
                currentConfigInfo = after;
                throw new BulkSettingException("wrong value for parameter : " 
                        + path.toString()
                        + ", expected : "+submittedChanges.get(path)
                        + ", actual : " + cpi.getCurrentValue());
            }
        }
        
        for(Map.Entry<String,String> entry : categoriesToCommit.entrySet()) {
            String category = entry.getKey();
            String tag = entry.getValue();
            getCategoryHandler(category).setRunningConfiguration(tag);
        }
        dropAllSubmittedChanges();
        currentConfigInfo = buildConfigurationInfo(before);
    }
        
    public Set<String> findAvailableConfigurationsForCategory(String category) {
        return configurationProxy.findAvailableConfigurationsForCategory(category);
    }

    public void setSingleParameter(String componentName, String parameterName, Object value) {
        if (!isParameterConfigurable(componentName, parameterName)) {
            log.info("no such parameter " + new ParameterPath(componentName, parameterName));
            return;
        }
        // Pushing the existing submitted changes and drop them
        Map<String, Map<String, String>> previousChanges = getAllSubmittedChanges();
        dropAllSubmittedChanges();
        
        submitChange(componentName, parameterName, value);
        commitBulkChange(Collections.EMPTY_MAP);
        
        // popping the exisiting submitted changes
        dropAllSubmittedChanges();
        for (Map.Entry<String, Map<String, String>> entry : previousChanges.entrySet()) {
            String comp = entry.getKey();
            for(Map.Entry<String, String> e : entry.getValue().entrySet()) {
                String parmName = e.getKey();
                String parmVal = e.getValue();
                submitChange(comp, parmName, parmVal);
            }
        }
    }
    
    public ConfigurationInfo getConfigurationInfo() {
        return currentConfigInfo;
    }
    
    public Map<String, String> getCurrentValuesForComponent(String componentName, Set<String> categorySet) {
        return getParameterSet(componentName).getCurrentValues(categorySet)
                .entrySet().stream().collect(Collectors.toMap(entry -> entry.getKey().getParameterName(), 
                        entry -> entry.getValue()));
    }
    
}
