package org.lsst.ccs.config;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Handles writing and reading of configuration properties files. It builds the
 * configuration file name and then delegates to a {@code WriterProvider}
 *
 * @author LSST CCS Team
 */
class ConfigurationWriterProvider {

    private final WriterProvider wp;
    
    ConfigurationWriterProvider() {
        wp = WriterProvider.getInstance();
    }
    
    public PrintWriter getConfigurationWriter(String descriptionName, String tagName, String categoryName) throws IOException {
        return wp.getPrintWriter(new ConfigurationFileName(descriptionName, tagName, categoryName).getFullName());
    }
    
    public Properties getConfigurationProperties(String descriptionName, String tagName, String categoryName) throws IOException {
        return wp.getProperties(new ConfigurationFileName(descriptionName, tagName, categoryName).getFullName());
    }
    
    public Set<String> findMatchingConfigurations(String descriptionName, String category) {
        String pattern = descriptionName;
        if (category.isEmpty()) {
            pattern+="(_[^_]+?)??";
        } else {
            pattern+="_"+".*?"+"_"+category;
        }
        pattern+="\\.properties";
        Set<String> fullNames =  wp.findMatchingPropertiesFiles(pattern);
        
        return fullNames.stream()
                .map(s -> { return new ConfigurationFileName(descriptionName, s).configName;}
                ).collect(Collectors.toSet());
    }
    
    static class ConfigurationFileName {
        
        /** The description name. */
        private final String tagName;
        
        /** The configuration name. */
        private final String configName;
        
        /** the category name. */
        private final String catName;
        
        /** The full configuration file name. */
        private final String fullName;
        
        ConfigurationFileName(String tagName, String configName, String categoryName) {
            this.tagName = tagName;
            this.configName = configName;
            this.catName = categoryName;
                String baseName = tagName;
            if (configName.isEmpty()){
                if (!catName.isEmpty())
                    baseName +="__"+catName;
            } else {
                baseName += "_" + configName;
                if (!catName.isEmpty())
                    baseName +="_"+catName;
            }
            this.fullName = baseName + ".properties";
        }
        
        ConfigurationFileName(String tagName, String fullName) {
            this.tagName = tagName;
            String trimmed = fullName.replace(".properties", "").replace(this.tagName,"");
            String[] split = trimmed.split("_");
            switch (split.length) {
                case 1:
                    configName = "";
                    catName = "";
                    break;
                case 2:
                    configName = split[1];
                    catName = "";
                    break;
                case 3:
                    configName = split[1];
                    catName = split[2];
                    break;                        
                default:
                    throw new IllegalArgumentException("file name is not a configuration file name : " + fullName);
            }
            this.fullName = fullName;
        }
        
        public String getFullName() {
            return fullName;
        }
    }
    

    
}
