package org.lsst.ccs.config;

import java.util.Map;

/**
 * An interface for classes that can internally handle configuration bulk changes.
 * The two configuration steps covered by this interface are:
 *  - parameter bulk validation
 *  - parameter bulk setting
 * 
 * <p>The bulk change process is started when invoking commitBulkChange on the subsystem,
 * and consists in traveling the tree of components twice : a first time to validate the
 * submitted changes and a second time to set the submitted changes. This interface offers
 * two methods for implementing those two steps for a given component.
 * 
 * <p>The bulk validation step is provided with the full view of the component parameters
 * (that is, the current values of each parameter merged with the submitted changes).
 * This view is a Map of parameter name, parameter value where the value is either the
 * new proposed value if it has been changed, or the current value of the parameter
 * if it has not been changed.
 * 
 * <p>The bulk validation step is intended to validate that all (or a subset) of the
 * provided parameters full view is in a consistent state. If problems are detected
 * an exception is meant to be thrown. If an exception is thrown, the bulk change
 * process is interrupted.
 * 
 * <p>The bulk parameter setting method is invoked after a successful validation step
 * with the map of parameters that are still left to be set. Other components might
 * have been already had a chance to set parameters while the component tree
 * was being traversed.
 * 
 * <p>During the bulk parameter setting, a component is responsible to set only
 * the parameters it is designed to handle. The framework will be responsible to
 * understand which parameters have been set using introspection. The parameters
 * that have not been set will be passed on to the next component responsible for
 * setting configuration parameters. Any parameters that are still left to be set
 * at the end of the tree navigation will be set by the framework directly on
 * the corresponding fields using introspection.
 * 
 * @author The LSST CCS Team
 */
public interface ConfigurationBulkChangeHandler {
    
    /**
     * Override with more advanced parameter validation checking.
     * This method is meant to throw an exception when one or more parameters have 
     * a proposed value that is decided as not valid by the user.
     * @param parametersView the whole set of parameters for this component with
     * their value to be validated. it maps the parameter names to their value.
     */
    public default void validateBulkChange(Map<String, Object> parametersView) {
        
    }
    
    /**
     * Override with by hand setting of the parameters in parametersView
     * @param parametersView a map of parameter names with the new value for this 
     * parameter
     * @return the subset of parameters of the input map that are not handled by
     * this method.
     * @deprecated use setParameterBulk instead
     */
    @Deprecated
    public default Map<String, Object> setBulkParameter(Map<String, Object> parametersView) {
        return parametersView;
    }
    
    /**
     * Override with by hand setting of the parameters in parametersView.
     * @param parametersView a map of parameter names with the new value for this 
     * parameter
     */
    public default void setParameterBulk(Map<String, Object> parametersView) {
        
    }
    
    
}
