package org.lsst.ccs.config;


import java.io.Serializable;

class ParameterPath implements Serializable {
    
    private static final long serialVersionUID = -8374146985540286600L;
    
    /**
     * unique name of Module/component in subsystem context
     */
    final String componentName;
    /**
     *  constructor or method. null or empty value means constructor
     */
    final String codeName ;
    /**
     * name of parameter (may be the number of a positionnal parameter starting at 0)
     */
    final String parameterName ;
    
    //////////////////////////////////// CONSTRUCTORS

    /**
     * "real" constructor
     * @param componentName should not be null
     * @param codeName null of empty means constructor
     * @param parameterName should not be null
     */
    ParameterPath(String componentName, String codeName, String parameterName) {
        if(componentName == null || parameterName == null) {
            throw new IllegalArgumentException("null argument in ParameterPath");
        }
        if(codeName == null) {
            codeName = "" ;
        }
        this.componentName = componentName;
        this.codeName = codeName;
        this.parameterName = parameterName;
    }
    
    ////////////////////////////////// ACCESSORS/MUTATORS

    public String getComponentName() {
        return componentName;
    }

    public String getCodeName() {
        return codeName;
    }

    public String getParameterName() {
        return parameterName;
    }

    ////////////////////////////// IDENT METHODS

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ParameterPath that = (ParameterPath) o;

        if (codeName != null ? !codeName.equals(that.codeName) : that.codeName != null) return false;
        if (!componentName.equals(that.componentName)) return false;
        if (!parameterName.equals(that.parameterName)) return false;

        return true;
    }


    @Override
    public int hashCode() {
        int result = componentName.hashCode();
        result = 31 * result + (codeName != null ? codeName.hashCode() : 0);
        result = 31 * result + parameterName.hashCode();
        return result;
    }
    
    @Override
    public String toString() {
        return this.componentName + '/'+ this.codeName + '/'+this.parameterName ;
    }

    /**
     * reverse operation from toString(): creates a ParameterPath from a String.
     * @param pathString should be of the form "componentName/codeName/parameterName" (example: "carousel//tickMillis")
     * @return a ParameterPath out of the given path String
     * @throws  IllegalArgumentException if format is not correct
     */
    public static ParameterPath valueOf(String pathString) {
        ParameterPath path ;
        String[] elements = pathString.split("/") ;
        if(elements.length != 3) {
            throw new IllegalArgumentException("PathString should be componentName/codeName/parameterName");
        }
        path = new ParameterPath(elements[0], elements[1], elements[2]) ;
        return path ;
    }
}