package org.lsst.ccs.config;

import java.io.IOException;
import java.io.PrintWriter;
import org.lsst.ccs.utilities.logging.Logger;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import static org.lsst.ccs.config.ConfigUtils.DEFAULT_CONFIG_NAME;
import static org.lsst.ccs.config.ConfigUtils.SAFE_CONFIG_NAME;

/**
 * Configuration proxy that registers configurations locally as properties files.
 * @author bamade
 */
// Date: 04/10/12

class LocalConfigurationProxy implements ConfigurationProxy {
    
    public static final Comparator<ConfigurationParameterInfo> configurationParameterComparator = new Comparator<ConfigurationParameterInfo>() {

        @Override
        public int compare(ConfigurationParameterInfo o1, ConfigurationParameterInfo o2) {
            String componentName = o1.getComponentName();
            String otherComponentName = o2.getComponentName();
            
            if(componentName.equals("main") && ! otherComponentName.equals("main")) {
                return 1 ;
            }
            if(otherComponentName.equals("main") && ! otherComponentName.equals("main/")) {
                return -1 ;
            }
            if (componentName.equals(otherComponentName)) {
                return o1.getParameterName().compareTo(o2.getParameterName());
            } else {
                return componentName.compareTo(otherComponentName);
            }
        }
    };
    
    static WriterProvider writerProvider;
    
    protected static WriterProvider getWriterProvider() { 
        if (writerProvider == null) {
            if ( System.getProperty("org.lsst.ccs.config.WriterProvier", "").equals("org.lsst.ccs.config.InMemoryWriterProvider") ) {
                writerProvider = new InMemoryWriterProvider();                            
            } else {
                writerProvider = new FileWriterProvider();            
            }
        }
        return writerProvider;
    }

    protected static void setWriterProvider(WriterProvider wp) {
        writerProvider = wp;
    }
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.config");
    
    /** the description name. */
    private final String descriptionName;
    
    Logger logger = Logger.getLogger("org.lsst.ccs.config");
    
    /**
     * Builds a configuration proxy for the subsystem described by {@code subsystemDesc}.
     * @param descriptionName
     */
    LocalConfigurationProxy(String descriptionName) {
        this.descriptionName = descriptionName;
        writerProvider = getWriterProvider();
    }
    
    /**
     * Tagged categories are saved under a new tag name.
     * Other parameters are left unchanged, the configuration context remains
     * active.
     * @param taggedCategories
     */
    @Override
    public synchronized void saveChangesForCategoriesAs(Map<String, String> taggedCategories, List<ConfigurationParameterInfo> values) throws ConfigurationServiceException {
        Map<String, List<ConfigurationParameterInfo>> valuesPerCategory = ConfigurationInfo.getParameterInfoGroupByCategory(values);
        for (Map.Entry<String, String> entry : taggedCategories.entrySet()) {
            PrintWriter writer;
            try {
                writer = writerProvider.getPrintWriter(entry.getValue(), descriptionName, entry.getKey());
            } catch(IOException ex) {
                throw new ConfigurationServiceException("could not open file", ex);
            }
            Set<ConfigurationParameterInfo> modifiedParameters = new TreeSet<ConfigurationParameterInfo>(configurationParameterComparator);
            
            for (ConfigurationParameterInfo cpi : valuesPerCategory.get(entry.getKey())) {
                if (!cpi.isFinal()) {
                    modifiedParameters.add(cpi);
                }
            }            

            for (ConfigurationParameterInfo parameter : modifiedParameters){
                writer.println(toPropertyString(parameter, false));
            }
            writer.flush();
            writer.close();
        }
    }
    
    /**
     * creates a String to be included in a .properties file
     * @param parmInfo
     * @param commentOutValue if true the description is commented out
     * @return a String representation of the parameter description with the 
     * given value
     */
    public static String toPropertyString(ConfigurationParameterInfo parmInfo, boolean commentOutValue) {
        String currentValue = parmInfo.getCurrentValue();
        if(currentValue == null) {currentValue ="" ;}
        StringBuilder builder = new StringBuilder() ;
        String pathName = parmInfo.getPathName();
        builder.append("#********  ").append(pathName) ;
        String description = parmInfo.getDescription();
        if(description != null) {
            String withPounds = description.replaceAll("\n","\n#") ;
            builder.append('\n').append("#** ").append(withPounds) ;
        }
        builder.append("\n#** type : " )
                    .append(getTypeInfo( parmInfo.getType())) ;
        if(parmInfo.getCategoryName()!= null && !"".equals(parmInfo.getCategoryName().trim())) {
            builder.append("\n#** category : ").append(parmInfo.getCategoryName());
        }
        String propName = parmInfo.getPathName();
        if( propName== null || "".equals(propName.trim())) {
            propName = pathName ;
        } else {
            propName = propName.trim().replaceAll(" ", "\\\\ ");
        }
        builder.append("\n#********\n").append(commentOutValue?"#":"").append(propName).append(" = ").append(currentValue).append('\n') ;

        return builder.toString() ;
    }
    
    private static String getTypeInfo(String typeName) {
        String res = typeName ;
        for(String[] keyVal : TABLE) {
            if(keyVal[0].equals(typeName)){
                return keyVal[1] ;
            }
        }
        return res ;
    }
     
     private static final String[][] TABLE = {
            {"java.lang.Integer", "integer (example : 6)"}      ,
            {"java.lang.String", "String"}      ,
            {"java.lang.Double", "double (example : 3.0)"}      ,
            {"java.lang.Float", "float (example : 3.0)"}      ,
            {"java.lang.Boolean", "boolean (example : true)"}      ,
            {"[I", "array of integers (example : [1,34,666] )"}      ,
            {"[D", "array of doubles (example : [1.0,34.0,666.66] )"}      ,
            {"[F", "array of floats (example : [1.0,34.0,666.66] )"}      ,
            {"[Ljava.lang.String;", "array of Strings (example : ['hello', 'world'] )"}
    } ;
 
    @Override
    public Map<String, Properties> loadCategories(Map<String,String> taggedCategories) throws ConfigurationServiceException {
        if (taggedCategories.isEmpty()) return Collections.EMPTY_MAP;
        Map<String, Properties> res = new HashMap<>();
        for(Map.Entry<String,String> entry : taggedCategories.entrySet()){
            String cat = entry.getKey();
            String tag = entry.getValue();
            try {
                Properties configProps = writerProvider.getConfigurationProperties(tag, descriptionName, cat);
                if (configProps == null) {
                    if (!tag.equals(DEFAULT_CONFIG_NAME) && !tag.equals(SAFE_CONFIG_NAME))
                        throw new IllegalArgumentException("could not find configuration file " + tag + " for category " + cat);
                } else {
                    for (Map.Entry<Object, Object> prop : configProps.entrySet()) {
                        String name = (String)prop.getKey();
                        String val = (String)prop.getValue();
                        String componentName = name.split("/")[0];
                        String parameterName = name.split("/")[2];
                        Properties props = res.get(componentName);
                        if (props == null) {
                            props=new Properties();
                            res.put(componentName, props);
                        }
                        props.setProperty(parameterName, val);
                    }
                }
                
            } catch (IOException ex) {
                throw new ConfigurationServiceException(ex.getMessage(), ex);
            }
        }
        return res;    
    }
    
    @Override
    public Set<String> findAvailableConfigurationsForCategory(String category) {
        return writerProvider.findAvailableConfigurationsForCategory(descriptionName, category);
    }

    @Override
    public boolean isAvailable() {
        return true;
    }
}
