package org.lsst.ccs.config;

import javax.persistence.*;
import java.io.Serializable;
import java.util.*;

/**
 * Description for a subsystem as saved in database.
 * The "subsystemName" and "tags" are keys (there is only one active subsystem with that name and tag, others
 * are in historical data).
 * <BR/>
 * These objects  contain :
 * <UL>
 *     <LI/> general informations : name, tag, user that created the description, version ...
 *     <LI/> an object (saved in database as a Lob) which contains a copy of the DescriptiveNode tree
 *     <LI/> a Set of <TT>ParameterDescriptions</TT>: objects that describe a parameter (see
 *     <TT>ParameterBase</TT> (<TT>ParameterPath</TT> + type and default value) with infos such as
 *     description, constraints, static nature ....
 * </UL>
 * <P/>
 * A <TT>ConfigProfile</TT> object will reference such a description but references actual values
 * for modifiable parameters.
 */
@MappedSuperclass
public abstract class SubsystemDescription implements Serializable {

    /**
	 * 
	 */
	private static final long serialVersionUID = 1931407188397328132L;
	/**
     * first time the description was valid : initially populated by the database
     * (but copies can get it from an original: it depends on the purpose of the copy )
     */
    private long startTimestamp;//generated
    /**
     * valid limit. defaults to eternity except when the object is pushed in history.
     * TODO: put in Ghost and make class immutable
     */
    private long endTimestamp = PackCst.STILL_VALID;
    /**
     * name of subsystem
     */
    private /*@NonNull*/ String subsystemName;
    /**
     * tag such as 'high wind'
     */
    private /*@NonNull*/ String tag;
    /**
     * Big object! see DescriptionType enum ...
     * but not necessarily that big (so simply serializable Basic could be ok)
     */
    @Lob
    protected /*@NonNull*/ Serializable descriptionData;

    /**
     * a link to the description the current object may replace
     */
    private long previousDescriptionID;
    
    
    ///////////////////////// CONSTRUCTORS


    /**
     * for bean  convention: not public!
     */
    SubsystemDescription() {
    }

    /**
     * used by subclasses
     *
     * @param subsystemName     should not be null or empty
     * @param tag               may be null or empty
     * @param descriptionData see DataFlavour documentation
     */
    protected SubsystemDescription(String subsystemName, String tag, Serializable descriptionData) {
        setSubsystemName(subsystemName);
        setTag(tag);
        //TODO: precondition configuration data should not be null
        if (descriptionData == null) {
            throw new IllegalArgumentException("null configuration data");
        }
        this.descriptionData = descriptionData;
    }
    //detach method ? copy constructor

    /**
     * creates a copy of the SubsystemDescription that is not (yet) registered to the database.
     * Subclasses <B>must</B> fill the corresponding paramDescriptions set.
     *
     * @param other
     */
    protected SubsystemDescription(SubsystemDescription other) {
        this(other.getSubsystemName(), other.getTag(), 
                other.getDescriptionData());
    }

    ///////////////////////////// ACCESSORS/MUTATORS

    /**
     * the technical id: zero if the object is not yet registered in database
     *
     * @return the id associated with the subsystem description, as given by the
     * underlying configuration service.
     */
    public abstract long getId();

    /**
     * used only by reconstruction code
     *
     * @param id
     */
    abstract void setId(long id);

    /**
     * Detailed description of parameters that can be changed
     *
     * @return a Map of parameter descriptions by their path String
     */
    public abstract Map<ParameterPath, ? extends ParameterDescription> getParamDescriptionMap();
    
    public abstract Set<String> getCategorySet();
    
    /**
     * tells if the modifying methods can be invoked on a newly created objects.
     *
     * @return false if the object has been already registered in the database
     */
    public boolean isReadOnly() {
        return getId() != 0L;
    }

    public String getSubsystemName() {
        return subsystemName;
    }
    
    void setSubsystemName(String subsystemName) {
        if (subsystemName == null) {
            throw new IllegalArgumentException("null subsystemName");
        }
        this.subsystemName = subsystemName;
    }

    public String getTag() {
        return tag;
    }

    void setTag(String tag) {
        if (tag == null) tag = "";
        this.tag = tag;
    }

    public long getStartTimestamp() {
        return startTimestamp;
    }

    void setStartTimestamp(long startTimestamp) {
        this.startTimestamp = startTimestamp;
    }

    public long getEndTimestamp() {
        return endTimestamp;
    }

    void setEndTimestamp(long endTimestamp) {
        this.endTimestamp = endTimestamp;
    }

    public Serializable getDescriptionData() {
        return descriptionData;
    }

    void setDescriptionData(Serializable descriptionData) {
        this.descriptionData = descriptionData;
    }

    /**
     * get the id of the previous subsystemDescription with same Name and tag.
     * This data is modified by the configuration facade (when replacing a subsystemDescription)
     *
     * @return 0L if there is none
     */
    public long getPreviousDescriptionID() {
        return previousDescriptionID;
    }

    void setPreviousDescriptionID(long previousDescriptionID) {
        this.previousDescriptionID = previousDescriptionID;
    }

    /* problems with mapping on superClass
    public Set<ParameterDescription> getParamDescriptions() {
        return paramDescriptions;
    }

     void setParamDescriptions(Set<ParameterDescription> paramDescriptions) {
        this.paramDescriptions = paramDescriptions;
    }
    */

    ///////////////////////////////// IDENT METHODS


    /**
     * compares only the name and tag of subsystems not their content!
     * @param o
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof SubsystemDescription)) return false;

        SubsystemDescription that = (SubsystemDescription) o;

        if (getId() != that.getId()) return false;
        if (!getSubsystemName().equals(that.getSubsystemName())) return false;
        String tag = getTag();
        if (tag != null ? !tag.equals(that.getTag()) : that.getTag() != null) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = getSubsystemName().hashCode();
        String tag = getTag();
        result = 31 * result + (tag != null ? tag.hashCode() : 0);
        long id = getId();
        result = 31 * result + (int) (id ^ (id >>> 32));
        return result;
    }

    @Override
    public String toString() {
        return "{" +
                "id=" + getId() +
                "; name/tag=" + subsystemName +"/"+tag +
                ";descriptions=" + this.getParamDescriptionMap() +
                '}';
    }
//////////////////////////////  METHODS


    /**
     * Looks for a parameter description that matches the given path
     *
     * @param path
     * @return the parameter description that matches the given path
     */
    public ParameterDescription fetch(PathObject path) {
        return getParamDescriptionMap().get(path.getPath());
    }
    
    /**
     * Looks for a parameter description that matches the given path string
     * @param pathString
     * @return the parameter description that matches the given path String
     */
    public ParameterDescription fetch(String pathString) {
        return fetch(ParameterPath.valueOf(pathString));
    }

    /**
     * adds a list of parameter descriptions
     *
     * @param descriptions
     * @throws ImmutableStateException if called on an immutable object
     */
    public abstract void addParameterDescriptions(ParameterDescription... descriptions);

    public abstract void addParameterDescriptions(Collection<ParameterDescription> descriptions);
    
    /**
     * removes a list of parameter descriptions
     *
     * @param descriptions
     * @throws ImmutableStateException if called on an immutable object
     */
    public abstract void removeParameterDescriptions(ParameterDescription... descriptions);
    
}

