package org.lsst.ccs.config;

import javax.persistence.MappedSuperclass;
import java.io.Serializable;

/**
 * The base class for all Parameter descriptions
 */
@MappedSuperclass
public abstract class ParameterDescription implements Serializable, PathObject {

    /**
	 * 
	 */
	private static final long serialVersionUID = 5660908759518934491L;
	/**
     * possible details. Immutable in Ghost data
     */
    protected /*@Nullable*/ String description ;
    private  /*@NotNull*/ ParameterBase parameterBase ;
    /**
     * a user-friendly name: it's up to the config "designer" to make sure it's a unique name
     * otherwise the "pathName" of the parameterBase will be used. Immutable in ghost Data
     */
    //TODO: derived field! Formula is too cumbersome! see hack in code
    private /*@NotNull*/ String simpleName ;
    
    private String category;

    //TODO:  startDate, endDate, user, tag are normally part of SubsystemDescription do we duplicate?
    //TODO: harmonize with level in Commands
    /**
     * used for configuration "rights"; a designer may have
     * the right to modify a parameter and an end-user not.
     * when automatically generated from a subsytemdescription
     * the level of all modifiable parameters is set to 10,
     * a parameter which is modifiable by an end-user is  less than 5.
     * Immutable in Ghost Data.
     *
     */
    private int level = 0;

    ////////////////////////////// CONSTRUCTORS

    protected ParameterDescription() {

    }

    /**
     * builds a default Description out of a ParameterBase which is described in the original setup.
     * @param parameterBase
     */
    protected ParameterDescription(ParameterBase parameterBase) {
        this.setParameterBase( parameterBase) ;
    }

    /**
     * copy constructor.
     * Beware: the id is copied so do not use to make a new Parameterdescription that is to be registered later.
     * @param other
     */
    protected ParameterDescription(ParameterDescription other) {
        //this(other.getParameterBase()); no will not work!
        this.setId(other.getId());
        this.setParameterBase( other.parameterBase) ;
        this.description = other.description ;
        this.simpleName = other.simpleName;
        this.level = other.level ;
        this.category = other.category;
    }


    //////////////////////////// ACCESSORS/MUTATORS

    public abstract long getId() ;

    protected abstract void setId(long id) ;

    public boolean isReadOnly() {
        return this.getId() != 0L ;
    }


    public ParameterBase getParameterBase() {
        return parameterBase;
    }

    protected void setParameterBase(ParameterBase parameterBase) {
        this.parameterBase = parameterBase;
        String codeName = parameterBase.getCodeName() ;
        if(codeName == null) codeName = "" ;
        this.simpleName = String.format("%s/%s/%s",
                parameterBase.getComponentName(),
                codeName,
                parameterBase.getParameterName()) ;
    }

    public String getDescription() {
        return description;
    }

    /**
     * this data is not essential to the logic so it is modifiable anyway.
     * @param description
     */
    public  void setDescription(String description) {
        this.description = description ;
    }

    public String getSimpleName() {
        return simpleName;
    }

    public void setSimpleName(String simpleName) {
        if(isReadOnly()) {
            throw new ImmutableStateException("simple name read Only") ;
        }
        // this is a hack : because simpleName is a derived field
        if(this.simpleName == null || "".equals(this.simpleName)) {
            this.simpleName = simpleName;
        }
    }
    
    public String getCategory(){
        return category;
    }
    
    public void setCategory(String category){
        if(isReadOnly()) {
            throw new ImmutableStateException("Category read Only") ;
        }
        this.category = category;
    }

    public int getLevel() {
        return level;
    }

    public void setLevel(int level) {
        if(isReadOnly()) {
            throw new ImmutableStateException("Level read Only") ;
        }
        this.level = level;
    }

    //////////////// IDENT METHODS
    // TODO: inclusion of id?

    /**
     * equals is special: it works with any Path Object!
     * @param o
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof PathObject)) return false;

        PathObject that = (PathObject) o;

        if (!parameterBase.getPath().equals(that.getPath())) return false;

        return true;
    }

    @Override
    public int hashCode() {
        return parameterBase.getPath().hashCode();
    }

    @Override
    public String toString() {
        return "{" + getId() +
                ": base=" + parameterBase +
                '}';
    }


    public String getComponentName() {
        return parameterBase.getComponentName();
    }

    public String getCodeName() {
        return parameterBase.getCodeName();
    }

    public String getParameterName() {
        return parameterBase.getParameterName();
    }

    public String getDefaultValue() {
        return parameterBase.getDefaultValue() ;
    }

    @Override
    public ParameterPath getPath() {
        return parameterBase.getPath();
    }

    public String getTypeName() {
        return parameterBase.getTypeName();
    }

    /**
     * creates a String to be included in a .properties file
     * @param value default value to be included in the text
     * @param commentOutValue if true the description is commented out
     * @return a String representation of the parameter description with the 
     * given value
     */
    public String toPropertyString(String value, boolean commentOutValue) {
        if(value == null) {value ="" ;}
        StringBuilder builder = new StringBuilder() ;
        String pathName = parameterBase.getPath().toString() ;
        builder.append("#********  ").append(pathName) ;
        if(description != null) {
            String withPounds = description.replaceAll("\n","\n#") ;
            builder.append('\n').append("#** ").append(withPounds) ;
        }
        builder.append("\n#** type : " )
                    .append(TypeInfos.get( parameterBase.getTypeName())) ;
        if(category!= null && !"".equals(category.trim())) {
            builder.append("\n#** category : ").append(category);
        }
        String propName = this.simpleName;
        if( propName== null || "".equals(propName.trim())) {
            propName = pathName ;
        } else {
            propName = propName.trim().replaceAll(" ", "\\\\ ");
        }
        builder.append("\n#********\n").append(commentOutValue?"#":"").append(propName).append(" = ").append(value).append('\n') ;

        //System.out.println(" -> -> " +builder );
        return builder.toString() ;
    }

    /**
     * default value is included in the text that describes the parameter as a property
     * @return a String representation of the parameter description associated with
     * the defautl value of this parameter
     */
    public String toPropertyString() {
        return toPropertyString(getDefaultValue(), true) ;
    }
}
