package org.lsst.ccs.config;

import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.config.utilities.ConfigUtils;
import org.lsst.ccs.framework.ConfigurationServiceException;
import org.lsst.ccs.framework.annotations.ParameterSetter;
import org.lsst.ccs.utilities.conv.TypeUtils;

/**
 * Abstract subclass of subsystems which provides access to the configuration proxy.
 * @author The LSST CCS Team
 */
public abstract class ConfigurableSubsystem extends Subsystem {
    
    /** Provides access to the configuration service. */
    protected ConfigurationProxy configurationProxy;
    
    /** The set of configuration categories the parameters of the subsystem are split into. */
    private final Set<String> categories;

    /** Used only to know the startup state. */
    private final ConfigurationState initialState;
    
    /** The dictionary of config changer methods for each component. */
    protected final Map<String, ParameterSetter> parameterSetterDictionary;
    
    public ConfigurableSubsystem(String name, ConfigurationProxy configProxy,
            ConfigurationState initialState, Map<String, ParameterSetter> parameterSetterDictionary) {
        super(name, AgentInfo.AgentType.WORKER);
        this.configurationProxy = configProxy;
        categories = configProxy.getCategorySet();
        this.initialState = initialState;
        this.parameterSetterDictionary = parameterSetterDictionary;
        //configProxy.writeMissingDefaultConfigs();
    }
    
    /**
     * @return tag name
     */
    public String getTag() {
        return configurationProxy.getTagName();
    }

    /**
     * @return the configuration proxy
     */
    public ConfigurationProxy getConfigurationProxy() {
        return configurationProxy;
    }

    /**
     * this method should not be used once the proxy is delegated to all components
     * so it should not be public!
     * @param configurationProxy
     */
    protected void setConfigurationProxy(ConfigurationProxy configurationProxy) {
        this.configurationProxy = configurationProxy;
    }

    /**
     * Saves all changes in their current configuration.
     * State switches to Configured is no exception.
     * @throws ConfigurationServiceException if configuration service unavailable
     */
    @Command(description = "Saves all changes in the current configurations", type = Command.CommandType.CONFIGURATION)
    public final void saveAllChanges() throws ConfigurationServiceException {
        saveChangesForCategoriesAsInternal(configurationProxy.getTaggedCategoriesForCats(categories));
    }

    /**
     * Changes made in the specified categories are saved under the current configuration
     * for this category, changes on parameters that belong to other categories
     * are left unchanged.
     * @param categories a list of categories
     * @throws ConfigurationServiceException if configuration service
     * unavailable.
     */
    @Command(description = "Saves the specified categories with a name", type = Command.CommandType.CONFIGURATION)
    public final void saveChangesForCategories(
            @Argument(name = "taggedCategories", description = "A list of categories") String... categories) {
        saveChangesForCategoriesAsInternal(configurationProxy.getTaggedCategoriesForCats(ConfigUtils.parseCategories(this.categories, categories)));
    }

    /**
     * Changes made in the specified categories are saved under the newly specified
     * name for this category, changes on parameters that belong to other categories
     * are left unchanged.
     * @param taggedCategories a list of pairs categoryName:configurationName
     * @throws ConfigurationServiceException if the configuration service is not available.
     */
    @Command(description = "Saves the specified categories with a name", type = Command.CommandType.CONFIGURATION)
    public final void saveChangesForCategoriesAs(
            @Argument(name = "taggedCategories", description = "A list of pairs categoryName:configurationName") String... taggedCategories) {
        saveChangesForCategoriesAsInternal(parseConfigurationString(taggedCategories));
    }

    private void saveChangesForCategoriesAsInternal(Map<String, String> taggedCategories) {
        if (taggedCategories.isEmpty()) return;
        String message = "";
        ConfigurationServiceException exc = null;
        try {
            this.configurationProxy.saveChangesForCategoriesAs(taggedCategories);
        } catch (ConfigurationServiceException ex) {
            exc = ex;
            message = ex.getMessage();
        }
        // 
        updateStateAndSendStatusConfigurationInfo(message, Collections.emptySet());
        if (exc != null) throw exc;
    }

    /**
     * All unsaved changes are dropped, each parameter goes back to the value specified
     * by the current configuration for this category.
     */
    @Command(description = "drop all unsaved changes", type = Command.CommandType.CONFIGURATION)
    public void dropAllChanges() {
        dropChangesInternal(categories);
    }

    /**
     * Changes occurring for one of the mentioned categories in categories are dropped,
     * other categories are left untouched.
     * @param categories A list of categories
     */
    @Command(description = "drop unsaved changes for the specified categories", type = Command.CommandType.CONFIGURATION)
    public void dropChangesForCategories(@Argument(name = "categories", description = "A list of categories") String... categories) {
        dropChangesInternal(ConfigUtils.parseCategories(this.categories, categories));
    }

    private void dropChangesInternal(Set<String> categoriesSet) {
        dropAllSubmittedChanges();
        Map<String, Properties> changes;
        changes = configurationProxy.loadCategories(configurationProxy.getTaggedCategoriesForCats(categoriesSet));
        
        for (Map.Entry<String, Properties> changesForComponent : changes.entrySet()) {
            for (String parameterName : changesForComponent.getValue().stringPropertyNames()) {
                submitChange(changesForComponent.getKey(), parameterName, changesForComponent.getValue().getProperty(parameterName));
            }
        }
        // validation step should be safe in this context
        validateBulkChanges();
        // setting
        Set<ParameterPath> processedParms = bulkChange();
        configurationProxy.saveModifications(configurationProxy.getTaggedCategoriesForCats(categoriesSet));
        updateStateAndSendStatusConfigurationInfo("", processedParms);
    }

    /**
     * Loads the configuration specified for each category specified in configName
     * and loads the default configuration for the other categories.
     * Changes that are not saved yet are dropped.
     *
     * @param taggedCategories a list of pairs categoryName:configurationName
     * @throws org.lsst.ccs.framework.ConfigurationServiceException
     * @throws org.lsst.ccs.utilities.conv.TypeConversionException
     */
    @Command(description = "loads a new configuration", type = Command.CommandType.CONFIGURATION)
    public void loadConfiguration(
            @Argument(name = "taggedCategories", description = "a list of pairs categoryName:configurationName") String... taggedCategories) {
        loadCategoriesInternal(ConfigUtils.parseConfigurationStringWithDefaults(categories, taggedCategories));
    }

    /**
     * Loads the configuration specified for each category, parameters that belong
     * to other categories are left unchanged.
     * @param taggedCategories a list of pairs categoryName:configurationName
     * @throws org.lsst.ccs.framework.ConfigurationServiceException
     */
    @Command(description = "loads the configuration for the specified categories", type = Command.CommandType.CONFIGURATION)
    public void loadCategories(
            @Argument(name = "taggedCategories", description = "a list of pairs categoryName:configurationName") String... taggedCategories) {
        if (taggedCategories.length == 0) {
            return;
        }
        loadCategoriesInternal(parseConfigurationString(taggedCategories));
    }

    private void loadCategoriesInternal(Map<String, String> categoryProfile) {
        dropAllSubmittedChanges();
        Map<String, Properties> changes;
        changes = configurationProxy.loadCategories(categoryProfile);

        for (Map.Entry<String, Properties> changesForComponent : changes.entrySet()) {
            for (String parameterName : changesForComponent.getValue().stringPropertyNames()) {
                submitChange(changesForComponent.getKey(), parameterName, changesForComponent.getValue().getProperty(parameterName));
            }
        }
        // validation step should be safe in this context
        validateBulkChanges();
        // setting
        Set<ParameterPath> processedParms = bulkChange();
        
        configurationProxy.saveModifications(categoryProfile);
        
        updateStateAndSendStatusConfigurationInfo("", processedParms);
    }

    /**
     * @return a ConfigurationInfo object depicting the current configuration state 
     */
    @Command(description = "return a ConfigurationInfo object", type = Command.CommandType.CONFIGURATION)
    public ConfigurationInfo getConfigurationInfo() {
        return configurationProxy.buildConfigurationInfo((ConfigurationState) getState(ConfigurationState.class), Collections.emptySet());
    }
    
    private Map<String,String> parseConfigurationString(String ... taggedCategories){
        return ConfigUtils.parseConfigurationString(categories, taggedCategories);
    }

    /**
     * Single change of parameter.
     * The value is validated and the parameter is immediately set to this value,
     * without interfering with the current set of submitted changes.
     *
     * @param componentName the name of the component the parameter belongs to.
     * @param parameterName the name of the parameter.
     * @param value the new value to affect to this parameter
     */
    @Command(description = "Submits a single change to be processed immediately", type = Command.CommandType.CONFIGURATION)
    public void change(String componentName, String parameterName, Object value) {
        if (!configurationProxy.isParameterConfigurable(componentName, parameterName)) {
            throw new IllegalArgumentException("no such parameter : " + componentName+"//"+parameterName);
        }
        ParameterSetter parmSetter = parameterSetterDictionary.get(componentName);
        updateInternalState("", ConfigurationState.CONFIGURING);
        
        parmSetter.invokeSetSingleParameter(parameterName, value, configurationProxy.getCurrentValuesForComponent(componentName, Collections.emptySet()), getState().isInState(PhaseState.OPERATIONAL));
        configurationProxy.notifyParameterChange(componentName, parameterName, TypeUtils.stringify(value));
        Set<ParameterPath> processedParms = new HashSet<>();
        processedParms.add(new ParameterPath(componentName, "", parameterName));
        updateStateAndSendStatusConfigurationInfo("", processedParms);
    }

    /**
     * Stores a change parameter to be validated later.
     *
     * @param componentName
     * @param parameterName
     * @param value
     * @throws IllegalArgumentException if <code>componentName</code> does not have
     * a configurable parameter named <code>parameterName</code>
     */
    @Command(description = "Submits a change of parameter to be validated later", type = Command.CommandType.CONFIGURATION)
    public void submitChange(String componentName, String parameterName, Object value) {
        if (!configurationProxy.isParameterConfigurable(componentName, parameterName)) {
            throw new IllegalArgumentException("no such parameter : " + componentName+"//"+parameterName);
        }
        parameterSetterDictionary.get(componentName).submitChange(parameterName, value);
    }

    /**
     * Changes that have been submitted are processed as a group.
     */
    @Command(description = "processes the bulk change", type = Command.CommandType.CONFIGURATION)
    public void commitBulkChange() {
    }
    
    /**
     * Validates the set of submitted changes as a group.
     */
    protected abstract void validateBulkChanges();
        
    /**
     * Processes the set of submitted changes as a group.
     * changes.
     * @return a map of processed parameters ordered by the 
     * component they belong to.
     */
    protected abstract Set<ParameterPath> bulkChange();
    
    /**
     * Drops the submitted changes.
     * @deprecated use dropSubmittedChanges instead
     */
    @Deprecated
    @Command(description = "Drops the submitted changes", type = Command.CommandType.CONFIGURATION)
    public void dropBulkChange() {
        for (ParameterSetter parmSetter : parameterSetterDictionary.values()) {
            parmSetter.dropSubmittedChanges();
        }
    }

    @Command(description = "Drops the submitted changes for all components", type = Command.CommandType.CONFIGURATION)
    public void dropAllSubmittedChanges() {
        for (ParameterSetter parmSetter : parameterSetterDictionary.values()) {
            parmSetter.dropSubmittedChanges();
        }
    }
    
    @Command(description = "Drops the submitted changes for the given component", type = Command.CommandType.CONFIGURATION)
    public void dropSubmittedChangesForComponent(@Argument(description="the component name") String name) {
        parameterSetterDictionary.get(name).dropSubmittedChanges();
    }
    

    @Override
    public void doStart(){
        // publish the configuration the configurable is started with
        ConfigurationInfo configInfo = configurationProxy.buildConfigurationInfo(initialState, Collections.emptySet());
        updateInternalState("", initialState);
        getMessagingAccess().sendStatusMessage(new StatusConfigurationInfo(configInfo, getState()));
    }
    
    /**
     * Updates the state of the subsystem and sends a StatusConfigurationInfo status message
     * @param message 
     * @param recentChanges 
     */
    protected void updateStateAndSendStatusConfigurationInfo(String message, Set<ParameterPath> recentChanges) {
        ConfigurationState configState = configurationProxy.isDirty() ? ConfigurationState.DIRTY : ConfigurationState.CONFIGURED;
        updateInternalState(message, configState);
        
        getMessagingAccess().sendStatusMessage(new StatusConfigurationInfo(configurationProxy.buildConfigurationInfo(configState, recentChanges), getState()));        
    }

    @Command(description = "Returns the current submitted changes for the given component", type = Command.CommandType.CONFIGURATION)
    public Map<String, String> getSubmittedChangesForComponent(@Argument(description="the component name") String name) {
        return parameterSetterDictionary.get(name).getSubmittedChanges();
    }
    
    /**
     * Returns the current submitted changes for each component.
     * @return a map ordered by component name.
     */
    @Command(description = "Returns the current submitted changes for each component", type = Command.CommandType.CONFIGURATION)
    public Map<String, Map<String, String>> getAllSubmittedChanges() {
        Map<String, Map<String, String>> res = new TreeMap<>();
        for (Map.Entry<String, ParameterSetter> entry : parameterSetterDictionary.entrySet()) {
            Map<String, String> submittedChanges = entry.getValue().getSubmittedChanges();
            if (!submittedChanges.isEmpty()) {
                res.put(entry.getKey(), new TreeMap<String, String>(submittedChanges));
            }
        }
        return res;
    }
  
    /**
     * Returns the set of categories the subsystem's configurable parameters are
     * split into.
     * @return a set of category names.
     */
    @Command(description = "returns the categories of this subsystem")
    public Set<String> getCategories() {
        return Collections.unmodifiableSet(categories);
    }
    
    /**
     * Returns the available configuration names for the given category.
     * @param category
     * @return a set of configuration names.
     */
    @Command(description = "returns the available configurations for the given category")
    public Set<String> findAvailableConfigurationsForCategory(String category) {
        return configurationProxy.findAvailableConfigurationsForCategory(category);
    }
}
