package org.lsst.ccs.config;


import java.io.IOException;
import java.io.PrintWriter;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * When locally registering properties the <TT>LocalConfigurationProxy</TT> may need to write
 * properties in a File or in memory for tests purposes.
 * Classes that support this interface are able to create
 * the corresponding resource.
 */
public interface WriterProvider {
    
    static class ConfigurationFileName {
        
        /** The description name. */
        private final String tagName;
        
        /** The configuration name. */
        private final String configName;
        
        /** the category name. */
        private final String catName;
        
        /** The full configuration file name. */
        private final String fullName;
        
        ConfigurationFileName(String tagName, String configName, String categoryName) {
            this.tagName = tagName;
            this.configName = configName;
            this.catName = categoryName;
                String baseName = tagName;
            if (configName.isEmpty()){
                if (!catName.isEmpty())
                    baseName +="__"+catName;
            } else {
                baseName += "_" + configName;
                if (!catName.isEmpty())
                    baseName +="_"+catName;
            }
            this.fullName = baseName + ".properties";
        }
        
        ConfigurationFileName(String tagName, String fullName) {
            this.tagName = tagName;
            String trimmed = fullName.replace(".properties", "").replace(tagName,"");
            String[] split = trimmed.split("_");
            switch (split.length) {
                case 1:
                    configName = "";
                    catName = "";
                    break;
                case 2:
                    configName = split[1];
                    catName = "";
                    break;
                case 3:
                    configName = split[1];
                    catName = split[2];
                    break;                        
                default:
                    throw new IllegalArgumentException("file name is not a configuration file name : " + fullName);
            }
            this.fullName = fullName;
        }
        
        public String getConfigName() {
            return configName;
        }

        public String getFullName() {
            return fullName;
        }
    }
    
    public PrintWriter getPrintWriter(String configName, String tagName, String categoryName) throws IOException;

    public Properties getConfigurationProperties(String configName, String tagName, String categoryName) throws IOException;
    
    public default Set<String> findAvailableConfigurationsForCategory(String tagName, String category) {
        String pattern = tagName;
        if (category.isEmpty()) {
            pattern+="(_[^_]+?)??";
        } else {
            pattern+="_"+".*?"+"_"+category;
        }
        pattern+="\\.properties";
        Set<String> fullNames = findMatchingConfigurations(pattern);
        return fullNames.stream()
                .map(s -> { return new ConfigurationFileName(tagName, s).configName;}
                ).collect(Collectors.toSet());
    }
    
    Set<String> findMatchingConfigurations(String pattern);
}