package org.lsst.ccs.config;

import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.conv.TypeUtils;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import static java.util.stream.Collectors.toSet;
import org.lsst.ccs.CCSCst;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import static org.lsst.ccs.config.utilities.ConfigUtils.DEFAULT_CONFIG_NAME;
import org.lsst.ccs.framework.ConfigurationServiceException;

/**
 * Configuration proxy that registers configurations locally as properties files.
 * @author bamade
 */
// Date: 04/10/12

public class LocalConfigurationProxy implements ConfigurationProxy {
    
    private static Logger log = Logger.getLogger("org.lsst.ccs.config");
    
    private static WriterProvider writerProvider = null;
    
    protected static WriterProvider getWriterProvider() {
        if (writerProvider == null) {
            if ( System.getProperty("org.lsst.ccs.config.WriterProvier", "").equals("org.lsst.ccs.config.InMemoryWriterProvider") ) {
                writerProvider = new InMemoryWriterProvider();                            
            } else {
                writerProvider = new FileWriterProvider();            
            }
        }
        return writerProvider;
    }
    
    private final ASubsystemDescription subsystemDescription ;
    
    private final String tagName;
    /**
     * the configuration that is running in engineering/normal mode
     * (in normal mode both configurations are the same)
     */
    private final Map<String, AConfigProfile> configProfileMap = new HashMap<>();
    
    Logger logger = Logger.getLogger("org.lsst.ccs.config");
    
    /**
     * Builds a configuration proxy for the subsystem described by {@code subsystemDesc}.
     * @param subsystemDesc 
     */
    public LocalConfigurationProxy(SubsystemDescription subsystemDesc){
        getWriterProvider();
        this.subsystemDescription = (ASubsystemDescription)subsystemDesc;
        tagName = subsystemDescription.getTag();
    }
    
    /**
     * 
     * @return the set of categories the subsystem has its parameters split into 
     */
    @Override
    public Set<String> getCategorySet(){
        return subsystemDescription.getCategorySet();
    }
    
    /**
     * Loads the startup configuration represented by taggedCategories, initializes the config profiles
     * and returns a set of values to assign to parameters before startup
     * @param taggedCategories
     * @return 
     */
    @Override
    public Map.Entry<ConfigurationState, Set<ParameterConfiguration>> getInitialParameterConfigurations(Map<String, String> taggedCategories){
           
        //these Configuration objects: SubsystemDescription and ConfigProfile should be handled by a DAO
        // right now we operate locally so it is a special dummy DAO: MemoryDAO
        CCSCst.LOG_TODO.fine("TODO: DAO at creation of SubsystemDescription should also be able to interact with remote database (switch between local and remote?)") ;
        /*
        since No SubsystemDescription can be used unless it is under the control
        of a DAO this description is passed to a dummy "MemoryDAO"
        */
//        MemoryDAO dao = new MemoryDAO();
//        dao.saveAbstractDescription(subsystemDescription);
        boolean exceptionThrown = false;
        
        /*
        we create a <TT>ConfigProfile</TT> (where parameters  have values)
        out of the subsystem description
        This ConfigProfile is modified with the configuration Properties
        and saved to the MemoryDAO
        */
        // if no configuration create an empty configuration
        // TO remove
        for(Map.Entry<String,String> entry : taggedCategories.entrySet()){
            String cat = entry.getKey();
            String tag = entry.getValue();
            Properties configProps = null;
            AConfigProfile profileForCat;
            try {
                configProps = writerProvider.getConfigurationProperties(tag, tagName, cat);
                if (configProps == null && !tag.equals(DEFAULT_CONFIG_NAME)) {
                    throw new IllegalArgumentException("could not find configuration file " + tag + " for category " + cat);
                }
                profileForCat = createConfigProfileOutOfProperties(configProps, cat, tag);
            } catch(IOException ex) {
                exceptionThrown = true;
                profileForCat = createConfigProfileOutOfProperties(null, cat, DEFAULT_CONFIG_NAME);
            }
            configProfileMap.put(profileForCat.getCategoryName(), profileForCat);
//            dao.saveAbstractConfig(newConfigProfile);
        }
        return new AbstractMap.SimpleEntry<ConfigurationState, Set<ParameterConfiguration>>(
                exceptionThrown ? ConfigurationState.UNCONFIGURED : ConfigurationState.CONFIGURED,
                new HashSet(getParameterConfigurations())
        );
    }
    
    @Override
    public String getTagName() {
        return tagName;
    }
    
    /**
     * Builds a ConfigurationInfo object reflecting the configuration state and
     * ready to be sent on the buses.
     * @param configState
     * @param recentChangesPaths
     * @return the {@code ConfigurationInfo} object
     */
    
    @Override
    public synchronized ConfigurationInfo buildConfigurationInfo(ConfigurationState configState, Set<ParameterPath> recentChangesPaths){
        Map<String,String> tags = new HashMap<>();
        Map<String, Boolean> hasCatChanges = new HashMap<>();
        ArrayList<ConfigurationParameterInfo> parametersView = new ArrayList<>();
        ArrayList<String> recentChanges = new ArrayList<>();
        
        for (ConfigProfile profile : configProfileMap.values()){
            String category = profile.getCategoryName();
            // set tag for category
            tags.put(category, profile.getConfigName());
            hasCatChanges.put(profile.getCategoryName(), profile.isDirty());
            for (ParameterDescription parmDesc : subsystemDescription.getParamDescriptions().stream()
                    .filter(desc -> desc.getCategory().equals(category))
                    .collect(toSet())){
                ConfigurationParameterInfo configParameterInfo;
                ParameterConfiguration parmConfig = profile.fetch(parmDesc);
                if(parmConfig != null){
                    configParameterInfo = new ConfigurationParameterInfo(parmDesc.getPath().toString(),
                            parmDesc.getCategory(), parmDesc.getTypeName(), 
                            parmConfig.getConfiguredValue(), parmConfig.getValueAt(PackCst.STILL_VALID), parmConfig.hasChanged());
                } else {
                    configParameterInfo = new ConfigurationParameterInfo(parmDesc.getPath().toString(), 
                            parmDesc.getCategory(), parmDesc.getTypeName(), 
                            parmDesc.getDefaultValue(), parmDesc.getDefaultValue(), false);
                }
                parametersView.add(configParameterInfo);
                
                if(recentChangesPaths.contains(parmDesc.getPath())){
                    recentChanges.add(configParameterInfo.getPathName());
                }
            }
        }
        return new ConfigurationInfo(configState, tagName, tags, hasCatChanges, parametersView, recentChanges);
    }
    
    @Override
    public Boolean isParameterConfigurable(String componentName, String parameterName) {
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        ParameterDescription parameterDescription = subsystemDescription.fetch(path) ;
        return parameterDescription != null; 
    }

    
    @Override
    public synchronized void notifyParameterChange(String componentName, String parameterName, String value) {
        ParameterPath parameterPath = new ParameterPath(componentName,"",parameterName);
        //if static
        String category = subsystemDescription.fetch(parameterPath).getCategory();
        configProfileMap.get(category).temporaryChangeConfigurationValue(parameterPath.toString(), System.currentTimeMillis(), value);
    }
    
    @Override
    public synchronized void notifyUncheckedParameterChange(String componentName, String parameterName, Object value) {
        ParameterPath parameterPath = new ParameterPath(componentName,"",parameterName);
        
        String strValue ;
        if( value instanceof String) {
            strValue = (String)  value ;
        } else {
            strValue = TypeUtils.stringify(value) ;
        }
        String category = subsystemDescription.fetch(parameterPath).getCategory();
        configProfileMap.get(category).temporaryChangeConfigurationValue(parameterPath.toString(), System.currentTimeMillis(), strValue);
    }
    
    /**
     * Tagged categories are saved under a new tag name.
     * Other parameters are left unchanged, the configuration context remains
     * active.
     * @param taggedCategories
     */
    @Override
    public synchronized void saveChangesForCategoriesAs(Map<String, String> taggedCategories) throws ConfigurationServiceException {
        // Build a map of parameter descriptions grouped by category
        Map<String, Set<AParameterDescription>> parmDescGroupedByCategory = subsystemDescription.getParamDescriptions().stream()
                .collect(Collectors.groupingBy(AParameterDescription::getCategory, Collectors.toCollection(TreeSet::new)));
        
        for(Map.Entry<String,String> taggedCategory : taggedCategories.entrySet()){
            String cat = taggedCategory.getKey();
            String tag = taggedCategory.getValue();
            try {
                ConfigProfile profileForCat = configProfileMap.get(cat);
                
                Set<? extends ParameterConfiguration> modifiedParameters = new TreeSet<>(profileForCat.getModifiedParameters());
                
                if (modifiedParameters.isEmpty() && tag.equals(profileForCat.getConfigName())) continue;

                PrintWriter printWriter = writerProvider.getPrintWriter(tag, tagName, cat);

                for (ParameterConfiguration parameter : modifiedParameters){
                    String currentValue = parameter.getValueAt(PackCst.STILL_VALID);
                    boolean commentOut = currentValue.equals(parameter.getDescription().getDefaultValue());
                    if (!commentOut) {
                        printWriter.println(parameter.getDescription().toPropertyString(currentValue, commentOut));
                    }
                }
                // -- write other values, which are to be the default values : they are commented out
                for (ParameterDescription parmDesc : parmDescGroupedByCategory.get(cat)) {
                    if (modifiedParameters.contains(parmDesc)) continue;
                     printWriter.println(parmDesc.toPropertyString(parmDesc.getDefaultValue(), true));
                }
                printWriter.flush();
                printWriter.close();
                saveModifications(cat, tag);
            } catch(IOException ex){
                throw new ConfigurationServiceException("configuration service unavailable", ex);
            }
        }
    }
    
    @Override
    public Map<String,String> getTaggedCategoriesForCats(Set<String> categories){
        Map<String,String> res = new HashMap<>();
        for (String category : categories){
            res.put(category, configProfileMap.get(category).getConfigName());
        }
        return res;
    }
    
    
    private synchronized void saveModifications(String category, String configName){
        AConfigProfile oldProfile = configProfileMap.get(category);
        AConfigProfile newProfile = new AConfigProfile(oldProfile, false, configName);
        configProfileMap.put(category, newProfile);
    }
    
    /**
     * Modification occurring in one of the tagged categories are saved.
     * Other changes are left untouched.
     * @param taggedCategories
     */
    @Override
    public synchronized void saveModifications(Map<String, String> taggedCategories){
        for (Map.Entry<String, String> entry : taggedCategories.entrySet()){
            saveModifications(entry.getKey(), entry.getValue());
        }
    }
    
    /**
     * Returns the current value of all parameters of a given configurable component
     * @param componentName the name of the configurable component
     * @param categories
     * @return a map from parameter name to its current value.
     */
    public Map<String, String> getCurrentValuesForComponent(String componentName, Set<String> categories){
        if (categories.isEmpty()) {
            return subsystemDescription.getParamDescriptionMap().entrySet().stream()
                    .filter(entry -> entry.getKey().getComponentName().equals(componentName))
                    .collect(Collectors.toMap(entry -> entry.getKey().getParameterName(),
                            entry -> configProfileMap.get(entry.getValue().getCategory()).getValueAt(entry.getKey(), PackCst.STILL_VALID)
                    ));
        } else {
            return subsystemDescription.getParamDescriptionMap().entrySet().stream()
                    .filter(entry -> 
                        entry.getKey().getComponentName().equals(componentName) && 
                                categories.contains(entry.getValue().getCategory())
                            )
                    .collect(Collectors.toMap(entry -> entry.getKey().getParameterName(),
                            entry -> configProfileMap.get(entry.getValue().getCategory()).getValueAt(entry.getKey(), PackCst.STILL_VALID)
                    ));
        }
    }
    
    protected static void setWriterProvider(WriterProvider wp) {
        writerProvider = wp;
    }
    
    @Override
    public boolean isDirty(){
        return configProfileMap.values().stream()
                .map(ConfigProfile::isDirty)
                .reduce(false, (a,b) -> a||b);
    }
    
    @Override
    public Map<String, Properties> loadCategories(Map<String,String> taggedCategories) throws ConfigurationServiceException {
        if (taggedCategories.isEmpty()) return Collections.EMPTY_MAP;
        Map<String, Properties> res = new HashMap<>();
        for(Map.Entry<String,String> entry : taggedCategories.entrySet()){
            String cat = entry.getKey();
            String tag = entry.getValue();
            try {
                Properties configProps = writerProvider.getConfigurationProperties(tag, tagName, cat);
                if (configProps == null && !tag.equals(DEFAULT_CONFIG_NAME)) {
                    throw new IllegalArgumentException("could not find configuration file " + tag + " for category " + cat);
                }
                AConfigProfile newConfigProfile = createConfigProfileOutOfProperties(configProps, cat, tag);
                Map<String, Properties> changesForCategory = buildChangesBetweenProfiles(configProfileMap.get(cat), newConfigProfile);
                
                for(Map.Entry<String, Properties> e : changesForCategory.entrySet()){
                    Properties changesForComponent = res.get(e.getKey());
                    if (changesForComponent == null){
                        changesForComponent = new Properties();
                        res.put(e.getKey(), changesForComponent);
                    }
                    changesForComponent.putAll(e.getValue());
                }
            } catch (IOException ex){
                throw new ConfigurationServiceException("configuration service problem", ex);
            }
        }
        return res;
    }
    
    /**
     * Returns a map of components and their corresponding changes to switch from
     * oldProfile to newProfile.
     * @param oldProfile
     * @param newProfile
     * @return a map of changes
     */
    private Map<String, Properties> buildChangesBetweenProfiles(ConfigProfile oldProfile, ConfigProfile newProfile){
        // map the name of module to the list of parameters modified by newProfile for this module
        Set<? extends ParameterConfiguration> newChanges = newProfile.getModifiedParameters();
        // map the name of module to the list of parameters modified by oldProfile for this module
        Set<? extends ParameterConfiguration> oldChanges = new HashSet<>(oldProfile.getModifiedParameters());
        
        Map<String, Properties> res = new HashMap<>();
        
        for (ParameterConfiguration parmConfig : newChanges){
            // The new changes overrides the old change : we remove it
            oldChanges.remove(parmConfig);
            ParameterPath path = parmConfig.getPath();
            String parameterName = path.getParameterName();
            // The new value to assign
            String value = parmConfig.getConfiguredValue();
            String oldValue = oldProfile.getValueAt(path, PackCst.STILL_VALID);
            //TODO: some problems if strings are not exactly the same! (though the values are equals! examples : maps , unordered lists, floating point)
            // change the test
            if (oldValue.equals(value)) {
                continue;
            }
            
            Properties changesForComponent = res.get(path.getComponentName());
            if (changesForComponent == null){
                changesForComponent = new Properties();
                res.put(path.getComponentName(), changesForComponent);
            }
            changesForComponent.setProperty(parameterName, value);
        }
        // Process the remaining parameters in oldChanges : set to their default value
        for (ParameterConfiguration parmConfig : oldChanges){
            ParameterPath path = parmConfig.getPath();
            String parameterName = path.getParameterName();
            String defaultValue = parmConfig.getDescription().getDefaultValue() ;
            
            Properties changesForComponent = res.get(path.getComponentName());
            if (changesForComponent == null){
                changesForComponent = new Properties();
                res.put(path.getComponentName(), changesForComponent);
            }
            changesForComponent.setProperty(parameterName, defaultValue);
        }
        return res;
    }
    
    private AConfigProfile createConfigProfileOutOfProperties(Properties configProps, String categoryName, String configName){
        AConfigProfile res = new AConfigProfile(subsystemDescription, categoryName, configName);
        if (configProps != null){
            for (String name : configProps.stringPropertyNames()){
                res.addParameterConfiguration(name, configProps.getProperty(name));
            }
        }
        return res;
    }
    
    public ConfigProfile getProfileForCat(String categoryName){
        return configProfileMap.get(categoryName);
    }
        
    private Set<ParameterConfiguration> getParameterConfigurations(){
        Set<ParameterConfiguration> res = new HashSet<>();
        for(String category : subsystemDescription.getCategorySet()){
            res.addAll(configProfileMap.get(category).getParameterConfigurations());
        }
        return res;
    }
    
    @Override
    public void setDefaultValueForParameter(String componentName, String parameterName, Object val) {
        subsystemDescription.paramDescriptions.get(new ParameterPath(componentName, "", parameterName)).getParameterBase().setDefaultValue(
                TypeUtils.stringify(val));
    }
    
    @Override
    public void writeMissingDefaultConfigs() {
        Map<String, Set<ParameterDescription>> parmDescs = subsystemDescription.getParamDescriptions().stream()
                .collect(Collectors.groupingBy(ParameterDescription::getCategory, Collectors.toCollection(TreeSet::new)));
        for (Map.Entry<String, Set<ParameterDescription>> entry : parmDescs.entrySet()) {
            String cat = entry.getKey();
            
            try {
                if (writerProvider.getConfigurationProperties(DEFAULT_CONFIG_NAME, tagName, cat) == null) {
                    // there is no default configuration file for this category
                    PrintWriter writer = writerProvider.getPrintWriter(DEFAULT_CONFIG_NAME, tagName, cat);
                    for (ParameterDescription parmDesc : entry.getValue()) {
                        writer.println(parmDesc.toPropertyString(parmDesc.getDefaultValue(), true));
                    }
                    writer.flush();
                    writer.close();
                }
            } catch (IOException ex) {
                logger.warn("could not write default configuration file for category : " + cat);
            }
        }
    }
    
    @Override
    public Set<String> findAvailableConfigurationsForCategory(String category) {
        return writerProvider.findAvailableConfigurationsForCategory(tagName, category);
    }
}
