package org.lsst.ccs.config;

import javax.persistence.*;
import java.util.ArrayList;
import java.util.List;

/**
 * Actual parameter configuration.
 * TODO: when setting up the value this data will be checked for constraints and type against
 * its ParameterDescription.
 * <P>
 *  All values are represented as String.
 *  The type in ParameterDescription should have a static <TT>valueOf(String val)</TT>
 *  method. This is the case for the standard "wrapper" classes.
 *  Other non-standard classes may be provided with the same feature (including classes that
 *  may represent "structures" such as arrays, lists or maps).
 *  If the value is generated then it will be checked with the constraints type (to be specified).
 *
 * @author bamade
 */
// Date: 11/04/12

    @Entity
    //@Immutable no more
//@Table(name="ParmConfiguration")
class AParameterConfiguration extends ParameterConfiguration implements Cloneable{
    private static final long serialVersionUID = 8134633715625475333L;
    @Id
    @GeneratedValue
    private long id ; //generated
    /*
    Beware: constraint: configuration should be deleted BEFORE any pointed description
    but in that case
     */
    //TODO: EAGER not obvious
    @ManyToOne (fetch=FetchType.EAGER)
    //@NaturalId not true! ther can be copy of the same
    private /*@NotNull*/ AParameterDescription parameterDescription ;

    private boolean copy ;

    @OneToMany (cascade = CascadeType.ALL, fetch=FetchType.EAGER)
    private List<AValueEvent> valueEvents ;

    ////////////////////////////// CONSTRUCTORS


    AParameterConfiguration() {

    }

    /**
     * create an empty parameter configuration whose value is to be modified
     * @param parameterDescription
     */
    AParameterConfiguration(AParameterDescription parameterDescription) {
        super(parameterDescription.getParameterBase().getDefaultValue()) ;
        this.parameterDescription = parameterDescription;
    }

    /**
     *
     * @param parameterDescription
     * @param value
     * @throws IllegalArgumentException if value is not coherent with type and constraints
     */
    AParameterConfiguration(AParameterDescription parameterDescription, String value) {
        super();
        this.parameterDescription = parameterDescription;
        modifyValue(value);
    }
    
    /**
     * Copy constructor of a parameter configuration
     * @param other
     * @param copyEvents if set to true, the history of changes is copied. If 
     * set to false, the last value event is used as the value for the new
     * ParameterConfiguration object.
     */
    AParameterConfiguration(AParameterConfiguration other, boolean copyEvents){
        this(other.getParameterDescription(),other.getConfiguredValue());
        if (other.hasChanged()){
            List<? extends ValueEvent> eventList = other.getValueEvents();
            if (copyEvents){
                if (eventList != null) {
                    for (ValueEvent event : eventList) {
                        addValueEvent(event);
                    }
                }
                this.copy = true;
            } else {
                //move last in history to value
                if (eventList != null) {
                    int size = eventList.size();
                    // "belt and suspenders" strategy
                    if (size > 0) {
                        ValueEvent event = eventList.get(size - 1);
                        modifyValue(event.getValue());
                    }
                }
            }
        }
    }
    
    // REVIEW : what is this used for ?
    public static AParameterConfiguration copyWithoutEvents(ParameterConfiguration other) {
        // if other is Past -> description is ghost
        // if ghost still valid -> get RealDescription
        //       else Exception no valid description
       throw new UnsupportedOperationException("copy without events (general)")  ;
    }

    ////////////////////////////// ACCESSORS/MUTATORS
    @Override
    public long getId() {
        return id;
    }

    @Override
    protected void setId(long id) {
        this.id = id ;
    }

    @Override
    public ParameterDescription getDescription() {
        return parameterDescription;
    }

    @Override
    public List<? extends ValueEvent> getValueEvents() {
        return this.valueEvents ;
    }

    void setValueEvents(List<AValueEvent> valueEventsList) {
        this.valueEvents = valueEventsList ;
    }

    /**
     * this method should be called from a ConfigProfile where property "beenInEngineeringMode" is set.
     * this is not checked in the body of this current method.
     * @param event a newly created ValueEvent
     */
    //@Override
    public final void addValueEvent(ValueEvent event) {
        if(! (event instanceof AValueEvent)) {
            throw new IllegalArgumentException("trying to add a deprecated ValueEvent");
        }
        AValueEvent anEvent = (AValueEvent) event ;
        if(valueEvents == null) {
          valueEvents = new ArrayList<AValueEvent>()   ;
        }
        valueEvents.add(anEvent);
    }


    AParameterDescription getParameterDescription() {
        return parameterDescription;
    }

     void setParameterDescription(AParameterDescription parameterDescription) {
        this.parameterDescription = parameterDescription;
    }

    public boolean isCopy() {
        return copy;
    }

     void setCopy(boolean copy) {
        this.copy = copy;
    }

    public AParameterConfiguration clone() {
        AParameterConfiguration res = null ;
        try {
            res = (AParameterConfiguration) super.clone();
            if (valueEvents != null) {
                for (AValueEvent event : valueEvents) {
                    res.addValueEvent(event);
                }
            }
            res.setCopy(true);
        } catch (CloneNotSupportedException e) {/*IGNORE*/ }
        
        return res ;
    }
}
