package org.lsst.ccs.config;


import java.util.Map;
import java.util.Properties;
import java.util.Set;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.framework.ConfigurationServiceException;

/**
 * Subsystem access to configuration information.
 * <P/>
 * <B>BEWARE</B>: there are two potential sources for description and configuration
 * (database or/and local file) and two potential sink when registering changes
 * (database or/and local file). There may be potential concurrency issues (mostly when dealing
 * with the remote database) and so commands that fire the modifications should be sure
 * to get a proper lock from the lock manager.
 *
 * @author bamade
 */
// Date: 03/10/12

public interface ConfigurationProxy {
    
    /**
     * Builds a ConfigurationInfo object reflecting the configuration state and
     * ready to be sent on the buses.
     * @param configState
     * @param recentChangesNames
     * @return the {@code ConfigurationInfo} object
     */
    ConfigurationInfo buildConfigurationInfo(ConfigurationState configState, Set<ParameterPath> recentChangesNames);
    
    /**
     * 
     * @return the set of categories the subsystem has its parameters split into 
     */
    Set<String> getCategorySet();
    

    /**
     * the current tag name (can be null or empty)
     *
     * @return the tag of the subsystem this configuration proxy is running for
     */
    String getTagName();
    
    /**
     * To be invoked before startup.
     * Returns the set of ParameterConfiguration to apply to the subsystem description before startup
     * and the configuration state the associated ConfigurableSubsystem will start with. This state 
     * is CONFIGURED if access to the configuration service is successful, UNCONFIGURED if not. In that
     * case, the subsystem is started with parameters set to the default value specified in the subsystem 
     * description.
     * @param taggedCategories the name of the initial configuration.
     * @return the initial configuration state and a set of ParameterConfiguration.
     */
    Map.Entry<ConfigurationState, Set<ParameterConfiguration>> getInitialParameterConfigurations(Map<String, String> taggedCategories);
    
    /**
     * Checks if a parameter named parameterName is part of the configuration data. 
     * @param componentName the name of the component
     * @param parameterName the name of the parameter
     * @return true if the parameter is part of the configuration data, false otherwise.
     */
    public Boolean isParameterConfigurable(String componentName, String parameterName);
    
    
    /**
     * second part of a two phase commit. this new value is registered as a new version of configuration data.
     *
     * @param componentName
     * @param parameterName
     * @param stringValue
     */
    //TODO exception if configuration service unavailable
    void notifyParameterChange(String componentName, String parameterName, String stringValue);

    /**
     * notifies directly of a change without preliminary check to the configuration service.
     * Use at you own risk!
     *
     * @param componentName
     * @param parameterName
     * @param Value
     */
    void notifyUncheckedParameterChange(String componentName, String parameterName, Object Value);

    /**
     * Tagged categories in taggedCategories are saved under a new tag name.
     * If saveOtherCategories is true, parameters belonging to other categories 
     * are saved in their current tagged configuration. If set to false, they are
     * left unchanged
     * @param taggedCategories 
     * @throws org.lsst.ccs.framework.ConfigurationServiceException 
     */
    void saveChangesForCategoriesAs(Map<String,String> taggedCategories) throws ConfigurationServiceException ;
    
    /**
     * Returns the current value of parameters that belong to the given category for the given configurable component
     * @param componentName the name of the configurable component
     * @param categories
     * @return a map from parameter name to its current value.
     */
    Map<String, String> getCurrentValuesForComponent(String componentName, Set<String> categories);
    
    /**
     * For each pair 'category:configuration' in taggedCategories, the category is loaded
     * with its specified configuration, categories that are not mentioned are left untouched.
     * The configuration context remains active
     * @param taggedCategories a list of tagged categories
     * @return a map of component name to properties
     * @throws org.lsst.ccs.framework.ConfigurationServiceException 
     */
    Map<String, Properties> loadCategories(Map<String,String> taggedCategories) throws ConfigurationServiceException;
    
    void saveModifications(Map<String,String> categoryProfile);
    
    boolean isDirty();
    
    Map<String,String> getTaggedCategoriesForCats(Set<String> categories);

    public void setDefaultValueForParameter(String componentName, String parameterName, Object val);
    
    public void writeMissingDefaultConfigs();



}
