package org.lsst.ccs.config;


import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * Writes properties files in memory.
 * Use for test purposes only
 * @author The LSST CCS Teams
 */

public class InMemoryWriterProvider implements   LocalConfigurationProxy.WriterProvider {

    private static InMemoryWriterProvider instance = null;
    private final Map <String, ByteArrayOutputStream> propertiesMap = new HashMap<String, ByteArrayOutputStream>();
    
    /**
     * Private default constructor
     */
    private InMemoryWriterProvider(){
        
    }
    
    public static InMemoryWriterProvider getInstance(){
        if(instance == null){
            instance = new InMemoryWriterProvider();
        }
        return instance;
    }
    
    public static InMemoryWriterProvider newInstance(){
        instance = new InMemoryWriterProvider();
        return instance;
    }
    
    @Override
    public PrintWriter getPrintWriter( String configFileName) throws IOException {
        if ( ! configFileName.endsWith(".properties") ) {
            configFileName += ".properties";
        }
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        Writer writer = new OutputStreamWriter(bos) ;
        propertiesMap.put(configFileName, bos);
        return new PrintWriter(writer) ;
    }

    public  String printConfig(String configFileName) {
        if ( ! configFileName.endsWith(".properties") ) {
            configFileName += ".properties";
        }
        return propertiesMap.get(configFileName).toString() ;
    }
    
    
    public String printConfigNames(){
        String result = "[";
        for (String s : propertiesMap.keySet()){
            result += s+",";
        }
         return result.substring(0, result.length()-1)+"]";
    }

    public Properties getConfigurationProperties(String configFileName) {
        if ( ! configFileName.endsWith(".properties") ) {
            configFileName += ".properties";
        }
        Properties props = new Properties() ;
        ByteArrayOutputStream bos = propertiesMap.get(configFileName);
        if (bos == null) {
            throw new IllegalArgumentException("Could not find configuration file : " + configFileName);
        }
        byte[] array = bos.toByteArray() ;
        
        try {
            props.load(new InputStreamReader(new ByteArrayInputStream(array)));
        } catch (IOException e) {
            System.err.println("error while reading properties from memory");
        }
        return props ;
    }
    
    public void clearPropertiesMap(){
        propertiesMap.clear();
    }
}
