package org.lsst.ccs.config;

import org.hibernate.annotations.Immutable;

import javax.persistence.*;
import java.util.*;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * An actual configuration profile for a given category.
 * Once registered instances of this class are immutable except for  comments
 * and registration of method calls (to be specified).
 *
 * @author bamade
 */
// Date: 12/04/12

@Table(name = "AConfigProfile",
        uniqueConstraints = {@UniqueConstraint(columnNames = {"name", "tag", "subsystemName"})}
)
@Entity
@Immutable
class AConfigProfile extends ConfigProfile implements Cloneable {
    private static final long serialVersionUID = -6199141309488812631L;
    @Id
    @GeneratedValue
    private long id; //generate

    @ManyToOne (fetch=FetchType.EAGER)
    private /*@NotNull*/ ASubsystemDescription subsystemDesc;

    /**
     * duplicate information : helps when in an intermediate state where
     * the subsystemDescription is being deprecated ;
     */
    private long subsystemId;
    
    //TODO:  annotate method instead to avoid bug with unModifiableConfigurations (the set was not invoked)
    @OneToMany(cascade = CascadeType.ALL, fetch=FetchType.EAGER)
    private /*@NotNull*/ Set<AParameterConfiguration> parameterConfigurations =
            // just in case! not ideal when handled as a bean
            new HashSet<AParameterConfiguration>();
    
    @Transient
    private /*@NotNull*/ Set<AParameterConfiguration> unModifiableConfigurations =
            Collections.unmodifiableSet(parameterConfigurations);
    
    private boolean isDirty;
    
    ////////////////////////////////// CONSTRUCTORS

    AConfigProfile() {
    }

    /**
     * creates a config profile pointing to a Subsystem Description that must be registered in the database.
     *
     * @param subsystemDesc
     * @param name
     * @param userName
     * @throws IllegalArgumentException if the subsystem description is not registered or is deprecated
     */
    AConfigProfile(ASubsystemDescription subsystemDesc, String categoryName, String configName) {
        super(subsystemDesc.getSubsystemName(), subsystemDesc.getTag(), categoryName, configName);
        this.subsystemDesc = subsystemDesc;
        this.subsystemId = subsystemDesc.getId();
        if (subsystemId == 0) {
            throw new IllegalArgumentException("Subsystem description should have been registered in database");
        }
        //TODO: tautological all instances of ASubsystemDescription have a STILL_VALID date!
        long endDate = subsystemDesc.getEndTimestamp();
        // when in remote operation this tet may not suffice
        if (!(endDate == PackCst.STILL_VALID)) {
            throw new IllegalArgumentException("using deprecated subsystem");
        }
        isDirty = false;
    }


    /**
     * creates a copy of an active ConfigProfile (for instance to start an engineering mode or to validate
     * a new tagged configuration from an engineering mode configuration).
     *
     * @param anotherConfig
     * @param name
     * @param userName
     * @param level
     * @param toEngineering if set to true, the history of parameter change is copied
     */
    AConfigProfile(AConfigProfile anotherConfig,
            boolean toEngineering, String newName) {
        this(anotherConfig.getSubsystemDesc(), anotherConfig.getCategoryName(), newName);

        for (AParameterConfiguration parmConfig : anotherConfig.getParameterConfigurations()) {
            // TODO : this code should be moved to ParameterConfiguration
            AParameterConfiguration copyParm = new AParameterConfiguration(parmConfig, toEngineering);
            this.addParameterConfigurations(copyParm);
        }
        isDirty = toEngineering ? anotherConfig.isDirty : false;
    }
    
    /////////////////////////////////// ACCESSORS/MUTATORS
    @Override
    public long getId() {
        return id;
    }
    
    @Override
    protected void setId(long id) {
        this.id = id;
    }


    @Override
    public SubsystemDescription getSubsystemDescription() {
        return subsystemDesc;
    }

    /**
     * return <TT>getParameterConfigurations</TT>
     *
     * @return
     */
    @Override
    public Set<? extends ParameterConfiguration> getModifiedParameters() {
        return getParameterConfigurations();
    }
    
    /**
     * Runtime change of parameter value.
     * 
     * @param parameterPath the name of the parameter
     * @param time
     * @param value
     * @param checked True if the value has already been checked
     * @throws IllegalArgumentException if the parameter is not modifiable at runtime
     */
    @Override
    public ParameterConfiguration temporaryChangeConfigurationValue(String parameterPath, long time, String value, boolean checked) {
        ParameterPath path = ParameterPath.valueOf(parameterPath) ;
        // fetch the ParameterConfiguration
        AParameterConfiguration parameter = (AParameterConfiguration)this.fetch(path) ;
        if(parameter == null) {
            //creates a new Parameter
            AParameterDescription description = (AParameterDescription)subsystemDesc.fetch(path) ;
            // register it to the  configProfile
            parameter = new AParameterConfiguration(description) ;
            if(parameter.changingStaticData) {
                throw new IllegalArgumentException("changing static data at runtime not allowed") ;
            }
            this.addParameterConfigurations(parameter);
        }
        parameter.addValueEvent(new AValueEvent(time, value), checked);
        isDirty = true;
        return parameter ;
    }

    ASubsystemDescription getSubsystemDesc() {
        return subsystemDesc;
    }

    void setSubsystemDesc(ASubsystemDescription subsystemDesc) {
        this.subsystemDesc = subsystemDesc;
    }

    /**
     * @return an unmodifiable set
     */
    public Set<AParameterConfiguration> getParameterConfigurations() {
        //TODO : annotate method instead of implementing this hack
        //HIBERNATE HACK
        if(unModifiableConfigurations.size() != parameterConfigurations.size()) {
            this.unModifiableConfigurations = Collections.unmodifiableSet(parameterConfigurations);
        }
        return unModifiableConfigurations;
    }
    
    void setParameterConfigurations(Set<AParameterConfiguration> parameterConfigurations) {
        this.parameterConfigurations = parameterConfigurations;
        this.unModifiableConfigurations = Collections.unmodifiableSet(parameterConfigurations);
    }

    public AConfigProfile clone() {
        AConfigProfile res = null;
        try {
            res = (AConfigProfile) super.clone();
            HashSet<AParameterConfiguration> newSet = new HashSet<AParameterConfiguration>();
            for (AParameterConfiguration parmConfig : parameterConfigurations) {
                newSet.add(parmConfig.clone());
            }
            res.setParameterConfigurations(newSet);
        } catch (CloneNotSupportedException e) { /*IGNORE*/}
        return res;

    }

    public long getSubsystemId() {
        return subsystemId;
    }

    void setSubsystemId(long subsystemId) {
        this.subsystemId = subsystemId;
    }

    /**
     * the only public mutator
     * <p/>
     * TODO: no persistence of this for the moment
     *
     * @param remarks
     */
    public void setRemarks(String remarks) {
        //TODO add a transient modification boolean checked by the persistence
        super.setRemarks(remarks);
    }

    //////////////////////////// IDENT METHODS
///////////////////////////////// UTILITIES METHODS

    /**
     * Static addition of a parameter configuration.
     * @param parameterName
     * @param value 
     */
    public void addParameterConfiguration(String parameterName, String value){
          ParameterDescription description = subsystemDesc.fetch(parameterName);
          if (description != null) {
              addParameterConfigurations(new AParameterConfiguration((AParameterDescription)description, value, true));
          } else {
              Logger.getLogger("org.lsst.ccs.config").warn("incoherent name for parameter : " + parameterName + " : ignored");
          }
    }
    /**
     * registers a list of parameter configurations
     *
     * @param parameterConfigurations
     * @throws ImmutableStateException  if the object is already registered in database
     * @throws IllegalArgumentException if the corresponding parameter descriptions are not alive in the current
     *                                  subsystem description
     */
    // TODO: check if Description not deprecated!
    public void addParameterConfigurations(AParameterConfiguration... parameterConfigurations) {
        if (isReadOnly()) {
            throw new ImmutableStateException("no modification of registered data");
        }
        for (AParameterConfiguration parameterConfiguration : parameterConfigurations) {
            AParameterDescription description = parameterConfiguration.getParameterDescription();
            if (description.getCategory().equals(getCategoryName())){
                if(parameterConfiguration.changingStaticData) {
                    this.setChangingStaticData(true);
                }
                // check for deprecation
                this.parameterConfigurations.add(parameterConfiguration);
            } else {
                Logger.getLogger("org.lsst.ccs.config").warn("The property " 
                        + description.getParameterName() 
                        + " does not belong to the category specified by the property file name : " + getCategoryName());
            }
        }
    }

    /**
     * adds parameter configuration objects to this profile
     * @param parameterConfigurations
     */
    public void addParameterConfigurations(ParameterConfiguration... parameterConfigurations) {
        addParameterConfigurations(buildSafeArray(parameterConfigurations));
    }

    /**
     * Casts the array of parameterConfiguration into an array of AParameterConfiguration.
     * @param parameterConfigurations
     * @throws IllegalArgumentException if at least one of the argument is not 
     * of instance AParameterConfiguration.
     * @return 
     */
    private AParameterConfiguration[] buildSafeArray(ParameterConfiguration... parameterConfigurations) {
        AParameterConfiguration[] parms = new AParameterConfiguration[parameterConfigurations.length];
        for (int ix = 0; ix < parameterConfigurations.length; ix++) {
            ParameterConfiguration conf = parameterConfigurations[ix];
            if (!(conf instanceof AParameterConfiguration)) {
                throw new IllegalArgumentException("deprecated parameter Configuration");
            }
            parms[ix] = (AParameterConfiguration) conf;

        }
        return parms;
    }

    /**
     * removes a list of parameter configurations
     *
     * @param parameterConfigurations
     * @throws ImmutableStateException if operated on an object registered to the database
     */
    public void removeParameterConfigurations(ParameterConfiguration... parameterConfigurations) {
        if (isReadOnly()) {
            throw new ImmutableStateException("Parameter Configuration list");
        }
        //todo: problem how to remove property "changingStaticData"? should we check all other parmConfigs?
        for (ParameterConfiguration parameterConfiguration : parameterConfigurations) {
            this.parameterConfigurations.remove(parameterConfiguration);
        }
    }
    
    public boolean isDirty(){
        return isDirty;
    }

}
