package org.lsst.ccs.config.utilities;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.config.NamesAndTag;

/**
 * @author bamade
 */
// Date: 19/06/13

public class ConfigUtils {
    
    public static final String DEFAULT_CONFIG_NAME = "";
    public static final String DEFAULT_CAT = "";
    
    private ConfigUtils() {
    }
    
    /**
     * gets subsystem name, configName and tag name from pathName
     * @param pathName
     * @return a {@code NamesAndTag} object structuring the names extracted from
     * the path name.
     */
    public static NamesAndTag namesFromPath(String pathName) {
        //split name
        String subsystemName = "" ;
        String configName = "" ;
        String tag = "" ;
        String cat = "";
        if (isAPath(pathName) ){ // pathName is a config filename
            int indexDot =  pathName.lastIndexOf('.');
            int lastPath = pathName.lastIndexOf('/');
            String rawName = pathName.substring(lastPath + 1, indexDot);
            String[] elements = rawName.split("_");
            //falltrough !
                switch(elements.length) {
                    case 3: configName = elements[2]+":";
                        cat = elements[2];
                    case 2: configName += elements[1] ;
                    case 1: tag = elements[0] ;
                }
        } else { // pathName is a config name
        configName = pathName;
        }
        return new NamesAndTag(subsystemName,configName,tag,cat);
    }
    
    public static NamesAndTag namesFromDescriptionPath(String pathName) {
        String subsystemName ="";
        String configName = "";
        String tag = "";
        
        tag = pathName.substring(pathName.lastIndexOf("/")+1);
        return new NamesAndTag(subsystemName,configName,tag);
    }
    
    /**
     * gets subsystem name, configName and tag name from the description and configuration path names
     * <BR>
     * rules:
     * <UL>
     *     <LI/>
     *     the subsystem name in both file names should be the same
     *     <LI/>
     *     if the configuration name in the description file name is not set
     *     then the configuration name is in the configuration file name.
     *     Otherwise there is a name clash if it is set in the configuration file name
     *     and it differs.
     *     <LI/>
     *     Same rule applies for the tag name.
     *
     * </UL>
     * @param descriptionName 
     * @param configurationName 
     * @throws IllegalArgumentException if subsystem, config and tag names are not consistent in description and configuration file names.
     * @return a {@code NamesAndTag} object representing the extracted names from
     * the description and configuration names.
     */
    public static NamesAndTag namesFromPaths(String descriptionName, String configurationName){
        String subsystemName, configName, tagName, catName;
        NamesAndTag descNT = namesFromDescriptionPath(descriptionName);
        subsystemName = descNT.getSubsystemName();
        configName = descNT.getConfigName();
        tagName = descNT.getTag();
        catName = descNT.getCat();
        
        if (configurationName != null){
            NamesAndTag confNT = namesFromPath(configurationName);

            // check coherency between both NamesAndTag objects
            if (!confNT.getSubsystemName().isEmpty()){
                if (!subsystemName.equals(confNT.getSubsystemName())) {
                    throw new IllegalArgumentException(" subsystem names differs :" + subsystemName + " <> " + confNT.getSubsystemName());
                }
            }
            String newConfigName = confNT.getConfigName();
            if (configName.length() != 0 && newConfigName.length() != 0) {
                if (!configName.equals(newConfigName)) {
                    throw new IllegalArgumentException(" configuration name clash :" + configName + " <> " + newConfigName);
                }
            }
            if (newConfigName.length() != 0) {
                configName = newConfigName;
            }
            
            String newTagName = confNT.getTag();
            if (tagName.length() != 0 && newTagName.length() != 0) {
                if (!descNT.getTag().equals(newTagName)) {
                    throw new IllegalArgumentException(" tag name clash :" + tagName + " <> " + newTagName);
                }
            }
            if (newTagName.length() != 0) {
                tagName = newTagName;
            }
            catName = confNT.getCat();
        
        }
        
        return new NamesAndTag(subsystemName, configName, tagName, catName);
    }
    
    public static String baseNameFromNames(String configName, String tagName){
        return baseNameFromNames(configName,tagName,"");
    }
    
    public static String baseNameFromNames(String configName, String tagName, String catName){
        String part1,part2,part3,part4=null;
        if(isAPath(configName)){
            return configName;
        }
            part1 = tagName;
            part2 = configName;
            part3 = catName;
        String baseName = part1;
        if (part2.isEmpty()){
            if (!part3.isEmpty())
                baseName +="__"+part3;
        } else {
            baseName += "_" + part2;
            if (!part3.isEmpty())
                baseName +="_"+part3;
        }
        if(part4 != null && !part4.isEmpty()){
            baseName+="_" + part4;
        }
        return baseName;
    }
    
    private static boolean isAPath(String string){
        if (string == null )return false;
        int lastDot = string.lastIndexOf('.');
        int lastPath = string.lastIndexOf('/');
        return lastDot > lastPath;
    }
    
    
    // Utility methods to process configuration input String
    
    /**
     * ParseConfigurationString without checking for category consistency
     *
     * @param configNames
     * @return a map with category as keys and their configuration name as values
     */
    public static Map<String,String> rawParseConfigurationString(String... configNames){
        return parseConfigurationString(null, configNames);
    }

    /**
     * Parses a string of tagged categories.
     *
     * @param categorySet
     * @param configNames
     * @throws IllegalArgumentException if one of the categories is not present in categorySet.
     * @return a map with category as keys and their configuration name as values
     */
    public static Map<String,String> parseConfigurationString(Set<String> categorySet, String... configNames){
        Map<String, String> res = new HashMap<>();
        for (String s : configNames){
            if (s.isEmpty()){
                res.put(DEFAULT_CAT, DEFAULT_CONFIG_NAME);
            } else if (!s.contains(":")){
                res.put(DEFAULT_CAT, s);
            } else {
                String[] keyVal = s.split(":",-1);
                if (keyVal.length != 2) throw new IllegalArgumentException("configuration " + keyVal.length);
                if (categorySet != null){
                    if (!categorySet.contains(keyVal[0])) throw new IllegalArgumentException("unexisting category " + keyVal[0]);
                }
                res.put(keyVal[0],keyVal[1]);
                    }
                }
        return res;
    }

    public static Map<String,String> parseConfigurationStringWithDefaults(Set<String> categories, String ... taggedCategories){
        Map<String,String> res = new HashMap();
        for (String cat : categories){
            res.put(cat, DEFAULT_CONFIG_NAME);
        }
        res.putAll(parseConfigurationString(categories, taggedCategories));
        return res;
    }

    public static Set<String> parseCategories(Set<String> categorySet, String ... categories){
        Set<String> res = new HashSet();
        for (String cat : categories) {
            if (!categorySet.contains(cat)) {
                throw new IllegalArgumentException("unexisting category " + cat);
            }
            res.add(cat);
        }
        return res;
    }
}
