package org.lsst.ccs.config;

import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.structs.ViewValue;
import org.lsst.gruth.jutils.TypeUtils;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import static java.util.stream.Collectors.toSet;
import org.lsst.ccs.CCSCst;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.config.utilities.ConfigUtils;
import static org.lsst.ccs.config.utilities.ConfigUtils.DEFAULT_CONFIG_NAME;
import org.lsst.ccs.framework.ConfigurationServiceException;
import org.lsst.gruth.jutils.Constraints;

/**
 * Configuration proxy that registers configurations locally as properties files.
 * @author bamade
 */
// Date: 04/10/12

public class LocalConfigurationProxy implements ConfigurationProxy {
    
    private final static String FILE_SEPARATOR = System.getProperty("file.separator");
   
    
    /**
     * When locally registering properties the <TT>LocalConfigurationProxy</TT> may need to write
     * properties in a File or in memory for tests purposes.
     * Classes that support this interface are able to create
     * the corresponding resource.
     */
    public static interface WriterProvider {
        public PrintWriter getPrintWriter(String fileName) throws IOException;
        public Properties getConfigurationProperties(String configFileName) throws IOException;
        
    }
    private final ASubsystemDescription subsystemDescription ;
    
    private final String subsystemName;
    private final String tagName;
    /**
     * the configuration that is running in engineering/normal mode
     * (in normal mode both configurations are the same)
     */
    private final Map<String, AConfigProfile> configProfileMap = new HashMap<>();
    
    Logger logger = Logger.getLogger("org.lsst.ccs.config");
    
    private WriterProvider writerProvider ;
    
    /**
     * Builds a configuration proxy for the subsystem described by {@code subsystemDesc}.
     * @param subsystemDesc 
     */
    public LocalConfigurationProxy(SubsystemDescription subsystemDesc){
        this.writerProvider = new WriterProvider() {
            @Override
            public PrintWriter getPrintWriter( String baseName ) throws IOException {
                
                //Check if the properties file already exists in the Bootstrap Environment
                String pathInBootstrap = BootstrapResourceUtils.getPathOfPropertiesFileInUserResourceDirectories(baseName);
                //If it does not exist...
                if ( pathInBootstrap == null ) {
                    // then check if a there is a Bootstrap user defined directory in which to write it.
                    String topMostUserDirectory = BootstrapResourceUtils.getTopUserResourceDirectory();
                    //If there is not Bootstrap user Directory defined.....
                    if ( topMostUserDirectory == null ) {
                        //Set the path to org.lsst.ccs.workdir
                        String workdir = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.workdir", "");
                        if ( !workdir.isEmpty() && ! workdir.endsWith(FILE_SEPARATOR)) {
                            workdir += FILE_SEPARATOR;
                        }
                        baseName = workdir+baseName;
                    } else {
                        baseName = topMostUserDirectory+baseName;
                    }
                } else {
                    baseName = pathInBootstrap;
                }
                
                if ( ! baseName.endsWith(".properties") ) {
                    baseName += ".properties";
                }
                
                logger.info("Saving configuration  to "+baseName);
                
                
                return new PrintWriter(baseName, "ISO-8859-1") ;
            }
            
            @Override
            public Properties getConfigurationProperties(String configFileName) throws IOException {
                if ( ! configFileName.endsWith(".properties") ) {
                    configFileName += ".properties";
                }
                Properties configProps = new Properties();
                InputStream propsIs = null;
                propsIs = BootstrapResourceUtils.getBootstrapResource(configFileName);
                if (propsIs == null){
                    throw new IllegalArgumentException("Could not find configuration file : " + configFileName);
                }
                configProps.load(propsIs);
                return configProps;
            }
        };
        this.subsystemDescription = (ASubsystemDescription)subsystemDesc;
        subsystemName = subsystemDescription.getSubsystemName();
        tagName = subsystemDescription.getTag();
        if(System.getProperty("org.lsst.ccs.testcontext","false").equals("true")){
            this.writerProvider = InMemoryWriterProvider.getInstance();
        }
     
    }
    
    /**
     * 
     * @return the set of categories the subsystem has its parameters split into 
     */
    public Set<String> getCategorySet(){
        return subsystemDescription.getCategorySet();
    }
    
    /**
     * Loads the startup configuration represented by taggedCategories, initializes the config profiles
     * and returns a set of values to assign to parameters before startup
     * @param taggedCategories
     * @return 
     */
    public Map.Entry<ConfigurationState, Set<ParameterConfiguration>> getInitialParameterConfigurations(Map<String, String> taggedCategories){
           
        //these Configuration objects: SubsystemDescription and ConfigProfile should be handled by a DAO
        // right now we operate locally so it is a special dummy DAO: MemoryDAO
        CCSCst.LOG_TODO.fine("TODO: DAO at creation of SubsystemDescription should also be able to interact with remote database (switch between local and remote?)") ;
        /*
        since No SubsystemDescription can be used unless it is under the control
        of a DAO this description is passed to a dummy "MemoryDAO"
        */
        MemoryDAO dao = new MemoryDAO();
        dao.saveAbstractDescription(subsystemDescription);
        boolean exceptionThrown = false;
        
        /*
        we create a <TT>ConfigProfile</TT> (where parameters  have values)
        out of the subsystem description
        This ConfigProfile is modified with the configuration Properties
        and saved to the MemoryDAO
        */
        // if no configuration create an empty configuration
        // TO remove
        for(Map.Entry<String,String> entry : taggedCategories.entrySet()){
            Properties configProps = null;
            AConfigProfile profileForCat;
            try {
                configProps = loadPropertiesForCat(entry.getKey(),entry.getValue());
                profileForCat = createConfigProfileOutOfProperties(configProps, entry.getKey(), entry.getValue());
            } catch(IOException ex){
                exceptionThrown = true;
                profileForCat = createConfigProfileOutOfProperties(configProps, entry.getKey(), DEFAULT_CONFIG_NAME);
            }
            configProfileMap.put(profileForCat.getCategoryName(), profileForCat);
//            dao.saveAbstractConfig(newConfigProfile);
        }
        return new AbstractMap.SimpleEntry<ConfigurationState, Set<ParameterConfiguration>>(
                exceptionThrown ? ConfigurationState.UNCONFIGURED : ConfigurationState.CONFIGURED,
                Collections.unmodifiableSet(getParameterConfigurations())
        );
    }
    
    @Override
    public String getTagName() {
        return tagName;
    }
    
    /**
     * Builds a ConfigurationInfo object reflecting the configuration state and
     * ready to be sent on the buses.
     * @param configState
     * @return the {@code ConfigurationInfo} object
     */
    
    @Override
    public synchronized ConfigurationInfo buildConfigurationInfo(ConfigurationState configState, Set<Map.Entry<String, String>> recentChangesNames){
        Map<String,String> tags = new HashMap<>();
        Map<String, Boolean> hasCatChanges = new HashMap<>();
        ArrayList<ConfigurationParameterInfo> parametersView = new ArrayList<>();
        ArrayList<ConfigurationParameterInfo> recentChanges = new ArrayList<>();
        Set<ParameterPath> recentChangesPaths = recentChangesNames.stream()
                .map(entry -> new ParameterPath(entry.getKey(), "", entry.getValue()))
                .collect(Collectors.toSet());
        
        for (ConfigProfile profile : configProfileMap.values()){
            String category = profile.getCategoryName();
            // set tag for category
            tags.put(category, profile.getConfigName());
            hasCatChanges.put(profile.getCategoryName(), profile.isDirty());
            for (ParameterDescription parmDesc : subsystemDescription.getParamDescriptions().stream()
                    .filter(desc -> desc.getCategory().equals(category))
                    .collect(toSet())){
                ConfigurationParameterInfo configParameterInfo;
                ParameterConfiguration parmConfig = profile.fetch(parmDesc);
                if(parmConfig != null){
                    configParameterInfo = new ConfigurationParameterInfo(parmDesc.getPath().toString(),
                            parmDesc.getCategory(), parmDesc.getDefaultValue(), 
                            parmConfig.getConfiguredValue(), parmConfig.getValueAt(PackCst.STILL_VALID), parmConfig.hasChanged());
                } else {
                    configParameterInfo = new ConfigurationParameterInfo(parmDesc.getPath().toString(), 
                            parmDesc.getCategory(), parmDesc.getDefaultValue(), 
                            parmDesc.getDefaultValue(), parmDesc.getDefaultValue(), false);
                }
                parametersView.add(configParameterInfo);
                
                if(recentChangesPaths.contains(parmDesc.getPath())){
                    recentChanges.add(configParameterInfo);
                }
            }
        }
        return new ConfigurationInfo(configState, tagName, tags, hasCatChanges, parametersView, recentChanges);
    }
    
    @Override
    public ViewValue checkForParameterChange(String componentName, String parameterName, Object value) {
        String strValue ;
        if(value instanceof String){
            strValue = (String)  value ;
        } else {
            strValue = TypeUtils.stringify(value) ;
        }
        ParameterPath path = new ParameterPath(componentName,"",parameterName);
        ParameterDescription parameterDescription = subsystemDescription.fetch(path) ;
        if(null == parameterDescription) {
            throw new IllegalArgumentException("incoherent parameter name for " + parameterName + "-> "
                    //+ subsystemDescription.getParamDescriptions());
                    + subsystemName);
        }
        if(parameterDescription.isNotModifiableAtRuntime()) {
            throw new IllegalStateException(" parameter " + parameterName + " not modifiable at runtime");
        }
        Object res = parameterDescription.checkValue(strValue) ;
        return new ViewValue(strValue, res) ;
    }
    
    @Override
    public synchronized void notifyParameterChange(String componentName, String parameterName, String value) {
        ParameterPath parameterPath = new ParameterPath(componentName,"",parameterName);
        //if static
        String category = subsystemDescription.fetch(parameterPath).getCategory();
        configProfileMap.get(category).temporaryChangeConfigurationValue(parameterPath.toString(), System.currentTimeMillis(), value, true);
    }
    
    @Override
    public synchronized void notifyUncheckedParameterChange(String componentName, String parameterName, Object value) {
        ParameterPath parameterPath = new ParameterPath(componentName,"",parameterName);
        
        String strValue ;
        if( value instanceof String) {
            strValue = (String)  value ;
        } else {
            strValue = TypeUtils.stringify(value) ;
        }
        String category = subsystemDescription.fetch(parameterPath).getCategory();
        configProfileMap.get(category).temporaryChangeConfigurationValue(parameterPath.toString(), System.currentTimeMillis(), strValue, false);
    }
    
    /**
     * Tagged categories are saved under a new tag name.
     * Other parameters are left unchanged, the configuration context remains
     * active.
     * @param taggedCategories
     */
    @Override
    public synchronized void saveChangesForCategoriesAs(Map<String,String> taggedCategories) throws ConfigurationServiceException {
        for(Map.Entry<String,String> taggedCategory : taggedCategories.entrySet()){
            try {
                PrintWriter printWriter = writerProvider.getPrintWriter(ConfigUtils.baseNameFromNames(taggedCategory.getValue(), tagName, taggedCategory.getKey()));
                for (ParameterConfiguration parameter : configProfileMap.get(taggedCategory.getKey()).getModifiedParameters()){
                    String currentValue = parameter.getValueAt(PackCst.STILL_VALID);
                    boolean commentOut = currentValue.equals(parameter.getDescription().getDefaultValue());
                    printWriter.println(parameter.getDescription().toPropertyString(currentValue, commentOut));
                }
                printWriter.flush();
                printWriter.close();
                saveModifications(taggedCategory.getKey(), taggedCategory.getValue());
            } catch(IOException ex){
                throw new ConfigurationServiceException("configuration service unavailable", ex);
            }
        }
    }
    
    @Override
    public Map<String,String> getTaggedCategoriesForCats(Set<String> categories){
        Map<String,String> res = new HashMap<>();
        for (String category : categories){
            res.put(category, configProfileMap.get(category).getConfigName());
        }
        return res;
    }
    
    
    private synchronized void saveModifications(String category, String configName){
        AConfigProfile oldProfile = configProfileMap.get(category);
        AConfigProfile newProfile = new AConfigProfile(oldProfile, false, configName);
        configProfileMap.put(category, newProfile);
    }
    
    /**
     * Modification occurring in one of the tagged categories are saved.
     * Other changes are left untouched.
     * @param taggedCategories
     */
    public synchronized void saveModifications(Map<String, String> taggedCategories){
        for (Map.Entry<String, String> entry : taggedCategories.entrySet()){
            saveModifications(entry.getKey(), entry.getValue());
        }
    }
    
    /**
     * Returns the current value of all parameters of a given configurable component
     * @param componentName the name of the configurable component
     * @return a map from parameter name to its current value.
     */
    public Map<String, ViewValue> getCurrentValuesForComponent(String componentName){
        return subsystemDescription.getParamDescriptionMap().entrySet().stream()
                .filter(entry -> entry.getKey().getComponentName().equals(componentName))
                .collect(Collectors.toMap(entry -> entry.getKey().getParameterName(),
                        entry -> {
                            String type = subsystemDescription.fetch(entry.getKey()).getTypeName();
                            String stringVal = configProfileMap.get(entry.getValue().getCategory()).getValueAt(entry.getKey(), PackCst.STILL_VALID);
                            return new ViewValue(stringVal, Constraints.buildObject(type, stringVal));
                        }));
    }
    
    /**
     * Loads properties files for one category only
     * @param categoryName the category
     * @param configName the configuration name for the specified category
     * @throws IOException if the configuration does not exist, except if configName
     * is the default configuration.
     * @return
     */
    private synchronized Properties loadPropertiesForCat(String categoryName, String configName) throws IOException {
        Properties props = new Properties();
        try {
            props.putAll(writerProvider.getConfigurationProperties(ConfigUtils.baseNameFromNames(configName, tagName,categoryName)));
        } catch (IllegalArgumentException ex){
            if (!configName.equals(DEFAULT_CONFIG_NAME)){
                throw ex;
            }
        }
        return props;
    }
    
    public void setWriterProvider(WriterProvider writerProvider) {
        this.writerProvider = writerProvider;
    }
    
    public WriterProvider getWriterProvider(){
        return writerProvider;
    }
    
    public boolean isDirty(){
        return configProfileMap.values().stream()
                .map(ConfigProfile::isDirty)
                .reduce(false, (a,b) -> a||b);
    }
    
    @Override
    public Map<String, Properties> loadCategories(Map<String,String> taggedCategories) throws ConfigurationServiceException {
        if (taggedCategories.isEmpty()) return Collections.EMPTY_MAP;
        Map<String, Properties> res = new HashMap<>();
        for(Map.Entry<String,String> entry : taggedCategories.entrySet()){
            try {
                Properties configProps = loadPropertiesForCat(entry.getKey(), entry.getValue());
                AConfigProfile newConfigProfile = createConfigProfileOutOfProperties(configProps, entry.getKey(), entry.getValue());
                Map<String, Properties> changesForCategory = buildChangesBetweenProfiles(configProfileMap.get(entry.getKey()), newConfigProfile);
                
                for(Map.Entry<String, Properties> e : changesForCategory.entrySet()){
                    Properties changesForComponent = res.get(e.getKey());
                    if (changesForComponent == null){
                        changesForComponent = new Properties();
                        res.put(e.getKey(), changesForComponent);
                    }
                    changesForComponent.putAll(e.getValue());
                }
            } catch (IOException ex){
                throw new ConfigurationServiceException("configuration service problem", ex);
            }
        }
        return res;
    }
    
    /**
     * Returns a map of components and their corresponding changes to switch from
     * oldProfile to newProfile.
     * @param oldProfile
     * @param newProfile
     * @return a map of changes
     */
    private Map<String, Properties> buildChangesBetweenProfiles(ConfigProfile oldProfile, ConfigProfile newProfile){
        // map the name of module to the list of parameters modified by newProfile for this module
        Set<? extends ParameterConfiguration> newChanges = newProfile.getModifiedParameters();
        // map the name of module to the list of parameters modified by oldProfile for this module
        Set<? extends ParameterConfiguration> oldChanges = new HashSet<>(oldProfile.getModifiedParameters());
        
        Map<String, Properties> res = new HashMap<>();
        
        for (ParameterConfiguration parmConfig : newChanges){
            // The new changes overrides the old change : we remove it
            oldChanges.remove(parmConfig);
            ParameterPath path = parmConfig.getPath();
            String parameterName = path.getParameterName();
            // The new value to assign
            String value = parmConfig.getConfiguredValue();
            String oldValue = oldProfile.getValueAt(path, PackCst.STILL_VALID);
            //TODO: some problems if strings are not exactly the same! (though the values are equals! examples : maps , unordered lists, floating point)
            // change the test
            if (oldValue.equals(value)) {
                continue;
            }
            if (parmConfig.getDescription().isNotModifiableAtRuntime()) {
                throw new IllegalArgumentException(path + "not modifiable at runtime");
            }
            
            
            Properties changesForComponent = res.get(path.getComponentName());
            if (changesForComponent == null){
                changesForComponent = new Properties();
                res.put(path.getComponentName(), changesForComponent);
            }
            changesForComponent.setProperty(parameterName, value);
        }
        // Process the remaining parameters in oldChanges : set to their default value
        for (ParameterConfiguration parmConfig : oldChanges){
            ParameterPath path = parmConfig.getPath();
            String parameterName = path.getParameterName();
            String defaultValue = parmConfig.getDescription().getDefaultValue() ;
            if (parmConfig.getDescription().isNotModifiableAtRuntime()) {
                // TODO : throw exception or ignore ?
                throw new IllegalArgumentException(path + "not modifiable at runtime");
            }
            
            Properties changesForComponent = res.get(path.getComponentName());
            if (changesForComponent == null){
                changesForComponent = new Properties();
                res.put(path.getComponentName(), changesForComponent);
            }
            changesForComponent.setProperty(parameterName, defaultValue);
        }
        return res;
    }
    
    private AConfigProfile createConfigProfileOutOfProperties(Properties configProps, String categoryName, String configName){
        AConfigProfile res = new AConfigProfile(subsystemDescription, categoryName, configName);
        if (configProps != null){
            try{
                boolean exceptionThrown = false;
                for (String name : configProps.stringPropertyNames()){
                    try{
                        res.addParameterConfiguration(name, configProps.getProperty(name));
                    } catch (Exception ex){
                        exceptionThrown =true;
                    }
                }
                if (exceptionThrown) throw new IllegalArgumentException("Some parameters were not modifiable, invoke getReConfigurationFailures");
            } catch (IllegalArgumentException exc) {
                List<String> messages = res.reportFailures();
                throw new IllegalArgumentException(" configuration errors :"+ messages) ;
            }
        }
        return res;
    }
    
    public ConfigProfile getProfileForCat(String categoryName){
        return configProfileMap.get(categoryName);
    }
        
    private Set<ParameterConfiguration> getParameterConfigurations(){
        Set<ParameterConfiguration> res = new HashSet<>();
        for(String category : subsystemDescription.getCategorySet()){
            res.addAll(configProfileMap.get(category).getParameterConfigurations());
        }
        return res;
    }
    
}
