package org.lsst.ccs.config;

//TODO : important! embedded fails may not work: check

//TODO: provide facade to factories?

import org.lsst.ccs.utilities.exc.BundledException;

import java.io.Serializable;
import java.util.*;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;

/**
 * implements complex strategies on top of the DBInterface.
 * Dues to cache problems there should be only one instance of this class
 * operating on the network.
 * services are to be addressed remotely if needed.
 *
 * @author bamade
 */
// Date: 13/04/12

public class ConfigurationFacade {
    /**
     * data access object
     */
    private DBInterface dao;
    /**
     * listens to deprecation events (can be null).
     */
    private DeprecationListener deprecationListener;

    private static final Integer ZERO = Integer.valueOf(0);
    /**
     * technical value to deal with the call stack of transactions
     */
    //int stackTransact;
    ThreadLocal<Integer> stackState = new ThreadLocal<Integer>() {
        public Integer initialValue() {
            return ZERO;
        }
    };
    /**
     * fair lock
     */
    ReentrantLock lock = new ReentrantLock(true);

    /**
     * creates a facade object on top of a Data Access Object that implements the base services.
     *
     * @param dao
     */
    public ConfigurationFacade(DBInterface dao) {
        this.dao = dao;
    }

    public void setDeprecationListener(DeprecationListener deprecationListener) {
        this.deprecationListener = deprecationListener;
    }


    ///////////////////////////// internal utilities

    /**
     * internal (non public) method to handle a transaction stack
     */
    synchronized void push() {
        Integer level = stackState.get();
        if (level == ZERO) {
            // blocks
            //System.out.println("----> BEGIN");
            lock.lock();
            dao.begin();
        }
        stackState.set(level + 1);
        //System.out.println("------> level -> " + (level +1) + " thread :" +Thread.currentThread());
        /* old version
        if (stackTransact == 0) dao.begin();
        stackTransact++;
        */
    }

    /**
     * internal (non public) method to handle a transaction stack
     */
    synchronized void pop() {
        Integer level = stackState.get();
        if (level == ZERO) {
            throw new IllegalStateException("dao transaction stack empty");
        }
        level--;
        //System.out.println("------> level -> " + (level ) + " thread :" + Thread.currentThread());
        stackState.set(level);
        if (level == ZERO) {
            //System.out.println("-----> END");
            dao.end();
            lock.unlock();
        }

        /*
        if (stackTransact == 0) {
            throw new IllegalStateException("dao transaction stack empty");
        }
        stackTransact--;
        if (stackTransact == 0) dao.end();
        */
    }

    /**
     * deals with exceptions
     *
     * @param exc
     */

    synchronized PersistenceLayerException failNThrow(Exception exc)  {
        PackCst.CURLOG.log(Level.SEVERE, "", exc);
        Integer level = stackState.get();
        if (level == ZERO) {
            throw new IllegalStateException("dao transaction stack empty");
        }
        level--;
        stackState.set(level);
        if (level == ZERO) {
            if(exc instanceof PersistenceLayerException) {
                dao.fail(exc.getCause());
            } else {
                dao.fail(exc) ;
            }
            lock.unlock();
        }
        if(exc instanceof PersistenceLayerException) {
            return (PersistenceLayerException)exc ;
        }
        return new PersistenceLayerException(exc) ;
    }

    //////////////////////////// external utilities

    /**
     * this class is used by methods that modify the state of the database:
     * <UL>
     *     <LI> the registration of a <TT>SubsystemDescription</TT> will create a new
     *     "default" <TT>ConfigProfile</TT> and if there was a previous version of the same
     *     description then all profiles pointing to this old version will be returned in the <TT>notRegistered</TT> field</LI>
     *     <LI> the "repair" method will take such a <TT>Profiles</TT> object and
     *     will try to create a result report with registered and unregistered profiles</LI>
     * </UL>
     */
    public static class Profiles implements Serializable {
        final List<ConfigProfile> registered = new ArrayList<>();
        final List<ConfigProfile> notRegistered = new ArrayList<>();
        BundledException registrationExceptions;

        public Profiles() {
        }

        public List<ConfigProfile> getRegistered() {
            return registered;
        }

        public List<ConfigProfile> getNotRegistered() {
            return notRegistered;
        }

        public BundledException getRegistrationExceptions() {
            return registrationExceptions;
        }

        public void addRegistered(ConfigProfile conf) {
            this.registered.add(conf) ;
        }
        public void addNotRegistered(ConfigProfile conf) {
            this.notRegistered.add(conf) ;
        }

        public boolean hasUnRegisteredProfiles() {
            return notRegistered.size() > 0 ;
        }

        public Profiles clear() {
            registered.clear();
            notRegistered.clear();
            registrationExceptions = null ;
            return this;
        }

        @Override
        public String toString() {
            final StringBuffer sb = new StringBuffer("Profiles{");
            sb.append("registered=").append(registered);
            sb.append(", notRegistered=").append(notRegistered);
            sb.append(", registrationExceptions=").append(registrationExceptions);
            sb.append('}');
            return sb.toString();
        }
    }

    /// New (2014-2015) more global utilities

    // from a description File or a Descriptive Node -> create subsystem description + default configuration and register those in the database. should tell us if "repairs" are needed

    // entering engineering mode -> create a new temporary configuration register it (check if a new run registration is needed , this means that there should be a boolean to mark as a configuration change not a new run)

    // dropping engineering mode modifications -> back to previous configuration ( run registration?)

    // validating a new Profile after Engineering -option 1: is new current profile - option2: back to previous profile

    // change configuration at runtime

    ///////////////  subsystem descriptions and database access

    /**
     * registers a new (and complete) SubsystemDescription to the persistence Layer.
     * if a previous SubsystemDescription with same subsystemName AND tag is present
     * then it is pushed in history and replaced by the new one as a subsystem which is "alive".
     * <p/>
     * To avoid side effects always write code like this:
     * <PRE>
     * description = facade.registerSubsystemDescription(description);
     * </PRE>
     * <p/>
     * if trying to save a Subsystem description already under DB management this method does nothing.
     *
     * @param newDescription a new SubsystemDescription with all ParameterDescriptions (this object is managed and modified
     *                       by the persistence layer)
     * @param results (side effect) a <TT>Profiles</TT> object to be populated by the invocation
     * @return an unmodifiable version of the subsystem description
     * @throws PersistenceLayerException
     * @throws IllegalArgumentException  if argument not a "new" description (generated by the factories)
     */

    public SubsystemDescription registerSubsystemDescription(SubsystemDescription newDescription, Profiles results) throws PersistenceLayerException {
        if (!(newDescription instanceof ASubsystemDescription)) {
            //TODO : transform or exception? exception preferred
            throw new IllegalArgumentException("not a new description");
        }
        Objects.requireNonNull(results);
        return registerSubsystemDescription((ASubsystemDescription) newDescription, results);
    }

    @Deprecated
    public SubsystemDescription registerSubsystemDescription(SubsystemDescription newDescription) throws PersistenceLayerException {
        if (!(newDescription instanceof ASubsystemDescription)) {
            //TODO : transform or exception? exception preferred
            throw new IllegalArgumentException("not a new description");
        }
        return registerSubsystemDescription((ASubsystemDescription) newDescription, new Profiles());
    }

    /**
     * internal method to register a new desccription.
     * The algorithm is this:
     * <UL>
     * <LI/> tries to get an active description with same name and tag
     * <LI/> if this old description is present deprecates it and its id marks the current description's <TT>PreviousDescriptionID</TT>
     * <LI/> the new description is saved
     * <LI/> ghost description is created and saved
     * <p/>
     * </UL>
     *
     * @param newDescription
     * @return
     * @throws PersistenceLayerException
     */
    SubsystemDescription registerSubsystemDescription(ASubsystemDescription newDescription, Profiles results) throws PersistenceLayerException {
        // key is name + tag
        String subsystemName = newDescription.getSubsystemName();
        String tag = newDescription.getTag();
        SubsystemDescription previous = null;
        // if key exists deprecates the old one
        push();
        try {
            ASubsystemDescription oldOne = dao.getActiveSubsystemDescription(subsystemName, tag);
            long previousDescriptionID = 0L;
            if (oldOne != null) {
                if (oldOne.getId() == newDescription.getId()) {
                    //TODO: modifications of comments in ParameterDescriptions? check for boolean
                    PackCst.CURLOG.warning("trying to save an existing subsystem description : doing nothing");
                    //System.out.println("trying to save an existing subsystem description : doing nothing");
                    pop();
                    return newDescription;
                }
                previous = deprecateSubsystemDescription(oldOne, results);
                previousDescriptionID = previous.getId();
                newDescription.setPreviousDescriptionID(previousDescriptionID);
            }
            // creation of ghost and alive objects
            dao.saveSubsystemDescription(newDescription);
            //TODO: eliminate this request and have the modified newDescription instead
            ASubsystemDescription registered = dao.getActiveSubsystemDescription(subsystemName, tag);
            GhostSubsystemDescription ghost = new GhostSubsystemDescription(registered);
            dao.saveGhostDescriptions(ghost);

            // TODO : check if designer level is ok
            // REVIEW : here we register a default config profile for each category
            for (String category : newDescription.getCategorySet()){
                AConfigProfile defaultProfile = new AConfigProfile(newDescription,
                        category, "") ;
                registerConfigProfile(defaultProfile) ;
                results.addRegistered(defaultProfile);
            }
            pop();
            //TODO : is clone() necessary ? check the idea
            ASubsystemDescription res = newDescription.clone() ;
            return res;
            //return newDescription.clone();
            //return newDescription;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }

    /**
     * Internal code that deprecates a SubsystemDecription. All ConfigProfiles
     * pointing to it are deprecated and <TT>DeprecationListeners</TT>
     * are warned (if present).
     * <p/>
     * Since it is an "internal" method all exceptions are forwarded to the calling codes.
     * </P>
     *
     * @param currentDescriptionInDB a SubsystemDescription that MUST be in the database
     * @return a ghost subsystem description
     */
    SubsystemDescription deprecateSubsystemDescription(ASubsystemDescription currentDescriptionInDB, Profiles results) {
        push();
        GhostSubsystemDescription ghost = dao.getGhostDescription(currentDescriptionInDB.getId());
        // marks the Ghost's end date
        long endDate = System.currentTimeMillis();
        ghost.setEndTimestamp(endDate);
        currentDescriptionInDB.setEndTimestamp(endDate);
        // gets all ConfigProfile -> deprecates them and warn Listener
        Collection<AConfigProfile> profiles = dao.getActiveProfilesForSubsystem(currentDescriptionInDB);
        for (AConfigProfile profile : profiles) {
           ConfigProfile deprecated =  deprecateConfigProfile(ghost, profile);
            if(! "".equals(deprecated.getName())) {
                results.addNotRegistered(deprecated);
            }
        }
        // remove from alive DB
        if (deprecationListener != null) {
            //DONE: better register past one than current!
            //deprecationListener.subsystemDeprecating(currentDescriptionInDB);
            //TODO: clone ghost?
            deprecationListener.subsystemDeprecating(ghost);
        }
        dao.deleteActiveSubsystemDescription(currentDescriptionInDB);
        pop();
        return ghost;
    }

    /**
     * Deprecates a Subsystem description already in the database.
     * If it's not there nothing happens!
     *
     * @param subsystemName
     * @param tag           use "" if there is no tag
     * @return the deprecated description (or null if none was deprecated)
     * @throws PersistenceLayerException
     */

    public SubsystemDescription deprecateSubsystemDescription(String subsystemName, String tag, Profiles results) throws PersistenceLayerException {

       if(tag == null)  tag = "" ;
        push();
        try {
            SubsystemDescription res = null;
            ASubsystemDescription oldOne = dao.getActiveSubsystemDescription(subsystemName, tag);
            if (oldOne != null) {
                res = deprecateSubsystemDescription(oldOne, results);
            }
            pop();
            return res;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }

    @Deprecated
    public SubsystemDescription deprecateSubsystemDescription(String subsystemName, String tag) throws PersistenceLayerException {
        //TODO : check for preconditions (if tag is null then "")
        push();
        try {
            SubsystemDescription res = null;
            ASubsystemDescription oldOne = dao.getActiveSubsystemDescription(subsystemName, tag);
            if (oldOne != null) {
                res = deprecateSubsystemDescription(oldOne, new Profiles());
            }
            pop();
            return res;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }

    /**
     * returns the active subsystem description with name and tag.
     *
     * @param name
     * @param tag
     * @return null if none is found
     * @throws PersistenceLayerException
     */
    public SubsystemDescription getActiveSubsystemDescription(String name, String tag) throws PersistenceLayerException {
        push();
        try {
            SubsystemDescription res = dao.getActiveSubsystemDescription(name, tag);
            pop() ;
            return res ;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }
    ///////////// Profiles and database access

    /**
     * registers a ConfigProfile in the database. if one with the same name and tag
     * exists it is pushed in history and a DeprecationListener is warned.
     * <p/>
     * Beware of side effect: better assign the returned object to the reference holding the argument.
     * <p/>
     * if trying to save a Config Profile already under DB management this method does nothing.
     *
     * @param newProfile
     * @return unmodifiable copy of argument
     * @throws PersistenceLayerException
     */


    public ConfigProfile registerConfigProfile(ConfigProfile newProfile) throws PersistenceLayerException {
        if (!(newProfile instanceof AConfigProfile)) {
            throw new IllegalArgumentException("deprecated Profile");
        }
        return registerConfigProfile((AConfigProfile) newProfile);
    }

    /**
     * internal method to register a profile:
     * <UL>
     * <LI/> if profile  with same name and tag is found  it is deprecated and a link
     * is established between profiles . If they refer to distinct subsystem an exception is fired.
     * </UL>
     *
     * @param newProfile
     * @return
     * @throws PersistenceLayerException
     * @throws IllegalArgumentException  if there is a name clash between subsystem referenced by the new and old profile
     */
    //TODO: BUG! you should be able to register a Profile with same name but with different subsystem (for instance the default profile!)
    ConfigProfile registerConfigProfile(AConfigProfile newProfile) throws PersistenceLayerException {
        push();
        try {
            ConfigProfile previous = null;
            long previousConfigID = 0L;
            String subsystemName = newProfile.getSubsystemName() ;
            String name = newProfile.getName();
            String tag = newProfile.getTag();
            AConfigProfile oldProfile = (AConfigProfile) getActiveConfigProfile(subsystemName,name, tag);
            if (oldProfile != null) {
                if (oldProfile.getId() == newProfile.getId()) {
                    //TODO: modifications of remarks? check for modification boolean
                    PackCst.CURLOG.warning("trying to save an existing Config profile : doing nothing");
                    //System.out.println("trying to save an existing Config profile : doing nothing");
                    pop();
                    return newProfile;
                }
                // trying to detect an anomaly: the config in the database is refernecinf a different subsystem
                if (!newProfile.getSubsystemName().equals(oldProfile.getSubsystemName())) {
                    throw new IllegalArgumentException("Name clash :  trying to create a configProfile with same name and different subsystem "
                            + name + "/" + tag + " already exists for subsystem " + oldProfile.getSubsystemName());
                }
                GhostSubsystemDescription ghost = dao.getGhostDescription(oldProfile.getSubsystemId());
                previous = deprecateConfigProfile(ghost, oldProfile);
                //TODO: update differences between values
                previousConfigID = previous.getId();
                newProfile.setPreviousConfigID(previousConfigID);
            }
            /// what if subsystem exists no more ?
            // NOTE TO DEVELOPERS : though this looks like a duplicate
            // of text in AConfigProfile constructor
            // it doesn't: because when used in remote mode there may be
            // an object with wrong values on the client code
            //TODO : redo the tryc/catch architecture this is wrong!
            long idSubs = newProfile.getSubsystemId();
            //List listSubs = dao.simpleHQLRequest("from ASubsystemDescription where id = " + idSubs);
            ASubsystemDescription registered = dao.getActiveSubsystemDescription(idSubs) ;
            if (registered == null) {
                throw new IllegalArgumentException(" No such living subsystem description in database " +
                        newProfile.getSubsystemDescription());
            }

            dao.saveConfigProfile(newProfile);
            pop();
            //DONE: return clone?
            return newProfile.clone();
            //return newProfile ;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }

    /**
     * internal method for deprecating a ConfigProfile
     *
     * @param ghost
     * @param currentProfileInDB
     * @return deprecated ConfigProfile
     */
    ConfigProfile deprecateConfigProfile(GhostSubsystemDescription ghost, AConfigProfile currentProfileInDB) {
        push();
        PastConfigProfile past = new PastConfigProfile(ghost, currentProfileInDB);
        dao.savePastProfile(past);
        if (deprecationListener != null) {
            //DONE: better register past one than current!
            deprecationListener.configProfileDeprecating(past);
        }
        dao.deleteActiveConfigProfile(currentProfileInDB);
        pop();
        return past;
    }

    /**
     * Deprecates a ConfigProfile. If not in database does nothing.
     * <p/>
     * <B>Important note</B>: deprecating a configProfile to provide later a replacement will
     * break the link between  ConfigProfile (previousConfigID) so it is preferable to create
     * directly a new replacement that will deprecate the previous one and update the link.
     *
     * @param name
     * @param tag
     * @return the deprecated ConfigProfile (or null if there was none)
     * @throws PersistenceLayerException
     */

    public ConfigProfile deprecateConfigProfile(String subsystemName, String name, String tag) throws PersistenceLayerException {
        push();
        try {
            ConfigProfile res = null;
            AConfigProfile currentProfile = (AConfigProfile) getActiveConfigProfile(subsystemName, name, tag);
            if (currentProfile != null) {
                GhostSubsystemDescription ghost = dao.getGhostDescription(currentProfile.getSubsystemId());
                res = deprecateConfigProfile(ghost, currentProfile);
            }
            pop();
            return res;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }

    /**
     * returns an active ConfigProfile with name and tag
     * @param name
     * @param tag
     * @return null if none found
     * @throws PersistenceLayerException
     */
    public ConfigProfile getActiveConfigProfile(String subsystemName, String name, String tag) throws PersistenceLayerException {
        push();
        try {
            AConfigProfile res = dao.getActiveConfigProfile(subsystemName, name, tag);
            pop();
            return res;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }



    /**
     * changes a parameter value during engineering session
     * @param profile the configuration profile the parameter belongs to
     * @param path the path of the parameter on which to perform the change
     * @param timeStamp the timestamp of the change
     * @param value the new value to affect to the parameter
     * @return the modified {@code ParameterConfiguration}
     * @throws PersistenceLayerException
     */
    public ParameterConfiguration modifyParmDuringEngineering(ConfigProfile profile, String path, long timeStamp, String value) throws PersistenceLayerException {
        if(! (profile instanceof  AConfigProfile)) {
            throw new IllegalArgumentException(" profile not regsitered in database") ;
        }
//        AConfigProfile confProfile = (AConfigProfile) profile ;
//        AParameterConfiguration parmConf = (AParameterConfiguration)confProfile.fetch(path) ;
        // todo :make a copy ?
        /*
        if(! parmConf.isCopy()) {

            parmConf = new AParameterConfiguration() ;
        } */
        AParameterConfiguration parmConf = (AParameterConfiguration)profile.temporaryChangeConfigurationValue(path, timeStamp, value, true);

        parmConf = (AParameterConfiguration) engineerParmConfig(parmConf);
        return parmConf ;
    }

    /**
     * registration of a modified parameter during an engineering session in the
     * database
     * @param parameterConfiguration
     * @return the modified {@code ParameterConfiguration}
     * @throws PersistenceLayerException
     * @throws IllegalArgumentException if the parameterConfiguration is not a regsitered active parameter of if
     * the configuration is not in engineering mode
     */
    public ParameterConfiguration engineerParmConfig(ParameterConfiguration parameterConfiguration) throws PersistenceLayerException {
        if (!(parameterConfiguration instanceof AParameterConfiguration)) {
            throw new IllegalArgumentException("not a registered active  parameterConfiguraiton");
        }
        AParameterConfiguration config = (AParameterConfiguration) parameterConfiguration;
        if (config.getId() == 0L) {
            throw new IllegalArgumentException("not a registered parameterConfiguraiton");
        }
        //TODO: check if this test is correct in all cases of isCopy!
        /*
        if (!config.isCopy()) {
            throw new IllegalArgumentException("parameterConfiguration not in engineering mode");
        } */
        push();
        try {
            //TODO: test thoroughly may not work under some conditions !!!
            dao.modifyParmConfig(config);
            pop();
            return config;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }

    }


    /**
     * the run and end registrations are based on messages received from a subsystem.
     * As much as possible the code tries to deal with lost messages (there is only one subsystem with a given name
     * running at any time):
     * <UL>
     * <LI> if an end message is lost: next time a subsystem with same name is started
     * the previous run is searched, if this previous run has no end date it is "guessed" that is an arbitrary date
     * is set just before the current run.
     * <LI> if a start message is lost  (much more unlikely) an artificial start date is created jsut after
     * the latest run with no "next data"
     * <LI> if both message are lost there is nothing we can do to guess the startDate of a subsystem with differnet configuration
     * <p/>
     * </UL>
     *
     * @param subsystemName
     * @param configName
     * @param tag
     * @param startTime
     * @throws PersistenceLayerException
     */
    public void registerRun(String subsystemName, String configName, String tag, long startTime) throws PersistenceLayerException {
        // consistency of the algorithm supposes there are not two concurrent same subsystem running
        //gets latest run with same subsystemName
        // that is Timenext is zero
        push();
        try {
            List list = dao.simpleHQLRequest(
                    "from RunHistory where subsystemName = '" + subsystemName
                            + "' and timeNext = 0");
            if (list.size() != 0) {
                if (list.size() > 1) {
                    PackCst.CURLOG.warning( "multiple runs with no next run!");
                    //TODO repair!
                }
                //change its timenext (and possible their endTimestampLimit if it is STILL_VALID : issu a warning)
                RunHistory previousRun = (RunHistory) list.get(0);
                previousRun.setTimeNext(startTime);
                if (previousRun.getEndTimestampLimit() == PackCst.STILL_VALID) {
                    previousRun.setEndTimestampLimit(startTime - 1);
                    previousRun.setEndGuessed(true);
                }
                // save   previousRun?
            }
            RunHistory run = new RunHistory(subsystemName, configName, tag, startTime);
            // save the current RunHistory object
            dao.saveRun(run);
            pop();
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }

    /**
     * marks the end of a run.
     * See the documentation of <TT>registerRun</TT>
     * @param subsystemName
     * @param configName
     * @param tag
     * @param endTime
     * @throws PersistenceLayerException
     */
    public void endRun(String subsystemName, String configName, String tag, long endTime) throws PersistenceLayerException {
        push();
        if(configName == null) { configName="" ; }
        if(tag == null) { tag = "" ; }
        try {
            List<RunHistory> list = (List<RunHistory>) dao.simpleHQLRequest(
                    "from RunHistory where subsystemName = '" + subsystemName
                            + "' and endTimestampLimit = " + PackCst.STILL_VALID);
            if (list.size() == 0) {
                // means that a start message was lost
                // so gets previous and create an artificial start
                list = (List<RunHistory>) dao.simpleHQLRequest(
                        "from RunHistory where subsystemName = '" + subsystemName
                                + "' and timeNext = 0");
                if (list.size() != 0) {
                    if (list.size() > 1) {
                        PackCst.CURLOG.warning( "multiple runs with no next run!");
                        //TODO repair!
                    }
                    RunHistory previousRun = (RunHistory) list.get(0);
                    // we KNOW it is not STILL_VALID
                    long previousEnd = previousRun.getEndTimestampLimit();
                    long fakeStart = previousEnd + 1;
                    previousRun.setTimeNext(fakeStart);
                    RunHistory run = new RunHistory(subsystemName, configName, tag, fakeStart);
                    run.setStartGuessed(true);
                    run.setEndTimestampLimit(endTime);
                    dao.saveRun(run);
                }

            } else {
                // the try to reconstitute a run from other runs that have no next
                // what happens if other element in List without next?
                for (RunHistory run : list) {
                    // check if config and tag match
                    String runConfigName = run.getConfigurationName() ;
                    if(runConfigName == null) {runConfigName = ""; }
                    String runTagName = run.getTag() ;
                    if(runTagName == null) {runTagName = "" ; }
                    if (runConfigName.equals(configName) && runTagName.equals(tag)) {
                        run.setEndTimestampLimit(endTime);
                    } else {
                        // if list size > 1 end everything ?
                        PackCst.CURLOG.log(Level.WARNING, "run with no end date", run);
                        //TODO: SHOULD NOT HAPPEN?
                    }

                }
            }
            pop();
        } catch (Exception exc) {
            throw failNThrow(exc);
        }

    }

    /**
     * returns the ConfigProfile active when a subsystem was running at that date.
     * <BR/>
     * <B> PROBLEM</B>: a subsystem may be running without a configProfile. so beware
     * @param subsystemName
     * @param date
     * @return the {@code ConfigProfile} running at the specified date
     * @throws PersistenceLayerException
     */
    public ConfigProfile getConfigRunningAt(String subsystemName, long date) throws PersistenceLayerException {
        //get all RunHistory with same subsystemName , startDate < date and endDate > date
        push();
        try {
            //TODO: sort by database? with a MAX?
            List listRun = dao.simpleHQLRequest("from RunHistory where subsystemName = '"
                    + subsystemName + "' and startTimeStamp < " + date + " and endTimeStampLimit > " + date + " ");
            int numberRuns = listRun.size();
            if (numberRuns != 0) {
                if (numberRuns > 1) {
                    //anomaly but possible
                    PackCst.CURLOG.log(Level.WARNING, "multiple runs in same time span", listRun);
                    Collections.sort(listRun, new Comparator() {
                        public int compare(Object o, Object o1) {
                            Long longA = ((RunHistory) o).getStartTimestamp();
                            Long longB = ((RunHistory) o1).getStartTimestamp();
                            return longA.compareTo(longB);
                        }
                    });
                }
                RunHistory lastRun = (RunHistory) listRun.get(numberRuns - 1);
                //
                ConfigProfile res = getConfigValidAt(subsystemName, lastRun.getConfigurationName(), lastRun.getTag(), date);
                pop() ;
                return res ;
            }

            pop();
            return null;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }

    /**
     * tries to get a ConfigProfile which was valid at a given date.
     * Since there is only one Config profile active with a given name at a given date
     * there is only one parameter needed (there is no need for a subsystem name)
     *
     * @param configName
     * @param configTag
     * @return a {@code ConfigProfile} that was valid at the given date
     * @throws PersistenceLayerException
     */

    public ConfigProfile getConfigValidAt(String subsystemName, String configName, String configTag, long date) throws PersistenceLayerException {
        Objects.requireNonNull(subsystemName);
        Objects.requireNonNull(configName);
        Objects.requireNonNull(configTag);
        push();
        try {
            List list = dao.simpleHQLRequest("from AConfigProfile where subsystemName  = '" + subsystemName+"' and name ='" + configName + "' and tag ='" + configTag
                    + "' and startTimestamp <= " + date + " ");
            if (list.size() != 0) {
                pop();
                return (ConfigProfile) list.get(0);
            }
            list = dao.simpleHQLRequest("from PastConfigProfile where subsystemName = '" + subsystemName +"' and name ='" + configName + "' and tag ='" + configTag
                    + "' and startTimestamp <= " + date + " and endTimestamp >= " + date + " ");
            if (list.size() != 0) {
                pop();
                return (ConfigProfile) list.get(0);
            }
            pop();
            return null;
        } catch (Exception exc) {
            throw failNThrow(exc);

        }
    }


    /**
     *  returns the value of a parameter for a subsystem at a given date.
     * <P/>
     * <B>BUG:</B> this method is not correct when a "default" subsystem is run without a configuration.
     * Will be changed!
     * @param subsystemName
     * @param parameterPath
     * @param date
     * @return a string representation of the active value for the specified parameter
     * at the given date.
     * @throws PersistenceLayerException
     */
    public String getActiveValueAt(String subsystemName, String parameterPath, long date) throws PersistenceLayerException {
        ConfigProfile profile = getConfigRunningAt(subsystemName, date);
        if (profile == null) {
            PackCst.CURLOG.warning( "no running profile at " + new Date(date) + "for subsystem "
                    + subsystemName);
            return null;
        }
        return profile.getValueAt(parameterPath, date);

    }

    /**
     * returns the value of a parameter at a given time for a given Profile.
     *
     * @param profileName
     * @param profileTag
     * @param parameterPath a String in Path syntax ("componentName//parameterName")
     * @param date
     * @return null if profile or parameter not found
     * @throws PersistenceLayerException
     */

    public String getValueValidAt(String subsystemName ,String profileName, String profileTag, String parameterPath, long date) throws PersistenceLayerException {
        ConfigProfile profile = getConfigValidAt(subsystemName, profileName, profileTag, date);
        return profile.getValueAt(parameterPath, date);
    }


    /**
     * gets a previous ConfigProfile with the same name and tag.
     *
     * @param current
     * @return null if there is none (or if the deprecation was not carried out through a replacement or repair)
     * @throws PersistenceLayerException
     */

    public ConfigProfile getPrevious(ConfigProfile current) throws PersistenceLayerException {
        if (current == null) return null;
        long id = current.getPreviousConfigID();
        if (0L == id) {
            return null;
        }
        push();
        try {
            ConfigProfile res = null;
            List list = dao.simpleHQLRequest("from PastConfigProfile where id = '" + id + "'");
            if (list.size() != 0) {
                res = (ConfigProfile) list.get(0);
            }
            pop();
            return res;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }

    /**
     * gets the next modified ConfigProfile with same name and tag
     *
     * @param current
     * @return null if none or deprecation was not carried out through a replacement or repair
     * @throws PersistenceLayerException
     */

    public ConfigProfile getNext(ConfigProfile current) throws PersistenceLayerException {
        if (current == null) return null;
        long id = current.getId();
        if (0L == id) {
            return null;
        }
        push();
        try {
            ConfigProfile res = null;
            // tODO: if ghost then only one request to operate ....
            // first try if in a new Config
            List list = dao.simpleHQLRequest("from AConfigProfile where previousConfigID = '" + id + "'");
            if (list.size() != 0) {
                res = (ConfigProfile) list.get(0);
            } else {
                list = dao.simpleHQLRequest("from PastConfigProfile where previousConfigID = '" + id + "'");
                if (list.size() != 0) {
                    res = (ConfigProfile) list.get(0);
                }
            }
            pop();
            return res;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }


    /**
     * returns the previous description with same name and tag.
     *
     * @param current
     * @return previous description or null if there was none (or if the previous one was not deprecated through a replacement)
     * @throws PersistenceLayerException
     */

    public SubsystemDescription getPrevious(SubsystemDescription current) throws PersistenceLayerException {
        if (current == null) return null;
        long id = current.getPreviousDescriptionID();
        if (0L == id) {
            return null;
        }
        push();
        try {
            SubsystemDescription res = null;
            List list = dao.simpleHQLRequest("from GhostSubsystemDescription where id = '" + id + "'");
            if (list.size() != 0) {
                res = (SubsystemDescription) list.get(0);
            }
            pop();
            return res;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }

    /**
     * tries to repair automatically a set of <TT>ConfigProfiles</TT> that have been deprecated
     * as the result of a Description change.
     * If everything in a deprecated is compatible with the new Description  it will be changed
     * and registered, otherwise it will be put in the "unRegistered" list of the result.
     * <BR>
     *  A <TT>ConfigProfile</TT> is compatible if:
     *  <UL>
     *      <LI> All Modified parameters are known to the Description and if their value is correct
     *      with the description type and constraints.
     *  </UL>
     *  if its path is unknown or if the value is incompatible the "repair" is rejected.
     * @param description
     * @param requestedProfiles
     * @return another <TT>Profile</TT> object that knows about which ConfigProfiles wee registered and which were not
     * there may be cases where the ConfigProfile is compatible but has not been registered due to databse problems
     * see the  "registrationExceptions" in the resulting Profile object.
     */
    public Profiles repair (SubsystemDescription description, Profiles requestedProfiles)  {
        if(! (description instanceof  ASubsystemDescription)) {
            throw new IllegalArgumentException("no way to repair on  description not managed by databse") ;
        }
        ASubsystemDescription currentDescription = (ASubsystemDescription) description ;
        Profiles res = new Profiles();
        for(ConfigProfile oldProfile: requestedProfiles.getNotRegistered()) {
            AConfigProfile newProfile = new AConfigProfile(currentDescription, oldProfile.getCategoryName(), oldProfile.getConfigName()) ;
            boolean inError = false ;
            for(ParameterConfiguration parmConfig: oldProfile.getModifiedParameters()) {
                String oldValue = parmConfig.getConfiguredValue();
                AParameterDescription parmDesc = (AParameterDescription)currentDescription.fetch(parmConfig) ;
                if(parmDesc == null ) {
                    parmConfig.setReConfigurationFailure(new IllegalArgumentException("no parameter with path: " +parmDesc.getPath()));
                    inError = true ;
                    continue ;
                }
                try {
                    AParameterConfiguration newParmConfig = new AParameterConfiguration(parmDesc, oldValue, true) ;
                    newProfile.addParameterConfigurations(newParmConfig);
                } catch (Exception exc) {
                    inError =true ;
                    parmConfig.setReConfigurationFailure(exc);
                }
            }
            if(inError) {
                res.addNotRegistered(oldProfile);
            } else {
                try {
                    registerConfigProfile(newProfile);
                    res.addRegistered(newProfile);
                } catch (Exception exc) {
                    res.registrationExceptions = new BundledException(exc, res.registrationExceptions) ;
                    res.addNotRegistered(newProfile);
                }
            }
        }
        return res ;
    }
    ///////////// other utilities


    /**
     * forwards an HQLrequest to the DAO.
     * Though this is not yet tested only "from" (SELECT) request should be accepted
     * @param hqlString
     * @return the HQL request
     * @throws PersistenceLayerException
     */
    public List<?> simpleHQLRequest(String hqlString) throws PersistenceLayerException {
        push();
        try {
            List<?> res = dao.simpleHQLRequest(hqlString);
            pop();
            return res;
        } catch (Exception exc) {
            throw failNThrow(exc);
        }
    }

}
