package org.lsst.ccs.config;

import java.lang.reflect.Method;
import java.util.AbstractMap;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.config.utilities.ConfigUtils;
import org.lsst.ccs.description.ComponentLookupService;
import org.lsst.ccs.framework.Configurable;
import org.lsst.ccs.framework.ConfigurationServiceException;
import org.lsst.ccs.utilities.structs.ViewValue;

/**
 * Abstract subclass of subsystems which provides access to the configuration proxy.
 * @author The LSST CCS Team
 */
public abstract class ConfigurableSubsystem extends Subsystem{
    private ConfigurationProxy configurationProxy;
    private final Set<String> categories;
    /**
     * Used only to know the startup state
     */
    private final ConfigurationState initialState;

    /**
     * Maps the name of the configurable component to a Map of new values for
     * parameters that belong to this component.
     */
    private final Map<String, Map<String, ViewValue>> bulkChanges = new ConcurrentHashMap<>();

    public ConfigurableSubsystem(String name, ConfigurationProxy configProxy, ConfigurationState initialState) {
        super(name, AgentInfo.AgentType.WORKER);
        this.configurationProxy = configProxy;
        categories = configProxy.getCategorySet();
        this.initialState = initialState;
    }

    /**
     * @return tag name
     */
    public String getTag() {
        return configurationProxy.getTagName();
    }

    /**
     * @return the configuration proxy
     */
    public ConfigurationProxy getConfigurationProxy() {
        return configurationProxy;
    }

    /**
     * this method should not be used once the proxy is delegated to all components
     * so it should not be public!
     * @param configurationProxy
     */
    protected void setConfigurationProxy(ConfigurationProxy configurationProxy) {
        this.configurationProxy = configurationProxy;
    }

    /**
     * Saves all changes in their current configuration.
     * State switches to Configured is no exception.
     * @throws ConfigurationServiceException if configuration service unavailable
     */
    @Command(description = "Saves all changes in the current configurations", type = Command.CommandType.CONFIGURATION)
    public final void saveAllChanges() throws ConfigurationServiceException {
        saveChangesForCategoriesAsInternal(configurationProxy.getTaggedCategoriesForCats(categories));
    }

    /**
     * Changes made in the specified categories are saved under the current configuration
     * for this category, changes on parameters that belong to other categories
     * are left unchanged.
     * @param categories a list of categories
     * @throws ConfigurationServiceException if configuration service
     * unavailable.
     */
    @Command(description = "Saves the specified categories with a name", type = Command.CommandType.CONFIGURATION)
    public final void saveChangesForCategories(
            @Argument(name = "taggedCategories", description = "A list of categories") String... categories) throws ConfigurationServiceException {
        saveChangesForCategoriesAsInternal(configurationProxy.getTaggedCategoriesForCats(ConfigUtils.parseCategories(this.categories, categories)));
    }

    /**
     * Changes made in the specified categories are saved under the newly specified
     * name for this category, changes on parameters that belong to other categories
     * are left unchanged.
     * @param taggedCategories a list of pairs categoryName:configurationName
     * @throws ConfigurationServiceException
     */
    @Command(description = "Saves the specified categories with a name", type = Command.CommandType.CONFIGURATION)
    public final void saveChangesForCategoriesAs(
            @Argument(name = "taggedCategories", description = "A list of pairs categoryName:configurationName") String... taggedCategories) throws ConfigurationServiceException {
        saveChangesForCategoriesAsInternal(parseConfigurationString(taggedCategories));
    }

    private void saveChangesForCategoriesAsInternal(Map<String, String> taggedCategories) throws ConfigurationServiceException {
        if (taggedCategories.isEmpty()) return;
        String message = "";
        try {
            this.configurationProxy.saveChangesForCategoriesAs(taggedCategories);
        } catch (ConfigurationServiceException ex) {
            message = ex.getMessage();
            throw ex;
        } finally {
            updateStateAndSendStatusConfigurationInfo(message, Collections.EMPTY_SET);
        }
    }

    /**
     * All unsaved changes are dropped, each parameter goes back to the value specified
     * by the current configuration for this category.
     * @throws Exception
     */
    @Command(description = "drop all unsaved changes", type = Command.CommandType.CONFIGURATION)
    public void dropAllChanges() throws Exception {
        dropChangesInternal(categories);
    }

    /**
     * Changes occurring for one of the mentioned categories in categories are dropped,
     * other categories are left untouched.
     * @param categories A list of categories
     * @throws Exception
     */
    @Command(description = "drop unsaved changes for the specified categories", type = Command.CommandType.CONFIGURATION)
    public void dropChangesForCategories(@Argument(name = "categories", description = "A list of categories") String... categories) throws Exception {
        dropChangesInternal(ConfigUtils.parseCategories(this.categories, categories));
    }

    private void dropChangesInternal(Set<String> categoriesSet) throws Exception {
        dropBulkChange();
        Map<String, Properties> changes;
        try {
            changes = configurationProxy.loadCategories(configurationProxy.getTaggedCategoriesForCats(categoriesSet));
        } catch (ConfigurationServiceException ex) {
            throw ex;
        }
        for (Map.Entry<String, Properties> changesForComponent : changes.entrySet()) {
            for (String parameterName : changesForComponent.getValue().stringPropertyNames()) {
                submitChange(changesForComponent.getKey(), parameterName, changesForComponent.getValue().getProperty(parameterName));
            }
        }
        // validation step should be safe in this context
        validateBulkChanges();
        // setting
        Set<Map.Entry<String,String>> processedParameters = bulkChange();
        configurationProxy.saveModifications(configurationProxy.getTaggedCategoriesForCats(categoriesSet));
        updateStateAndSendStatusConfigurationInfo("", processedParameters);
    }

    /**
     * Loads the configuration specified for each category specified in configName
     * and loads the default configuration for the other categories.
     * Changes that are not saved yet are dropped.
     *
     * @param taggedCategories a list of pairs categoryName:configurationName
     * @throws org.lsst.ccs.framework.ConfigurationServiceException
     * @throws Exception
     */
    @Command(description = "loads a new configuration", type = Command.CommandType.CONFIGURATION)
    public void loadConfiguration(
            @Argument(name = "taggedCategories", description = "a list of pairs categoryName:configurationName") String... taggedCategories) throws ConfigurationServiceException, Exception {
        loadCategoriesInternal(ConfigUtils.parseConfigurationStringWithDefaults(categories, taggedCategories));
    }

    /**
     * Loads the configuration specified for each category, parameters that belong
     * to other categories are left unchanged.
     * @param taggedCategories a list of pairs categoryName:configurationName
     * @throws org.lsst.ccs.framework.ConfigurationServiceException
     * @throws Exception
     */
    @Command(description = "loads the configuration for the specified categories", type = Command.CommandType.CONFIGURATION)
    public void loadCategories(
            @Argument(name = "taggedCategories", description = "a list of pairs categoryName:configurationName") String... taggedCategories) throws ConfigurationServiceException, Exception {
        if (taggedCategories.length == 0) {
            return;
        }
        loadCategoriesInternal(parseConfigurationString(taggedCategories));
    }

    private void loadCategoriesInternal(Map<String, String> categoryProfile) throws ConfigurationServiceException, Exception {
        dropBulkChange();
        Map<String, Properties> changes;
        try {
            changes = configurationProxy.loadCategories(categoryProfile);
        } catch (ConfigurationServiceException ex) {
            throw ex;
        }

        for (Map.Entry<String, Properties> changesForComponent : changes.entrySet()) {
            for (String parameterName : changesForComponent.getValue().stringPropertyNames()) {
                submitChange(changesForComponent.getKey(), parameterName, changesForComponent.getValue().getProperty(parameterName));
            }
        }
        // validation step should be safe in this context
        validateBulkChanges();
        // setting
        Set<Map.Entry<String, String>> processedParms = bulkChange();
        
        configurationProxy.saveModifications(categoryProfile);
        
        updateStateAndSendStatusConfigurationInfo("", processedParms);
    }

    /**
     * @return a ConfigurationInfo object depicting the current configuration state 
     */
    @Command(description = "return a ConfigurationInfo object", type = Command.CommandType.CONFIGURATION)
    public ConfigurationInfo getConfigurationInfo() {
        return configurationProxy.buildConfigurationInfo((ConfigurationState) getState(ConfigurationState.class), Collections.EMPTY_SET);
    }

    /**
     * The structure of a Configurable Subsystem may be modular (eg {@link NodeModularSubsystem),
     * in that case changing a parameter requires knowledge of this modular structure.
     *
     * @return a ComponentLookupService which holds the internal structure of the
     * ConfigurableSubsystem it stands for.
     */
    public abstract ComponentLookupService getLookup();

    private Map<String,String> parseConfigurationString(String ... taggedCategories){
        return ConfigUtils.parseConfigurationString(categories, taggedCategories);
    }

    /**
     * Single change of parameter.
     *
     * @param componentName the name of the Configurable
     * @param parameterName the name of the parameter
     * @param value the new value to affect to this parameter
     * @throws java.lang.Exception
     */
    void singleChange(String componentName, String parameterName, Object value) throws Exception {
        ViewValue viewValue = configurationProxy.checkForParameterChange(componentName, parameterName, value);
        Configurable component = (Configurable) getLookup().getComponentByName(componentName);
        updateInternalState("", ConfigurationState.CONFIGURING);
        Map<String, ViewValue> parametersView
                = configurationProxy.getCurrentValuesForComponent(componentName);
        parametersView.put(parameterName, viewValue);

        component.validateBulkChange(parametersView.entrySet().stream()
                .collect(Collectors.toMap(
                                entry -> entry.getKey(),
                                entry -> entry.getValue().getValue()
                        )));

        // Bulk setting
        Map<String, Object> change = new HashMap<>();
        change.put(parameterName, value);
        Map<String, Object> others = component.setBulkParameter(change);
        // Notify the bulk changes to the configuration service
        if (others.isEmpty()) {
            configurationProxy.notifyParameterChange(componentName, parameterName, viewValue.getView());
        } else {
            // Parameter setting with @ConfigChanger
            Method m = component.getEnvironment().getConfigMethodOfComponent(parameterName);
            try {
                m.invoke(component, viewValue.getValue());
            } catch (Exception ex) {
                throw new RuntimeException(ex);
            }
            configurationProxy.notifyParameterChange(componentName, parameterName, viewValue.getView());
        }
        Set<Map.Entry<String, String>> changeName = new HashSet<>();
        changeName.add(new AbstractMap.SimpleEntry<>(componentName, parameterName));
        updateStateAndSendStatusConfigurationInfo("", changeName);
    }

    /**
     * Stores a change parameter to be validated later.
     *
     * @param componentName
     * @param parameterName
     * @param value
     */
    @Command(description = "Submits a change of parameter to be validated later", type = Command.CommandType.CONFIGURATION)
    public void submitChange(String componentName, String parameterName, Object value) {
        // Check if change proposal is legal with regards to the subsystem description constraints
        ViewValue viewValue = configurationProxy.checkForParameterChange(componentName, parameterName, value);

        // Stores the change in the bulkChange structure
        Map<String, ViewValue> map = bulkChanges.get(componentName);
        if (map == null) {
            map = new LinkedHashMap<>();
            bulkChanges.put(componentName, map);
        }
        map.put(parameterName, viewValue);
    }

    /**
     * Changes that have been submitted are processed as a group.
     */
    @Command(description = "processes the bulk change", type = Command.CommandType.CONFIGURATION)
    public void commitBulkChange() {
        validateBulkChanges();
        Set<Map.Entry<String, String>> processedParms = bulkChange();
        updateStateAndSendStatusConfigurationInfo("",processedParms);
        
    }

    private void validateBulkChanges() {
        // Test against the configuration validation methods
        getLookup().proceduralWalk(null, Configurable.class,
                null, 
                c -> {
                    Map<String, ViewValue> changesForComponent = bulkChanges.get(c.getName());
                    if (changesForComponent == null) {
                        return;
                    }
                    
                    Map<String, ViewValue> parametersView
                            = configurationProxy.getCurrentValuesForComponent(c.getName());
                    parametersView.putAll(changesForComponent);
                    
                    c.validateBulkChange(parametersView.entrySet().stream()
                            .collect(Collectors.toMap(
                                    entry -> entry.getKey(),
                                    entry -> entry.getValue().getValue()
                            )));
                });
    }
    
    /**
     * Processes the change of configurable parameters contained in the bulk
     * changes.
     */
    private Set<Map.Entry<String,String>> bulkChange() {
        updateInternalState("", ConfigurationState.CONFIGURING);
        
        // stores the pair component name / parameter name of each successfully processed component
        Set<Map.Entry<String, String>> processedParms = new HashSet<>();
        // procedural walk starting from the top component
        getLookup().proceduralWalk(null, Configurable.class, null, new Consumer<Configurable>() {
            
            public void accept(Configurable c) {
                Map<String, ViewValue> changesForComponent = bulkChanges.get(c.getName());
                if (changesForComponent == null) {
                    return;
                }
                Map<String, Object> others = Collections.EMPTY_MAP;
                try {
                    // Bulk setting
                    others = c.setBulkParameter(changesForComponent.entrySet().stream()
                            .collect(Collectors.toMap(
                                    entry -> entry.getKey(),
                                    entry -> entry.getValue().getValue()
                            )));
                } catch (Exception ex) {
                    updateInternalState("", ConfigurationState.UNCONFIGURED);
                    dropBulkChange();
                    throw new RuntimeException(ex);
                }
                // Notify the bulk changes to the configuration service
                for (String parmName : changesForComponent.keySet()) {
                    if (!others.containsKey(parmName)) {
                        configurationProxy.notifyParameterChange(
                                c.getName(),
                                parmName,
                                changesForComponent.get(parmName).getView()
                        );
                        // TODO : This step should probably take place in the notifyParameterChange method 
                        processedParms.add(new AbstractMap.SimpleEntry<String, String>(c.getName(), parmName));
                    }
                }
                
                // Parameter setting with @ConfigChanger
                for (String parmName : others.keySet()) {
                    Method m = c.getEnvironment().getConfigMethodOfComponent(parmName);
                    ViewValue parmViewValue = changesForComponent.get(parmName);
                    try {
                        m.invoke(c, parmViewValue.getValue());
                        configurationProxy.notifyParameterChange(c.getName(), parmName, parmViewValue.getView());
                        // TODO : This step should probably take place in the notifyParameterChange method 
                        processedParms.add(new AbstractMap.SimpleEntry<String, String>(c.getName(), parmName));
                    } catch (Exception ex) {
                        updateStateAndSendStatusConfigurationInfo(ex.getMessage(), processedParms);
                        dropBulkChange();
                        throw new RuntimeException(ex);
                    }
                }
            }
        });
        dropBulkChange();
        return processedParms;
    }

    @Command(description = "Drops the submitted changes", type = Command.CommandType.CONFIGURATION)
    public void dropBulkChange() {
        bulkChanges.clear();
    }

    @Override
    public void doStart(){
        // publish the configuration the configurable is started with
        ConfigurationInfo configInfo = configurationProxy.buildConfigurationInfo(initialState, Collections.EMPTY_SET);
        updateInternalState("", initialState);
        getMessagingAccess().sendStatusMessage(new StatusConfigurationInfo(configInfo, getState()));
    }
    
    /**
     * Updates the state of the subsystem and sends a StatusConfigurationInfo status message
     * @param message 
     * @param recentChanges 
     */
    private void updateStateAndSendStatusConfigurationInfo(String message, Set<Map.Entry<String, String>> recentChanges){
        ConfigurationState configState = configurationProxy.isDirty() ? ConfigurationState.DIRTY : ConfigurationState.CONFIGURED;
        updateInternalState(message, configState);
        getMessagingAccess().sendStatusMessage(new StatusConfigurationInfo(configurationProxy.buildConfigurationInfo(configState, recentChanges), getState()));
    }

   

}
