package org.lsst.ccs.config;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.lsst.ccs.CCSProxy;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.description.ComponentLookupService;
import org.lsst.ccs.framework.Configurable;
import org.lsst.ccs.framework.ConfigurationServiceException;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.utilities.structs.ViewValue;

/**
 * instances of this class are created by the subsystem startup.
 * They operate delegations on behalf of the <TT>Configurable</TT> Object.
 * The <TT>Configurable</TT> Object can thus access its <TT>ConfigurationProxy</TT>
 * and its <TT>ComponentLookupService</TT>
 */
public class ConfigurationEnvironment implements Configurable.Environment {

    private static final HashMap<String, Map<String, Method>> METHOD_MAP
            = new HashMap<String, Map<String, Method>>();

    /**
     * name of the object as known to the description. (for Modules it is a duplicate
     * of their name.)
     */
    private final String name;
    /**
     * the object for which this environment operates. In some circumstances the Environment
     * object is linked to a proxy that acts on behalf of a "relevantObject"
     */
    private final Object relevantObject;
    /**
     * a Map from parameter names to Methods for the current object.
     */
    private final Map<String, Method> configMethods;

    /**
     * a reference to the Configuration Proxy
     */
    private final ConfigurationProxy configurationProxy;
    /**
     * a reference to the <TT>ComponentLookupService</TT>
     */
    private final ComponentLookupService lookupService;

    /**
     * Creates an Environment for a Configurable object.
     * It builds a CommandDictionary for the object and populates a map of ConfigChanger methods.
     * <P/>
     * The configuration proxy provides an access to the configuration database but interactions
     * are of delicate nature :
     * <UL>
     * <LI/> the base is checked as wether the new parameter value is considered possible
     * <LI/> then the parameter is changed locally
     * <LI/> the base is notified about the change
     * </UL>
     *
     * @param name
     * @param currentObject
     * @param configurationProxy
     * @param lookupService
     */
    public ConfigurationEnvironment(String name, Configurable currentObject, ConfigurationProxy configurationProxy,
            ComponentLookupService lookupService) {
        this.name = name;
            // REVIEW : Seems pointless : the attempt is to access @configChanger annotated
        // methods in the delegate but @ConfigChanger annotation is not accessible
        // at the driver level : there cannot be @ConfigChanger annotated methods in 
        // the delegate
        if (currentObject instanceof CCSProxy) {
            this.relevantObject = ((CCSProxy) currentObject).getDelegate();
        } else {
            this.relevantObject = Objects.requireNonNull(currentObject);
        }
        this.configurationProxy = Objects.requireNonNull(configurationProxy);
        this.lookupService = Objects.requireNonNull(lookupService);
        // introspects the class
        Class clazz = relevantObject.getClass();
        String clazzName = clazz.getName();

            // Introspects the class for @ConfigChanger annotated methods and 
        // stores them in METHOD_MAP
        Map<String, Method> confMethods = METHOD_MAP.get(clazzName);
        if (confMethods == null) {
            configMethods = new HashMap<>();
            METHOD_MAP.put(clazzName, configMethods);

            ///////// configMethod initializer!
            for (Method method : clazz.getMethods()) {
                String methodName = method.getName();
                    // This part still deals with the Configchanger annotation. It will be removed in the 2.0.0 release
                // See jiras: https://jira.slac.stanford.edu/browse/LSSTCCS-394 and https://jira.slac.stanford.edu/browse/LSSTCCS-393
                ConfigChanger configChanger = method.getAnnotation(ConfigChanger.class);
                if (configChanger != null) {
                    //if method is "setter" then gets name of property
                    if (methodName.startsWith("set")) {
                        char firstLetter = methodName.charAt(3);
                        String propertyName = Character.toLowerCase(firstLetter) + methodName.substring(4);
                        configMethods.put(propertyName, method);
                        //System.out.println(clazz.toString() + " registering parameter: " +propertyName + " " + method);
                    }
                    // different or additional name used
                    String propertyName = configChanger.propertyName();
                    if (!"".equals(propertyName)) {
                        configMethods.put(propertyName, method);
                    }
                }
                    //This is the new code dealing with @Command annotations of type: Command.CommandType.CONFIGURATION
                    /* this code temporarily commented out
                 Command configCommand = method.getAnnotation(Command.class);
                 if (configCommand != null && configCommand.type() == Command.CommandType.CONFIGURATION) {
                    
                 //if method is "setter" then gets name of property
                 if (methodName.startsWith("set")) {
                 char firstLetter = methodName.charAt(3);
                 String propertyName = Character.toLowerCase(firstLetter) + methodName.substring(4);
                 configMethods.put(propertyName, method);
                 //System.out.println(clazz.toString() + " registering parameter: " +propertyName + " " + method);
                 }
                 // different or additional name used
                 String propertyName = configCommand.name();
                 if (!"".equals(propertyName)) {
                 configMethods.put(propertyName, method);
                 }
                 }
                 END TEMPORARY DELETION */

            }
        } else {
            configMethods = confMethods;
        }

    }
            //TODO: create a command Method map (same idea as for configMethod) so you can invoke a method with a simpleName (and publish)

    /**
     * performs a parameter change during an engineering session
     *
     * @param parameterName name of parameter for the current object.
     * @param value any object that is fit for the configuration system (see documentation)
     * @throws IllegalArgumentException if parameter has a name unknown to the configuration
     * @throws Exception that can be thrown by the <TT>ConfigurationProxy</TT>
     * (no connection, illegal format, constraints failure) or by the user's code
     * (preconditions)
     *
     */
    @Override
    public void change(String parameterName, Object value) throws Exception {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().singleChange(this.name, parameterName, value);
        //TODO: ROLLBACKS? may be not if check does not modify database
    }

    @Override
    public void submitChange(String parameterName, Object value) {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().submitChange(this.name, parameterName, value);
    }

    //------------- Methods kept for backward compatibility --------------//
    /**
     * registers a new Configuration
     *
     * @param configurationName
     * @throws ConfigurationServiceException
     * @deprecated use saveChangesForCategoriesAs(configurationName) instead
     */
    @Deprecated
    @Override
    public void register(String configurationName) throws ConfigurationServiceException {
        saveChangesForCategoriesAs(configurationName);
    }

    /**
     *
     * @throws ConfigurationServiceException
     * @deprecated use saveAllChanges instead
     */
    @Deprecated
    @Override
    public void saveConfiguration() throws ConfigurationServiceException {
        saveAllChanges();
    }

    /**
     *
     * @param configName
     * @throws ConfigurationServiceException
     * @deprecated use saveChangesForCategoriesAs(configName) instead
     */
    @Deprecated
    @Override
    public void saveConfiguration(String configName) throws ConfigurationServiceException {
        saveChangesForCategoriesAs(configName);
    }

        //--------------------------------------------------------------------//
    /**
     * Saves all changes in their current configuration.
     *
     * @throws org.lsst.ccs.framework.ConfigurationServiceException
     */
    @Override
    public void saveAllChanges() throws ConfigurationServiceException {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().saveAllChanges();
    }

    /**
     * Changes made in the specified categories are saved under the current configuration
     * for this category, changes on parameters that belong to other categories
     * are left unchanged.
     *
     * @param categories a list of categories
     * @throws org.lsst.ccs.framework.ConfigurationServiceException
     */
    @Override
    public void saveChangesForCategories(String... categories) throws ConfigurationServiceException {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().saveChangesForCategories(categories);
    }

    /**
     * Changes made in the specified categories are saved under the newly specified
     * name for this category, changes on parameters that belong to other categories
     * are left unchanged.
     *
     * @param taggedCategories a list of pairs categoryName:configurationName
     * @throws org.lsst.ccs.framework.ConfigurationServiceException
     */
    @Override
    public void saveChangesForCategoriesAs(String... taggedCategories) throws ConfigurationServiceException {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().saveChangesForCategoriesAs(taggedCategories);
    }

    /**
     * All unsaved changes are dropped, each parameter goes back to the value specified
     * by the current configuration for this category.
     * The configuration context is no longer active.
     *
     * @throws Exception
     */
    @Override
    public void dropAllChanges() throws Exception {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().dropAllChanges();
    }

    /**
     * Changes occurring for one of the mentioned categories in categories are dropped,
     * other categories are left untouched.
     * The configuration context remains active.
     *
     * @param categories A list of categories
     * @throws Exception
     */
    @Override
    public void dropChangesForCategories(String... categories) throws Exception {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().dropChangesForCategories(categories);
    }

    private boolean isInEngineeringMode() {
        return getSubsystem().get().isInEngineeringMode();
    }

    /**
     * Checks if a value can be accepted by the Configuration system.
     * to be used by "facade" commands that may set a value.
     * <P/>
     * The returned value is an Object that represent the real value as
     * generated by the Configuration: it is not necessarily the value passed
     * as argument because the argument can both be a real object or its String
     * representation (as extracted from a properties file or from a GUI interaction).
     * This returned object contains both this real value AND its String representation.
     *
     * @param parameterName
     * @param value can be a real Object or its String representation
     * @return An object <TT>ViewValue</TT> that returns both an Object value and
     * its String representation as used by the Configuration.
     * @throws Exception
     */
    @Override
    public ViewValue getCheckedValueFromConfiguration(String parameterName, Object value)
            throws Exception {
        ViewValue data = configurationProxy.checkForParameterChange(this.name, parameterName, value);
        return data;
    }

    /**
     * to be used by "facade" commands that may set a value
     * then notify the configuration without going through the preliminary check
     * of the configuration service.
     * <BR/>
     * <B>use at your own risk!</B>
     *
     * @param parameterName
     * @param value
     * @throws Exception
     */
    @Override
    public void notifyChangeWithoutPreliminaryChecks(String parameterName, Object value) throws Exception {
        configurationProxy.notifyUncheckedParameterChange(this.name, parameterName, value);
    }

    /**
     * to be used by "facade" commands that may set a value
     * <BR>
     * the calling must have made a preliminary check with the configuration service.
     * <BR/>
     * <B>indiscriminate use should be discouraged</B>
     *
     * @param parameterName
     * @param value
     * @throws Exception
     */
    @Override
    public void notifyChange(String parameterName, String value) throws Exception {
        configurationProxy.notifyParameterChange(this.name, parameterName, value);
    }

    /**
     * request a Component from a Description by its name.
     *
     * @param name
     * @return A Component. <B>beware</B> unless you are dead sure do not assume this component is
     * a <TT>Configurable</TT> object (any object can be aliased and so put in the dictionary)
     */
    @Override
    public Object getComponentByName(String name) {
        return lookupService.getComponentByName(name);
    }

    @Override
    public void alias(String name) {
        lookupService.aliasObject(name, relevantObject);
    }

    /**
     * get the children of the {@code Configurable} object that are assignable
     * to an instance of the class argument.
     *
     * @param classFilter
     * @param <T>
     * @return a map with name of the component as key and its class as value.
     */
    @Override
    public <T> LinkedHashMap<String, T> getChildren(Class<T> classFilter) {
        Objects.requireNonNull(classFilter);
        return lookupService.getChildren(this.name, classFilter);
    }

    /** *
     * gets the children of the {@code Configurable} object
     *
     * @return a list of {@code Configurable}
     */
    @Override
    public List<Configurable> listChildren() {
        return lookupService.listChildren(this.name);
    }

    /**
     * returns the parent of the current component
     *
     * @return the Component (value) and its name (key)
     * null if there is no parent.
     */
    @Override
    public Map.Entry<String, Object> getParent() {
        return lookupService.getParent(this.name);
    }

    /**
     *
     * @return the name associated with this component
     */
    @Override
    public String getNameOfComponent() {
        return name;
    }

    /**
     *
     * @return the object managed by the environment (may be different from the current Configurable when there is a Proxy
     */
    @Override
    public Object getRelevantObject() {
        return relevantObject;
    }

    @Deprecated
    @Override
    public Map<String, Method> getConfigMethodsOfComponent() {
        return configMethods;
    }

    @Override
    public Method getConfigMethodOfComponent(String parameterName) {
        Method res = configMethods.get(parameterName);
        if (res == null) {
            throw new IllegalArgumentException(" no such config property :" + parameterName);
        }
        return res;
    }

    @Override
    public Optional<Subsystem> getSubsystem() {
        return lookupService.getSubsystem();
    }
    
    private ConfigurableSubsystem getConfigurableSubsystem() {
        return (ConfigurableSubsystem) getSubsystem().get();
    }
    
    
    @Override
    public void dropBulkChange() {
        getConfigurableSubsystem().dropBulkChange();
    }
    
}
